/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.gecko.rsa.topology.imports.local;

import java.util.Collection;
import java.util.logging.Logger;

import org.gecko.rsa.core.helper.FilterHelper;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.hooks.service.ListenerHook;

/**
 * Listens for service listeners and informs {@link ServiceInterestListener} about 
 * added and removed interest in services
 */
public class LocalServiceListenerHook implements ListenerHook {

    private static final Logger logger = Logger.getLogger(LocalServiceListenerHook.class.getName());

    private final BundleContext bctx;
    private final ServiceInterestListener serviceInterestListener;
    private final String frameworkUUID;

    public LocalServiceListenerHook(BundleContext bc, ServiceInterestListener serviceInterestListener) {
        this.bctx = bc;
        this.frameworkUUID = bctx.getProperty(Constants.FRAMEWORK_UUID);
        this.serviceInterestListener = serviceInterestListener;
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.framework.hooks.service.ListenerHook#added(java.util.Collection)
     */
    @Override
    public void added(Collection<ListenerInfo> listeners) {
        logger.fine(String.format("Added service listeners %s", listeners));
        /*
         * Go over all listeners that are interested in any service
         */
        for (ListenerInfo listenerInfo : listeners) {
        	if (listenerInfo.getBundleContext().equals(bctx)) {
        		logger.fine("Skipping request from myself");
        		continue;
        	}
        	
        	String filter = listenerInfo.getFilter();
        	if (filter == null) {
        		logger.fine("Skipping service listener with empty filter");
        		continue;
        	}
        	
            logger.fine(String.format("Filter service listeners for filter '%s'", filter));

            // extract the service class name
            String className = FilterHelper.getObjectClass(filter);
            if (FilterHelper.isClassExcluded(className)) {
                logger.fine(String.format("Skipping import request for an excluded class [%s]", className));
                continue;
            }
            // create a filter for a potential service our system 
            String extendedFilter = FilterHelper.extendFilter(listenerInfo.getFilter(), frameworkUUID);
            serviceInterestListener.addServiceInterest(extendedFilter);
        }
    }

    /* 
     * (non-Javadoc)
     * @see org.osgi.framework.hooks.service.ListenerHook#removed(java.util.Collection)
     */
    @Override
    public void removed(Collection<ListenerInfo> listeners) {
        logger.fine(String.format("Removed service listeners %s", listeners));

        for (ListenerInfo listenerInfo : listeners) {
            logger.fine(String.format("Filter service listener %s", listenerInfo.getFilter()));

            String extendedFilter = FilterHelper.extendFilter(listenerInfo.getFilter(), frameworkUUID);
            serviceInterestListener.removeServiceInterest(extendedFilter);
        }
    }

}
