/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.runtime.application;

import java.util.Map;
import java.util.Set;

import org.osgi.framework.Filter;
import org.osgi.service.application.ScheduledApplication;

/**
 * Application manager to start, stop, schedule or restart applications
 * @author Mark Hoffmann
 * @since 23.03.2018
 */
/**
 * 
 * @author mark
 * @since 23.03.2018
 */
public interface ApplicationManager {
	
	/**
	 * Returns all id's of application descriptors or an empty {@link Set}
	 * @return all id's of application descriptors or an empty {@link Set}
	 */
	public Set<String> getAllApplications();
	
	/**
	 * Returns all id's of active application descriptors or an empty {@link Set}
	 * @return all id's of active application descriptors or an empty {@link Set}
	 */
	public Set<String> getActiveApplications();
	
	/**
	 * Locks the application with the given id
	 * @param applicationId the application to lock
	 * @return <code>true</code> on success
	 */
	public boolean lockApplication(String applicationId);
	
	/**
	 * Un-locks the application with the given id
	 * @param applicationId the application to lock
	 * @return <code>true</code> on success
	 */
	public boolean unlockApplication(String applicationId);
	
	/**
	 * Starts the application with the given id
	 * @param applicationId the application to start
	 * @param launchProperties optional launch properties
	 * @return <code>true</code> on success
	 */
	public boolean startApplication(String applicationId, Map<String, Object> launchProperties);
	
	/**
	 * Starts the application with the given id
	 * @param applicationId the application to start
	 * @param launchProperties optional launch properties
	 * @return <code>true</code> on success
	 */
	public boolean stopApplication(String applicationId);
	
	/**
	 * Uses the virtual timer of the system to schedule an application
	 * @param applicationId the application to be run 
	 * @param launchProperties the launch properties
	 * @param scheduleFilter the time based filter
	 * @return the scheduled application instance
	 */
	public ScheduledApplication scheduleApplication(String applicationId, Map<String, Object> launchProperties, Filter scheduleFilter);
	
	/**
	 * Revokes the scheduling of an application
	 * @param applicationId the application to be run 
	 * @return <code>true</code> on success
	 */
	public boolean unscheduleApplication(String applicationId);

}
