/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.runtime.application.internal;

import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.logging.Logger;

import org.osgi.framework.Filter;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.application.ScheduledApplication;
import org.osgi.service.event.Event;

/**
 * Scheduled timer implementation
 * @author Mark Hoffmann
 * @since 23.03.2018
 */
class ScheduledApplicationTimer implements Runnable {

	private static final Logger logger = Logger.getLogger("org.gecko.application.timer");
	private volatile int lastMin = -1;
	private Collection<GeckoScheduledApplication> scheduledApplications;

	/**
	 * Creates a new instance.
	 */
	public ScheduledApplicationTimer(Collection<GeckoScheduledApplication> scheduledApplications) {
		this.scheduledApplications = scheduledApplications == null ? Collections.emptySet() : scheduledApplications;
	}

	/* 
	 * (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		Calendar cal = Calendar.getInstance();
		int minute = cal.get(Calendar.MINUTE);
		if (minute == lastMin) {
			return;
		}
		lastMin = minute;
		Hashtable<String, Object> props = new Hashtable<>();
		props.put(ScheduledApplication.YEAR, new Integer(cal.get(Calendar.YEAR)));
		props.put(ScheduledApplication.MONTH, new Integer(cal.get(Calendar.MONTH)));
		props.put(ScheduledApplication.DAY_OF_MONTH, new Integer(cal.get(Calendar.DAY_OF_MONTH)));
		props.put(ScheduledApplication.DAY_OF_WEEK, new Integer(cal.get(Calendar.DAY_OF_WEEK)));
		props.put(ScheduledApplication.HOUR_OF_DAY, new Integer(cal.get(Calendar.HOUR_OF_DAY)));
		props.put(ScheduledApplication.MINUTE, new Integer(minute));
		Event timerEvent = new Event(ScheduledApplication.TIMER_TOPIC, (Dictionary<String, Object>) props);
		for (GeckoScheduledApplication gsa : scheduledApplications) {
			try {
				String filterString = gsa.getEventFilter();
				Filter filter = filterString == null ? null : FrameworkUtil.createFilter(filterString);
				if (filter == null || filter.match(props))
					gsa.handleEvent(timerEvent);
			} catch (Throwable t) {
				logger.warning("[" + gsa.getScheduleId() + "] Cannot handle virtual timer event");
			}
		}
	}
	
}