/**
 * 
 */
package org.gecko.runtime.bnd.export;

import java.io.File;
import java.io.IOException;
import java.util.AbstractMap.SimpleEntry;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import aQute.bnd.annotation.plugin.BndPlugin;
import aQute.bnd.build.Project;
import aQute.bnd.header.Attrs;
import aQute.bnd.header.Parameters;
import aQute.bnd.osgi.Constants;
import aQute.bnd.osgi.FileResource;
import aQute.bnd.osgi.Jar;
import aQute.bnd.osgi.JarResource;
import aQute.bnd.osgi.Resource;
import aQute.bnd.service.export.Exporter;

/**
 * BND Exporter Plugin for a Gecko Runtime
 * 
 * @author Juergen Albert
 */
@BndPlugin(name = "gecko")
public class GeckoExporter implements Exporter{

	private final static Logger logger = LoggerFactory.getLogger(GeckoExporter.class);
	
	private static String[] TYPES = new String[] {"gecko.export"};
	
	@Override
	public String[] getTypes() {
		return TYPES;
	}

	@Override
	public Entry<String, Resource> export(String type, Project project, Map<String, String> options) throws Exception {

		Jar jar = new Jar(project.getName().replace(".bndrun", ""));
		jar.setDoNotTouchManifest();

		Jar releng = getJarForPlugin(project);
		
		releng.getDirectories().forEach((k, v) -> {
			if(v != null && !k.equals("runtime") && k.startsWith("runtime")) {
				v.entrySet().stream().map(e -> new SimpleEntry<String, Resource>(e.getKey().substring(e.getKey().indexOf('/') + 1), e.getValue()))
				.forEach(e -> {
					logger.info("Adding resource {} from exporter jar ", e.getValue());
					jar.putResource(e.getKey(), e.getValue());
				});
			}
		});
		
		if(options.containsKey("gecko.runtime.dir")) {
			File runtimeFolder = new File(options.get("gecko.runtime.dir"));
			logger.info("Including resource from {} with projectbase {} in runtime export", runtimeFolder.toURI().toString(), project.getBase());
			if(runtimeFolder.listFiles() != null) {
				for (File child : runtimeFolder.listFiles()) {
					addToJar(jar, child, "");
				}
			}
		}
		
		logger.info("Running the jar export internally");
		Entry<String, Resource> export = project.export("bnd.executablejar", null);
		jar.putResource("lib/launch.jar", export.getValue());
		return new SimpleEntry<>(jar.getName() + ".zip", new JarResource(jar, true));
	}

	private Jar getJarForPlugin(Project project) {
		Parameters pluginpaths = new Parameters(project.mergeProperties(Constants.PLUGINPATH), project);
		for (Entry<String, Attrs> entry : pluginpaths.entrySet()) {
			File file = project.getFile(entry.getKey());
			try {
				Jar jar = new Jar(file);
				if(jar.getResource("runtime/lib/.keep") != null) {
					return jar;
				} else {
					jar.close();
				}
			} catch (IOException e) {
				logger.error("Could not open " + file, e);
			}
			
		}
		return null;
	}

	/**
	 * @param jar
	 * @param child
	 * @throws IOException 
	 */
	private void addToJar(Jar jar, File file, String dir) throws IOException {
		if(file.isFile()) {
			Resource remove = jar.remove(dir + file.getName());
			jar.putResource(dir + file.getName(), new FileResource(file), true);
			if(remove != null) {
				logger.info("Overwriting resource: {} with project specific Version", dir + file.getName());
			} else {
				logger.info("Adding Resource {}", dir + file.getName());
			}
		} else {
			for (File child : file.listFiles()) {
				addToJar(jar, child, dir + file.getName() + "/");
			}
		}
	}
}
