/**
 * Copyright (c) 2012 - 2018 Data In Motion Consulting.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion Consulting - initial API and implementation
 */
package org.gecko.runtime.config.internal;

import java.net.URL;

import org.gecko.core.resources.file.ServiceFileWatcher;
import org.gecko.runtime.boot.GeckoBootConstants;
import org.osgi.annotation.bundle.Capability;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;

/**
 * Configurator that looks into the configuration folder and creates
 * URL service for each file in it
 * @author Mark Hoffmann
 * @since 07.11.2017
 */
@Capability(
		namespace="gecko.addons",
		name="configuration",
		version="1.0.0"
		)
@Component(immediate=true, enabled=true)
public class ConfigurationFileConfigurator {

	private BundleContext bctx = null;
	private URL confDir = null;
	private ServiceFileWatcher watcher = null;
	private String[] filterList = new String[] {"*.properties", "*.conf", "*.config"};

	/**
	 * Called on component activation
	 * @param ctx the component context
	 */
	@Activate
	public void activate(ComponentContext ctx) {
		bctx = ctx.getBundleContext();
		/*
		 * URL was already injected, but no bundle context 
		 * for a service registration was availables.
		 */
		if (confDir != null && watcher == null) {
			watcher = new ServiceFileWatcher(confDir, bctx);
			watcher.setFilterList(filterList);
			watcher.start();
		}
	}

	/**
	 * Called on component de-activation
	 */
	@Deactivate
	public void deactivate() {
		watcher.stop();
	}

	/**
	 * Sets the configuration folder URL
	 * @param confDir the URL for the configuration folder
	 */
	@Reference(cardinality=ReferenceCardinality.OPTIONAL, target="(" + GeckoBootConstants.PROP_GECKO_CONFIG_DIR + "=true)")
	public void setConfigDirectory(URL confDir) {
		this.confDir = confDir;
		// Don't call update, if component wasn't already activated
		if (bctx != null) {
			if (watcher == null) {
				watcher = new ServiceFileWatcher(confDir, bctx);
				watcher.setFilterList(filterList);
				watcher.start();
			}
		}
	}

	/**
	 * Un-sets the configuration folder URL
	 * @param confDir the URL for the configuration folder
	 */
	public void unsetConfigDirectory(URL confDir) {
		this.confDir = null;
		watcher.stop();
	}

}
