/**
 * Copyright (c) 2012 - 2018 Data In Motion Consulting.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion Consulting - initial API and implementation
 */
package org.gecko.runtime.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;
import java.util.Dictionary;
import java.util.Hashtable;

import org.gecko.core.tests.AbstractOSGiTest;
import org.gecko.core.tests.ServiceChecker;
import org.gecko.runtime.boot.GeckoBootConstants;
import org.gecko.runtime.resources.GeckoResourcesConstants;
import org.gecko.runtime.resources.GeckoResourcesProvider;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.service.cm.Configuration;

/**
 * Integration test to test handling of the folder watcher in OSGi environments
 * @author Mark Hoffmann
 *
 */
@RunWith(MockitoJUnitRunner.class)
public class ResourcesIntegrationTest extends AbstractOSGiTest {

	public ResourcesIntegrationTest() {
		super(FrameworkUtil.getBundle(ResourcesIntegrationTest.class).getBundleContext());
	}

	private File dataFolder;

	@Override
	public void doBefore() {
		String tmp = System.getProperty(GeckoBootConstants.PROP_GECKO_DATA_DIR);
		dataFolder = new File(tmp);
		if (!dataFolder.exists()) {
			dataFolder.mkdirs();
		}
		assertTrue(dataFolder.exists());
	}

	@Override
	public void doAfter() {
		assertTrue(dataFolder.exists());
		if (dataFolder.listFiles().length > 0) {
			for (File f : dataFolder.listFiles()) {
				assertTrue(f.delete());
			}
		}
		assertTrue(dataFolder.delete());
	}

	@Test
	public void testResourceFolder() throws InvalidSyntaxException, IOException {
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(GeckoResourcesConstants.RESOURCE_NAME, "mytest");
		properties.put(GeckoResourcesConstants.RESOURCE_PATH, "/tmp/geckodata/test");
		
		ServiceChecker<GeckoResourcesProvider> providerChecker = createCheckerTrackedForCleanUp(GeckoResourcesProvider.class);
		providerChecker.start();
		assertEquals(0, providerChecker.getCurrentRemoveCount(false));
		assertEquals(0, providerChecker.getCurrentCreateCount(false));
		
		Configuration c = getConfigAdmin().getFactoryConfiguration(GeckoResourcesConstants.RESOURCES_FACTORY_ID, "test", "?");
		c.update(properties);
		
		assertEquals(0, providerChecker.getCurrentRemoveCount(false));
		assertEquals(1, providerChecker.getCurrentCreateCount(true));
		
		GeckoResourcesProvider provider = getService(GeckoResourcesProvider.class);
		assertNotNull(provider);
		assertEquals("mytest", provider.getName());
		assertNotNull(provider.getFile());
		
		c.delete();
		
		assertEquals(1, providerChecker.getCurrentRemoveCount(true));
		assertEquals(1, providerChecker.getCurrentCreateCount(false));
	}
	
	@Test
	public void testResourceSubstituteFolder() throws InvalidSyntaxException, IOException {
		Dictionary<String, Object> properties = new Hashtable<String, Object>();
		properties.put(GeckoResourcesConstants.RESOURCE_NAME, "mytest");
		properties.put(GeckoResourcesConstants.RESOURCE_PATH, "gecko.data.dir/test");
		
		ServiceChecker<GeckoResourcesProvider> providerChecker = createCheckerTrackedForCleanUp(GeckoResourcesProvider.class);
		providerChecker.start();
		assertEquals(0, providerChecker.getCurrentRemoveCount(false));
		assertEquals(0, providerChecker.getCurrentCreateCount(false));
		
		Configuration c = getConfigAdmin().getFactoryConfiguration(GeckoResourcesConstants.RESOURCES_FACTORY_ID, "test", "?");
		c.update(properties);
		
		assertEquals(0, providerChecker.getCurrentRemoveCount(false));
		assertEquals(1, providerChecker.getCurrentCreateCount(true));
		
		GeckoResourcesProvider provider = getService(GeckoResourcesProvider.class);
		assertNotNull(provider);
		assertEquals("mytest", provider.getName());
		assertNotNull(provider.getFile());
		
		c.delete();
		
		assertEquals(1, providerChecker.getCurrentRemoveCount(true));
		assertEquals(1, providerChecker.getCurrentCreateCount(false));
	}
	
}