/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.emf.osgi.urihandler;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ContentHandler;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.URIHandlerImpl;
import org.gecko.emf.osgi.EMFUriHandlerConstants;

/**
 * URI Handler with basic authentication
 * @author Juergen Albert
 * @since 17.12.2012
 */
public class RestfulURIHandlerImpl extends URIHandlerImpl {

	/** SCHEMA_HTTPS */
	private static final String SCHEMA_HTTPS = "https";
	/** SCHEMA_HTTP */
	private static final String SCHEMA_HTTP = "http";
	/** AUTH_BASIC */
	private static final String AUTH_BASIC = "Basic ";
	/** HEADER_AUTHORIZATION */
	private static final String HEADER_AUTHORIZATION = "Authorization";
	/** HEADER_CONTENT_LENGTH */
	private static final String HEADER_CONTENT_LENGTH = "Content-Length";
	/** HTTP_HEAD */
	private static final String HTTP_HEAD = "HEAD";
	/** HEADER_ALLOW */
	private static final String HEADER_ALLOW = "Allow";
	/** HTTP_OPTIONS */
	private static final String HTTP_OPTIONS = "OPTIONS";
	/** HTTP_DELETE */
	private static final String HTTP_DELETE = "DELETE";
	/** PROP_HTTP_RESPONSE_CODE */
	private static final String PROP_HTTP_RESPONSE_CODE = "HTTPResponseCode";
	/** HEADER_LAST_MODIFIED */
	private static final String HEADER_LAST_MODIFIED = "Last-Modified";
	/** HEADER_CONTENT_CLASS */
	private static final String HEADER_CONTENT_CLASS = "Content-Class";
	/** PROP_ECLASS */
	private static final String PROP_ECLASS = "EClass";
	/** HTTP_PUT */
	private static final String HTTP_PUT = "PUT";
	private static final Logger LOG = Logger.getLogger(RestfulURIHandlerImpl.class.getName());

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#createOutputStream(org.eclipse.emf.common.util.URI, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public OutputStream createOutputStream(URI uri, final Map<?, ?> options) throws IOException {
		URL url = new URL(uri.toString());
		final HttpURLConnection httpURLConnection = (HttpURLConnection) url.openConnection();
		String method = HTTP_PUT;
		if (options.containsKey(EMFUriHandlerConstants.OPTION_HTTP_METHOD)) {
			method = options.get(EMFUriHandlerConstants.OPTION_HTTP_METHOD).toString().toUpperCase();
		}
		httpURLConnection.setRequestMethod(method);
		httpURLConnection.setDoOutput(Boolean.TRUE);
		setRequestHeaders(httpURLConnection,
				(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
		if (options.containsKey(PROP_ECLASS)) {
			httpURLConnection.setRequestProperty(HEADER_CONTENT_CLASS, options.get(PROP_ECLASS).toString());
		}
		handleBasicAuth(httpURLConnection, options);
		return new FilterOutputStream(httpURLConnection.getOutputStream()) {
			@Override
			public void close() throws IOException {
				super.close();
				try {
					int responseCode = httpURLConnection.getResponseCode();
					Map<Object, Object> response = getResponse(options);
					if (response != null) {
						try {
							String lastModified = httpURLConnection.getHeaderField(HEADER_LAST_MODIFIED);
							if (lastModified != null) {
								Long lm = Long.parseLong(lastModified);
								response.put(URIConverter.RESPONSE_TIME_STAMP_PROPERTY, lm);
							}
						} catch (Exception e) {
							LOG.log(Level.SEVERE, "Error reading last modified header from the response", e);
						}
						response.put(PROP_HTTP_RESPONSE_CODE, responseCode);
						response.putAll(httpURLConnection.getHeaderFields());
					}
					InputStream in = extreactStreamAndLogResponse(options, httpURLConnection);
					switch (responseCode) {
					case HttpURLConnection.HTTP_OK:
					case HttpURLConnection.HTTP_CREATED:
						Resource responseResource = (Resource) options
								.get(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE);
						if (responseResource != null) {
							responseResource.load(in, (Map<?, ?>) options
									.get(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE_OPTIONS));
						}
						break;
					case HttpURLConnection.HTTP_NO_CONTENT: {
						break;
					}
					default: {
						throw new IOException(httpURLConnection.getRequestMethod() + " failed with HTTP response code " + responseCode);
					}
					}
				} finally {
					httpURLConnection.disconnect();
				}
			}
		};
	}

	/**
	 * Sets the given headers to the url connection
	 * 
	 * @param httpURLConnection
	 * @param headers
	 */
	private void setRequestHeaders(HttpURLConnection httpURLConnection, Map<String, String> headers) {
		if (headers != null) {
			for (Entry<String, String> entry : headers.entrySet()) {
				httpURLConnection.setRequestProperty(entry.getKey(), entry.getValue());
			}
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#createInputStream(org.eclipse.emf.common.util.URI, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public InputStream createInputStream(URI uri, Map<?, ?> options) throws IOException {
		try {
			URL url = new URL(uri.toString());
			final HttpURLConnection httpURLConnection = (HttpURLConnection) url.openConnection();
			handleBasicAuth(httpURLConnection, options);
			setRequestHeaders(httpURLConnection,
					(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
			final int responseCode = httpURLConnection.getResponseCode();
			Map<Object, Object> response = getResponse(options);
			if (response != null) {
				try {
					String lastModified = httpURLConnection.getHeaderField(HEADER_LAST_MODIFIED);
					if (lastModified != null) {
						Long lm = Long.parseLong(lastModified);
						response.put(URIConverter.RESPONSE_TIME_STAMP_PROPERTY, lm);
					}
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "Error reading last modified header from the response", e);
				}
			}
			InputStream result = extreactStreamAndLogResponse(options, httpURLConnection);
			return new FilterInputStream(result) {

				/*
				 * (non-Javadoc)
				 * 
				 * @see java.io.FilterInputStream#read()
				 */
				@Override
				public int read() throws IOException {
					if (responseCode == HttpURLConnection.HTTP_NO_CONTENT) {
						return -1;
					}
					return super.read();
				}

				@Override
				public void close() throws IOException {
					super.close();

					int responseCode = httpURLConnection.getResponseCode();
					httpURLConnection.disconnect();
					switch (responseCode) {
						case HttpURLConnection.HTTP_OK:
						case HttpURLConnection.HTTP_CREATED:
						case HttpURLConnection.HTTP_NO_CONTENT: {
							break;
						}
						default: {
							throw new IOException(httpURLConnection.getRequestMethod() + " failed with HTTP response code " + responseCode);
						}
					}
				}

			};
		} catch (RuntimeException exception) {
			throw new Resource.IOWrappedException(exception);
		}
	}

	/**
	 * @param options
	 * @param httpURLConnection
	 * @return
	 * @throws IOException
	 */
	private InputStream extreactStreamAndLogResponse(Map<?, ?> options, final HttpURLConnection httpURLConnection)
			throws IOException {
		InputStream result = httpURLConnection.getErrorStream();
		if (result == null) {
			result = httpURLConnection.getInputStream();
		}
		if (Boolean.TRUE.equals(options.get(EMFUriHandlerConstants.OPTIONS_LOG_RESPONSE))) {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			int read = result.read();
			while (read != -1) {
				baos.write(read);
				read = result.read();
			}
			byte[] responseArray = baos.toByteArray();
			baos.close();
			result = new ByteArrayInputStream(responseArray);
		}
		return result;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#delete(org.eclipse.emf.common.util.URI, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void delete(URI uri, Map<?, ?> options) throws IOException {
		try {
			URL url = new URL(uri.toString());
			final HttpURLConnection httpURLConnection = (HttpURLConnection) url.openConnection();
			;
			httpURLConnection.setDoOutput(true);
			handleBasicAuth(httpURLConnection, options);
			setRequestHeaders(httpURLConnection,
					(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
			httpURLConnection.setRequestMethod(HTTP_DELETE);
			int responseCode = httpURLConnection.getResponseCode();
			httpURLConnection.disconnect();
			switch (responseCode) {
			case HttpURLConnection.HTTP_OK:
			case HttpURLConnection.HTTP_ACCEPTED:
			case HttpURLConnection.HTTP_NO_CONTENT: {
				break;
			}
			default: {
				throw new IOException(httpURLConnection.getRequestMethod() + " failed with HTTP response code " + responseCode);
			}
			}
		} catch (RuntimeException exception) {
			throw new Resource.IOWrappedException(exception);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#getAttributes(org.eclipse.emf.common.util.URI, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	public Map<String, ?> getAttributes(URI uri, Map<?, ?> options) {
		Map<String, Object> result = new HashMap<String, Object>();
		Set<String> requestedAttributes = getRequestedAttributes(options);
		try {
			URL url = new URL(uri.toString());
			URLConnection urlConnection = null;
			if (requestedAttributes == null || requestedAttributes.contains(URIConverter.ATTRIBUTE_READ_ONLY)) {

				urlConnection = url.openConnection();
				if (urlConnection instanceof HttpURLConnection) {
					HttpURLConnection httpURLConnection = (HttpURLConnection) urlConnection;
					httpURLConnection.setRequestMethod(HTTP_OPTIONS);
					handleBasicAuth(httpURLConnection, options);
					setRequestHeaders(httpURLConnection,
							(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
					int responseCode = httpURLConnection.getResponseCode();
					if (responseCode == HttpURLConnection.HTTP_OK) {
						String allow = httpURLConnection.getHeaderField(HEADER_ALLOW);
						result.put(URIConverter.ATTRIBUTE_READ_ONLY, allow == null || !allow.contains(HTTP_PUT));
					}
					urlConnection = null;
				} else {
					result.put(URIConverter.ATTRIBUTE_READ_ONLY, true);
				}
			}

			if (requestedAttributes == null || requestedAttributes.contains(URIConverter.ATTRIBUTE_TIME_STAMP)) {
				if (urlConnection == null) {
					urlConnection = url.openConnection();
					if (urlConnection instanceof HttpURLConnection) {
						HttpURLConnection httpURLConnection = (HttpURLConnection) urlConnection;
						setRequestHeaders(httpURLConnection,
								(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
						httpURLConnection.setRequestMethod(HTTP_HEAD);
						httpURLConnection.getResponseCode();
					}
				}
				if (urlConnection.getHeaderField(HEADER_LAST_MODIFIED) != null) {
					result.put(URIConverter.ATTRIBUTE_TIME_STAMP, urlConnection.getLastModified());
				}
			}

			if (requestedAttributes == null || requestedAttributes.contains(URIConverter.ATTRIBUTE_LENGTH)) {
				if (urlConnection == null) {
					urlConnection = url.openConnection();
					if (urlConnection instanceof HttpURLConnection) {
						HttpURLConnection httpURLConnection = (HttpURLConnection) urlConnection;
						setRequestHeaders(httpURLConnection,
								(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
						httpURLConnection.setRequestMethod(HTTP_HEAD);
						httpURLConnection.getResponseCode();
					}
				}
				if (urlConnection.getHeaderField(HEADER_CONTENT_LENGTH) != null) {
					result.put(URIConverter.ATTRIBUTE_LENGTH, urlConnection.getContentLength());
				}
			}
		} catch (IOException exception) {
			// Ignore exceptions.
		}
		return result;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#exists(org.eclipse.emf.common.util.URI, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean exists(URI uri, Map<?, ?> options) {
		try {
			URL url = new URL(uri.toString());
			HttpURLConnection httpURLConnection = (HttpURLConnection) url.openConnection();
			httpURLConnection.setRequestMethod(HTTP_HEAD);
			handleBasicAuth(httpURLConnection, options);
			setRequestHeaders(httpURLConnection,
					(Map<String, String>) options.get(EMFUriHandlerConstants.OPTION_HTTP_HEADERS));
			int responseCode = httpURLConnection.getResponseCode();
			Map<Object, Object> response = getResponse(options);
			if (response != null) {
				try {
					String lastModified = httpURLConnection.getHeaderField(HEADER_LAST_MODIFIED);
					if (lastModified != null) {
						Long lm = Long.parseLong(httpURLConnection.getHeaderField(HEADER_LAST_MODIFIED));
						response.put(URIConverter.RESPONSE_TIME_STAMP_PROPERTY, lm);
					}
				} catch (Exception e) {
					LOG.log(Level.SEVERE, "Error reading last modified header from the response", e);
				}
			}
			httpURLConnection.disconnect();
			return responseCode == HttpURLConnection.HTTP_OK;
		} catch (Throwable exception) {
			return false;
		}
	}

	/**
	 * This implementation delegates to the {@link #getURIConverter(Map) URI
	 * converter}'s {@link URIConverter#getContentHandlers() content handlers}.
	 */
	public Map<String, ?> contentDescription(URI uri, Map<?, ?> options) throws IOException {
		return ContentHandler.INVALID_CONTENT_DESCRIPTION;
	}

	/**
	 * Handles the basic authentication setting for the {@link URLConnection}
	 * 
	 * @param httpURLConnection
	 *            the {@link URLConnection}
	 * @param options
	 *            a map with options
	 */
	private void handleBasicAuth(HttpURLConnection httpURLConnection, Map<?, ?> options) {
		String username = (String) options.get(EMFUriHandlerConstants.OPTIONS_AUTH_USER);
		String mandant = (String) options.get(EMFUriHandlerConstants.OPTIONS_AUTH_MANDANT);
		String password = (String) options.get(EMFUriHandlerConstants.OPTIONS_AUTH_PASSWORD);
		if (username != null && password != null) {

			String userpassword = username + (mandant != null ? "@" + mandant : "") + ":" + password;
			String encodedAuthorization = Base64.getEncoder().encodeToString(userpassword.getBytes());
			httpURLConnection.setRequestProperty(HEADER_AUTHORIZATION, AUTH_BASIC + encodedAuthorization);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.URIHandlerImpl#canHandle(org.eclipse.emf.common.util.URI)
	 */
	@Override
	public boolean canHandle(URI uri) {
		return uri.scheme().equalsIgnoreCase(SCHEMA_HTTP) || uri.scheme().equalsIgnoreCase(SCHEMA_HTTPS);
	}

}