package org.eclipse.fennec.ai.nsc.rest.application;

import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Logger;

import org.eclipse.fennec.ai.pdf.extractor.PDFService;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsName;
import org.osgi.service.jakartars.whiteboard.propertytypes.JakartarsResource;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;

@RequireJakartarsWhiteboard
@JakartarsResource()
@JakartarsName("SchemaPackagesResource")
@Component(name = "NSCResource", service = NSCResource.class, scope = ServiceScope.PROTOTYPE)
@Path("")
public class NSCResource {

	private static final Logger LOGGER = Logger.getLogger(NSCResource.class.getName());

	@Reference
	PDFService pdfService;

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/hello")
	public Response hello() {
		return Response.status(Response.Status.OK).entity("Hello").build();
	}

	@POST
	@Produces({ MediaType.APPLICATION_JSON })
	@Consumes("application/pdf")
	@Path("/upload/{fileName}")
	public Response uploadPDF(
			@PathParam("fileName") String fileName,
			@QueryParam("performTrendAnalysis") Boolean performTrendAnalysis,
			InputStream fileInputStream            
			) {
		try  {
			if(pdfService.reportExists(fileName)) {
				return Response.status(Status.BAD_REQUEST).build();
			}
			String filePath = pdfService.saveAsPDF(fileName, fileInputStream);
			if(performTrendAnalysis) {
				//				TODO: trigger the trend analysis for this document and return TrendAnalysis result
			}
			return Response.status(Response.Status.OK).entity(filePath).build();
		} catch (IOException e) {
			LOGGER.severe("File saving failed: " + e.getMessage());
			return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
					.entity("Failed to save the uploaded file: " + e.getMessage())
					.build();
		}
	}
	
	@GET
	@Produces({"application/pdf" })
	@Path("/report/{reportName}")
	public Response getReport(     
			@PathParam("reportName") String reportName
			) {
		try {
			if(!pdfService.reportExists(reportName)) {
				return Response.status(Status.NOT_FOUND).build();
			}
			byte[] reportData = pdfService.loadFileFromDisk(reportName);
            String filename = reportName.concat(".pdf");
            return Response
                    .status(Response.Status.OK) // HTTP 200 OK
                    .header("Content-Disposition", "attachment; filename=\"".concat(filename).concat("\""))
                    .header("Content-Length", reportData.length)
                    .entity(reportData)
                    .build();
		} catch (IOException e) {
			LOGGER.severe("File retrieval failed: " + e.getMessage());
			return Response.status(Response.Status.INTERNAL_SERVER_ERROR)
					.entity("Failed to retrieve the report file: " + e.getMessage())
					.build();
		}	
	}

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/trend-analysis/{trendAnalysisId}")
	public Response listTrendAnalysis(     
			@PathParam("trendAnalysisId") String trendAnalysisId
			) {

		if(trendAnalysisId == null) {
			//				TODO: list all the trend analysis
			return Response.status(Response.Status.NOT_IMPLEMENTED).build();
		} else {
			//				TODO: fetch that particular trend analysis
			return Response.status(Response.Status.NOT_IMPLEMENTED).build();
		}
	}

	@DELETE
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/trend-analysis/{trendAnalysisId}")
	public Response deleteTrendAnalysis(     
			@PathParam("trendAnalysisId") String trendAnalysisId
			) {
		//				TODO: delete trend analysis 
		return Response.status(Response.Status.NOT_IMPLEMENTED).build();

	}

}
