package org.gecko.emf.ngsi.tests;

import static org.junit.jupiter.api.Assertions.*;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.ngsi.NGSIEntityChangeTracker;
import org.gecko.emf.ngsi.NGSIEntityManager;
import org.gecko.emf.osgi.example.model.basic.BasicFactory;
import org.gecko.emf.osgi.example.model.basic.Person;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.emf.osgi.example.model.basic.Address;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.test.junit5.context.BundleContextExtension;

/**
 * JUnit5 Jupiter test for NGSI entity change tracking using the basic EMF model.
 * 
 * <p>This test suite verifies the functionality of the NGSI entity change tracking system,
 * which monitors changes to EMF model objects and provides notifications when properties
 * are modified, added, or removed.</p>
 * 
 * <h3>Test Coverage:</h3>
 * <ul>
 *   <li>Entity registration and management</li>
 *   <li>Property change detection for attributes and references</li>
 *   <li>Multiple property changes in a single update</li>
 *   <li>Multi-entity management</li>
 *   <li>Entity unregistration and cleanup</li>
 *   <li>No-change scenarios</li>
 * </ul>
 * 
 * <h3>Key Components Tested:</h3>
 * <ul>
 *   <li>{@link NGSIEntityChangeTracker} - Tracks changes to individual entities</li>
 *   <li>{@link NGSIEntityManager} - Manages multiple entities and their change trackers</li>
 *   <li>Change event notifications through custom listeners</li>
 * </ul>
 * 
 * @author Generated with Claude Code
 * @since 1.0.0
 */
@RequireEMF
@ExtendWith(BundleContextExtension.class)
@DisplayName("NGSI Entity Change Tracking Tests")
public class NGSIEntityChangeTrackingTest {

    private NGSIEntityManager entityManager;
    private BasicFactory basicFactory;
    private List<ChangeEvent> capturedEvents;
    private CountDownLatch eventLatch;

    /**
     * Represents a captured change event for testing purposes.
     * 
     * <p>This immutable data structure captures all relevant information about a change event
     * that occurred during entity tracking. It includes the type of change, the affected entity,
     * the property that changed, and the old and new values.</p>
     * 
     * @see NGSIEntityChangeTracker.NGSIChangeListener
     */
    private static class ChangeEvent {
        /** The type of change: "CHANGED", "ADDED", or "REMOVED" */
        public final String type;
        
        /** The entity that experienced the change */
        public final EObject entity;
        
        /** The name of the property that changed */
        public final String propertyName;
        
        /** The previous value of the property (null for ADDED events) */
        public final Object oldValue;
        
        /** The new value of the property (null for REMOVED events) */
        public final Object newValue;

        /**
         * Creates a new change event record.
         * 
         * @param type the type of change (CHANGED, ADDED, REMOVED)
         * @param entity the entity that changed
         * @param propertyName the name of the changed property
         * @param oldValue the previous value (null for ADDED events)
         * @param newValue the new value (null for REMOVED events)
         */
        public ChangeEvent(String type, EObject entity, String propertyName, Object oldValue, Object newValue) {
            this.type = type;
            this.entity = entity;
            this.propertyName = propertyName;
            this.oldValue = oldValue;
            this.newValue = newValue;
        }
    }

    /**
     * Test implementation of {@link NGSIEntityChangeTracker.NGSIChangeListener} that captures
     * change events for verification in test cases.
     * 
     * <p>This listener implementation stores all received change events in a list for later
     * inspection and uses a {@link CountDownLatch} to synchronize test execution with
     * asynchronous change notifications.</p>
     * 
     * <p>The listener handles three types of change events:</p>
     * <ul>
     *   <li><strong>CHANGED</strong>: Property value was modified</li>
     *   <li><strong>ADDED</strong>: New property was added to the entity</li>
     *   <li><strong>REMOVED</strong>: Property was removed from the entity</li>
     * </ul>
     * 
     * @see NGSIEntityChangeTracker.NGSIChangeListener
     */
    private class TestChangeListener implements NGSIEntityChangeTracker.NGSIChangeListener {

        /**
         * Called when a property value is changed.
         * 
         * @param entity the entity that changed
         * @param propertyName the name of the changed property
         * @param oldValue the previous value
         * @param newValue the new value
         */
        @Override
        public void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
            capturedEvents.add(new ChangeEvent("CHANGED", entity, propertyName, oldValue, newValue));
            eventLatch.countDown();
        }

        /**
         * Called when a new property is added to an entity.
         * 
         * @param entity the entity that changed
         * @param propertyName the name of the added property
         * @param newValue the value of the new property
         */
        @Override
        public void onPropertyAdded(EObject entity, String propertyName, Object newValue) {
            capturedEvents.add(new ChangeEvent("ADDED", entity, propertyName, null, newValue));
            eventLatch.countDown();
        }

        /**
         * Called when a property is removed from an entity.
         * 
         * @param entity the entity that changed
         * @param propertyName the name of the removed property
         * @param oldValue the previous value of the removed property
         */
        @Override
        public void onPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
            capturedEvents.add(new ChangeEvent("REMOVED", entity, propertyName, oldValue, null));
            eventLatch.countDown();
        }
    }

    @BeforeEach
    @DisplayName("Set up test environment")
    public void setUp(TestInfo testInfo) {
        System.out.println("Setting up test: " + testInfo.getDisplayName());
        entityManager = new NGSIEntityManager();
        basicFactory = BasicFactory.eINSTANCE;
        capturedEvents = new ArrayList<>();
        eventLatch = new CountDownLatch(1);
    }

    @Test
    @DisplayName("Should register and manage NGSI entities correctly")
    public void testEntityRegistration() {
        // Create a person entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John Doe");
        person.setTest("30");

        String entityId = "urn:ngsi-ld:Person:001";
        
        // Register entity
        entityManager.registerEntity(entityId, person);
        
        // Verify entity is registered
        assertTrue(entityManager.isEntityManaged(entityId));
        assertEquals(1, entityManager.getManagedEntityCount());
        
        // Verify current entity
        EObject currentEntity = entityManager.getCurrentEntity(entityId);
        assertNotNull(currentEntity);
        assertTrue(currentEntity instanceof Person);
        assertEquals("John Doe", ((Person) currentEntity).getFirstName());
        assertEquals("30", ((Person) currentEntity).getTest());
    }

    @Test
    @DisplayName("Should detect and notify property changes")
    public void testPropertyChangeDetection() throws InterruptedException {
        // Create initial person
        Person person1 = basicFactory.createPerson();
        person1.setFirstName("John Doe");
        person1.setTest("30");

        String entityId = "urn:ngsi-ld:Person:001";
        
        // Register entity and add listener
        entityManager.registerEntity(entityId, person1);
        TestChangeListener listener = new TestChangeListener();
        entityManager.addChangeListener(entityId, listener);

        // Create updated person
        Person person2 = basicFactory.createPerson();
        person2.setFirstName("John Smith");  // Changed name
        person2.setTest("30");             // Same age

        // Update entity - should trigger change detection
        eventLatch = new CountDownLatch(1); // Expect 1 change (name)
        entityManager.registerEntity(entityId, person2);

        // Wait for events
        assertTrue(eventLatch.await(5, TimeUnit.SECONDS), "Expected change event within 5 seconds");

        // Verify change was captured
        assertEquals(1, capturedEvents.size());
        ChangeEvent event = capturedEvents.get(0);
        assertEquals("CHANGED", event.type);
        assertEquals("firstName", event.propertyName);
        assertEquals("John Doe", event.oldValue);
        assertEquals("John Smith", event.newValue);
    }

    @Test
    @DisplayName("Should detect multiple property changes in a single update")
    public void testMultiplePropertyChanges() throws InterruptedException {
        // Create initial person
        Person person1 = basicFactory.createPerson();
        person1.setFirstName("John Doe");
        person1.setTest("30");

        String entityId = "urn:ngsi-ld:Person:001";
        
        // Register entity and add listener
        entityManager.registerEntity(entityId, person1);
        TestChangeListener listener = new TestChangeListener();
        entityManager.addChangeListener(entityId, listener);

        // Create updated person with multiple changes
        Person person2 = basicFactory.createPerson();
        person2.setFirstName("Jane Smith");  // Changed name
        person2.setTest("25");             // Changed age

        // Update entity - should trigger multiple change detections
        eventLatch = new CountDownLatch(2); // Expect 2 changes
        entityManager.registerEntity(entityId, person2);

        // Wait for events
        assertTrue(eventLatch.await(5, TimeUnit.SECONDS), "Expected change events within 5 seconds");

        // Verify changes were captured
        assertEquals(2, capturedEvents.size());
        
        // Check that both name and age changes were captured
        boolean foundNameChange = false;
        boolean foundAgeChange = false;
        
        for (ChangeEvent event : capturedEvents) {
            assertEquals("CHANGED", event.type);
            if ("firstName".equals(event.propertyName)) {
                foundNameChange = true;
                assertEquals("John Doe", event.oldValue);
                assertEquals("Jane Smith", event.newValue);
            } else if ("test".equals(event.propertyName)) {
                foundAgeChange = true;
                assertEquals("30", event.oldValue);
                assertEquals("25", event.newValue);
            }
        }
        
        assertTrue(foundNameChange, "Name change should be detected");
        assertTrue(foundAgeChange, "Age change should be detected");
    }

    @Test
    @DisplayName("Should detect changes to reference properties")
    public void testReferencePropertyChanges() throws InterruptedException {
        // Create person with address
        Person person1 = basicFactory.createPerson();
        person1.setFirstName("John Doe");
        
        Address address1 = basicFactory.createAddress();
        address1.setStreet("123 Main St");
        address1.setCity("Anytown");
        person1.setAddress(address1);

        String entityId = "urn:ngsi-ld:Person:001";
        
        // Register entity and add listener
        entityManager.registerEntity(entityId, person1);
        TestChangeListener listener = new TestChangeListener();
        entityManager.addChangeListener(entityId, listener);

        // Create updated person with different address
        Person person2 = basicFactory.createPerson();
        person2.setFirstName("John Doe");  // Same name
        
        Address address2 = basicFactory.createAddress();
        address2.setStreet("456 Oak Ave");
        address2.setCity("Other City");
        person2.setAddress(address2);

        // Update entity - should trigger reference change
        eventLatch = new CountDownLatch(1); // Expect 1 change (address reference)
        entityManager.registerEntity(entityId, person2);

        // Wait for events
        assertTrue(eventLatch.await(5, TimeUnit.SECONDS), "Expected change event within 5 seconds");

        // Verify reference change was captured
        assertEquals(1, capturedEvents.size());
        ChangeEvent event = capturedEvents.get(0);
        assertEquals("CHANGED", event.type);
        assertEquals("address", event.propertyName);
        assertEquals(address1, event.oldValue);
        assertEquals(address2, event.newValue);
    }

    @Test
    @DisplayName("Should manage multiple entities independently")
    public void testMultipleEntities() throws InterruptedException {
        // Create two different entities
        Person person = basicFactory.createPerson();
        person.setFirstName("John Doe");
        person.setTest("30");

        Address address = basicFactory.createAddress();
        address.setStreet("123 Main St");
        address.setCity("Anytown");

        String personId = "urn:ngsi-ld:Person:001";
        String addressId = "urn:ngsi-ld:Address:001";
        
        // Register both entities
        entityManager.registerEntity(personId, person);
        entityManager.registerEntity(addressId, address);
        
        // Verify both are managed
        assertTrue(entityManager.isEntityManaged(personId));
        assertTrue(entityManager.isEntityManaged(addressId));
        assertEquals(2, entityManager.getManagedEntityCount());

        // Add listeners to both
        TestChangeListener listener = new TestChangeListener();
        entityManager.addChangeListener(personId, listener);
        entityManager.addChangeListener(addressId, listener);

        // Update person
        Person updatedPerson = basicFactory.createPerson();
        updatedPerson.setFirstName("Jane Doe");  // Changed name
        updatedPerson.setTest("30");

        eventLatch = new CountDownLatch(1); // Expect 1 change
        entityManager.registerEntity(personId, updatedPerson);

        // Wait for events
        assertTrue(eventLatch.await(5, TimeUnit.SECONDS), "Expected change event within 5 seconds");

        // Verify only person change was captured
        assertEquals(1, capturedEvents.size());
        ChangeEvent event = capturedEvents.get(0);
        assertEquals("CHANGED", event.type);
        assertEquals("firstName", event.propertyName);
        assertTrue(event.entity instanceof Person);
    }

    @Test
    @DisplayName("Should unregister entities and clean up resources")
    public void testEntityUnregistration() {
        // Create and register entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John Doe");
        
        String entityId = "urn:ngsi-ld:Person:001";
        entityManager.registerEntity(entityId, person);
        
        // Verify entity is registered
        assertTrue(entityManager.isEntityManaged(entityId));
        assertEquals(1, entityManager.getManagedEntityCount());
        
        // Unregister entity
        entityManager.unregisterEntity(entityId);
        
        // Verify entity is no longer managed
        assertFalse(entityManager.isEntityManaged(entityId));
        assertEquals(0, entityManager.getManagedEntityCount());
        assertNull(entityManager.getCurrentEntity(entityId));
    }

    @Test
    @DisplayName("Should dispose change trackers and clear all entities")
    public void testChangeTrackerDisposal() {
        // Create and register entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John Doe");
        
        String entityId = "urn:ngsi-ld:Person:001";
        entityManager.registerEntity(entityId, person);
        
        // Get change tracker
        NGSIEntityChangeTracker tracker = entityManager.getChangeTracker(entityId);
        assertNotNull(tracker);
        assertEquals(person, tracker.getCurrentEntity());
        
        // Dispose entity manager
        entityManager.dispose();
        
        // Verify all entities are cleared
        assertEquals(0, entityManager.getManagedEntityCount());
        assertFalse(entityManager.isEntityManaged(entityId));
    }

    @Test
    @DisplayName("Should not trigger events when no changes are detected")
    public void testNoChangesDetection() throws InterruptedException {
        // Create identical persons
        Person person1 = basicFactory.createPerson();
        person1.setFirstName("John Doe");
        person1.setTest("30");

        Person person2 = basicFactory.createPerson();
        person2.setFirstName("John Doe");  // Same name
        person2.setTest("30");           // Same age

        String entityId = "urn:ngsi-ld:Person:001";
        
        // Register entity and add listener
        entityManager.registerEntity(entityId, person1);
        TestChangeListener listener = new TestChangeListener();
        entityManager.addChangeListener(entityId, listener);

        // Update with identical entity - should not trigger changes
        entityManager.registerEntity(entityId, person2);

        // Wait a bit to ensure no events are generated
        Thread.sleep(100);

        // Verify no changes were captured
        assertEquals(0, capturedEvents.size());
    }
}