/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.emf.ngsi.tests;

import static org.junit.jupiter.api.Assertions.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.ngsi.NGSIEntityManager;
import org.gecko.emf.ngsi.NGSISubscription;
import org.gecko.emf.ngsi.NGSISubscriptionManager;
import org.gecko.emf.osgi.example.model.basic.BasicFactory;
import org.gecko.emf.osgi.example.model.basic.Person;
import org.gecko.emf.osgi.example.model.basic.Address;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.test.junit5.context.BundleContextExtension;

/**
 * Comprehensive test suite for NGSI subscription management functionality.
 * 
 * <p>This test suite verifies the complete subscription lifecycle including:</p>
 * <ul>
 *   <li>Subscription creation and management</li>
 *   <li>Entity ID pattern matching</li>
 *   <li>Attribute filtering</li>
 *   <li>Notification delivery</li>
 *   <li>Integration with entity change tracking</li>
 *   <li>Subscription activation/deactivation</li>
 *   <li>Resource cleanup and disposal</li>
 * </ul>
 * 
 * <h3>Test Architecture:</h3>
 * <p>The tests use a mock notification handler to capture and verify that subscription
 * notifications are delivered correctly. The test suite covers both positive and negative
 * scenarios to ensure robust subscription behavior.</p>
 * 
 * <h3>Key Components Tested:</h3>
 * <ul>
 *   <li>{@link NGSISubscriptionManager} - Core subscription management</li>
 *   <li>{@link NGSIEntityManager} - Entity lifecycle integration</li>
 *   <li>Notification handlers and delivery mechanisms</li>
 * </ul>
 * 
 * @author Generated with Claude Code
 * @since 1.0.0
 */
@RequireEMF
@ExtendWith(BundleContextExtension.class)
@DisplayName("NGSI Subscription Manager Tests")
public class NGSISubscriptionManagerTest {

    private NGSISubscriptionManager subscriptionManager;
    private NGSIEntityManager entityManager;
    private BasicFactory basicFactory;
    private List<NotificationEvent> capturedNotifications;
    private CountDownLatch notificationLatch;

    /**
     * Represents a captured notification event for testing purposes.
     * 
     * <p>This immutable data structure captures all relevant information about a
     * subscription notification including the subscription details, entity information,
     * and the specific change that triggered the notification.</p>
     */
    private static class NotificationEvent {
        /** The subscription that triggered the notification */
        public final NGSISubscription subscription;
        
        /** The ID of the entity that changed */
        public final String entityId;
        
        /** The entity object that changed */
        public final EObject entity;
        
        /** The name of the property that changed */
        public final String changedProperty;
        
        /** The previous value of the property */
        public final Object oldValue;
        
        /** The new value of the property */
        public final Object newValue;

        /**
         * Creates a new notification event record.
         * 
         * @param subscription the subscription that matched
         * @param entityId the ID of the changed entity
         * @param entity the changed entity object
         * @param changedProperty the name of the changed property
         * @param oldValue the previous value
         * @param newValue the new value
         */
        public NotificationEvent(NGSISubscription subscription, String entityId, EObject entity, 
                               String changedProperty, Object oldValue, Object newValue) {
            this.subscription = subscription;
            this.entityId = entityId;
            this.entity = entity;
            this.changedProperty = changedProperty;
            this.oldValue = oldValue;
            this.newValue = newValue;
        }
    }

    /**
     * Test implementation of {@link NGSISubscriptionManager.NGSINotificationHandler} 
     * that captures notifications for verification in test cases.
     * 
     * <p>This handler stores all received notifications and provides synchronization
     * mechanisms for test coordination through {@link CountDownLatch}.</p>
     */
    private class TestNotificationHandler implements NGSISubscriptionManager.NGSINotificationHandler {
        
        @Override
        public void onNotification(NGSISubscription subscription, String entityId, EObject entity, 
                                 String changedProperty, Object oldValue, Object newValue) {
            capturedNotifications.add(new NotificationEvent(subscription, entityId, entity, 
                                                           changedProperty, oldValue, newValue));
            notificationLatch.countDown();
        }
    }

    @BeforeEach
    @DisplayName("Set up test environment")
    public void setUp(TestInfo testInfo) {
        System.out.println("Setting up subscription test: " + testInfo.getDisplayName());
        
        subscriptionManager = new NGSISubscriptionManager();
        entityManager = new NGSIEntityManager();
        basicFactory = BasicFactory.eINSTANCE;
        capturedNotifications = new ArrayList<>();
        notificationLatch = new CountDownLatch(1);
        
        // Connect the managers
        subscriptionManager.attachToEntityManager(entityManager);
        entityManager.registerSubscriptionManager(subscriptionManager);
    }

    @Test
    @DisplayName("Should create subscriptions with valid parameters")
    public void testSubscriptionCreation() {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription for person entities
        NGSISubscription subscription = subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("firstName", "lastName"),
            handler
        );
        
        assertNotNull(subscription);
        assertEquals(1, subscriptionManager.getTotalSubscriptionCount());
        assertEquals(1, subscriptionManager.getActiveSubscriptionCount());
        assertTrue(subscriptionManager.getActiveSubscriptionIds().contains("urn:ngsi-ld:Subscription:001"));
    }

    @Test
    @DisplayName("Should reject invalid subscription parameters")
    public void testInvalidSubscriptionParameters() {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Test null subscription ID
        assertThrows(IllegalArgumentException.class, () -> {
            subscriptionManager.createSubscription(null, "urn:ngsi-ld:Person:.*", null, handler);
        });
        
        // Test empty subscription ID
        assertThrows(IllegalArgumentException.class, () -> {
            subscriptionManager.createSubscription("", "urn:ngsi-ld:Person:.*", null, handler);
        });
        
        // Test null notification handler
        assertThrows(IllegalArgumentException.class, () -> {
            subscriptionManager.createSubscription("urn:ngsi-ld:Subscription:001", 
                                                  "urn:ngsi-ld:Person:.*", null, null);
        });
        
        // Test duplicate subscription ID
        subscriptionManager.createSubscription("urn:ngsi-ld:Subscription:001", 
                                              "urn:ngsi-ld:Person:.*", null, handler);
        
        assertThrows(IllegalArgumentException.class, () -> {
            subscriptionManager.createSubscription("urn:ngsi-ld:Subscription:001", 
                                                  "urn:ngsi-ld:Person:.*", null, handler);
        });
    }

    @Test
    @DisplayName("Should filter notifications by entity ID pattern")
    public void testEntityIdPatternFiltering() throws InterruptedException {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription for temperature sensors only
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:TemperatureSensor:.*",
            null,
            handler
        );
        
        // Create person entity (should not match)
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Update person (should not trigger notification)
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Wait briefly to ensure no notifications
        Thread.sleep(100);
        
        assertEquals(0, capturedNotifications.size());
        
        // Now test with matching entity ID
        Person sensor = basicFactory.createPerson(); // Using Person as sensor placeholder
        sensor.setFirstName("25.5");
        entityManager.registerEntity("urn:ngsi-ld:TemperatureSensor:001", sensor);
        
        // Update sensor (should trigger notification)
        notificationLatch = new CountDownLatch(1);
        sensor.setFirstName("26.0");
        entityManager.registerEntity("urn:ngsi-ld:TemperatureSensor:001", sensor);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for matching entity ID");
        
        assertEquals(1, capturedNotifications.size());
        NotificationEvent event = capturedNotifications.get(0);
        assertEquals("urn:ngsi-ld:TemperatureSensor:001", event.entityId);
        assertEquals("firstName", event.changedProperty);
        assertEquals("25.5", event.oldValue);
        assertEquals("26.0", event.newValue);
    }

    @Test
    @DisplayName("Should filter notifications by monitored attributes")
    public void testAttributeFiltering() throws InterruptedException {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription monitoring only firstName attribute
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("firstName"),
            handler
        );
        
        // Create person entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        person.setTest("30");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Update monitored attribute (should trigger notification)
        notificationLatch = new CountDownLatch(1);
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for monitored attribute");
        
        assertEquals(1, capturedNotifications.size());
        
        // Update non-monitored attribute (should not trigger notification)
        person.setTest("31");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Wait briefly to ensure no additional notifications
        Thread.sleep(100);
        
        assertEquals(1, capturedNotifications.size()); // Should still be 1
    }

    @Test
    @DisplayName("Should handle subscription activation and deactivation")
    public void testSubscriptionActivation() throws InterruptedException {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            null,
            handler
        );
        
        // Create and update entity (should trigger notification)
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        notificationLatch = new CountDownLatch(1);
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for active subscription");
        
        assertEquals(1, capturedNotifications.size());
        
        // Deactivate subscription
        assertTrue(subscriptionManager.setSubscriptionActive("urn:ngsi-ld:Subscription:001", false));
        assertEquals(0, subscriptionManager.getActiveSubscriptionCount());
        
        // Update entity (should not trigger notification)
        person.setFirstName("Bob");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Wait briefly to ensure no additional notifications
        Thread.sleep(100);
        
        assertEquals(1, capturedNotifications.size()); // Should still be 1
        
        // Reactivate subscription
        assertTrue(subscriptionManager.setSubscriptionActive("urn:ngsi-ld:Subscription:001", true));
        assertEquals(1, subscriptionManager.getActiveSubscriptionCount());
        
        // Update entity (should trigger notification again)
        notificationLatch = new CountDownLatch(1);
        person.setFirstName("Alice");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for reactivated subscription");
        
        assertEquals(2, capturedNotifications.size());
    }

    @Test
    @DisplayName("Should handle multiple subscriptions for same entity")
    public void testMultipleSubscriptions() throws InterruptedException {
        TestNotificationHandler handler1 = new TestNotificationHandler();
        TestNotificationHandler handler2 = new TestNotificationHandler();
        
        // Create two subscriptions for the same entity type
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("firstName"),
            handler1
        );
        
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:002",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("test"),
            handler2
        );
        
        assertEquals(2, subscriptionManager.getActiveSubscriptionCount());
        
        // Create person entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        person.setTest("30");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Update firstName (should trigger first subscription only)
        notificationLatch = new CountDownLatch(1);
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for firstName change");
        
        assertEquals(1, capturedNotifications.size());
        assertEquals("firstName", capturedNotifications.get(0).changedProperty);
        
        // Update test attribute (should trigger second subscription)
        notificationLatch = new CountDownLatch(1);
        capturedNotifications.clear();
        person.setTest("31");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for test attribute change");
        
        assertEquals(1, capturedNotifications.size());
        assertEquals("test", capturedNotifications.get(0).changedProperty);
    }

    @Test
    @DisplayName("Should handle subscription removal")
    public void testSubscriptionRemoval() throws InterruptedException {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            null,
            handler
        );
        
        assertEquals(1, subscriptionManager.getTotalSubscriptionCount());
        
        // Remove subscription
        assertTrue(subscriptionManager.removeSubscription("urn:ngsi-ld:Subscription:001"));
        assertEquals(0, subscriptionManager.getTotalSubscriptionCount());
        assertEquals(0, subscriptionManager.getActiveSubscriptionCount());
        
        // Verify removal of non-existent subscription
        assertFalse(subscriptionManager.removeSubscription("urn:ngsi-ld:Subscription:999"));
        
        // Create and update entity (should not trigger notification)
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Wait briefly to ensure no notifications
        Thread.sleep(100);
        
        assertEquals(0, capturedNotifications.size());
    }

    @Test
    @DisplayName("Should handle reference property changes")
    public void testReferencePropertySubscriptions() throws InterruptedException {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription for address changes
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("address"),
            handler
        );
        
        // Create person with address
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        
        Address address1 = basicFactory.createAddress();
        address1.setStreet("123 Main St");
        person.setAddress(address1);
        
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Update address reference (should trigger notification)
        notificationLatch = new CountDownLatch(1);
        Address address2 = basicFactory.createAddress();
        address2.setStreet("456 Oak Ave");
        person.setAddress(address2);
        
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertTrue(notificationLatch.await(5, TimeUnit.SECONDS), 
                  "Expected notification for address change");
        
        assertEquals(1, capturedNotifications.size());
        NotificationEvent event = capturedNotifications.get(0);
        assertEquals("address", event.changedProperty);
        assertEquals(address1, event.oldValue);
        assertEquals(address2, event.newValue);
    }

    @Test
    @DisplayName("Should handle entity unregistration")
    public void testEntityUnregistration() {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            null,
            handler
        );
        
        // Register entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Unregister entity
        entityManager.unregisterEntity("urn:ngsi-ld:Person:001");
        
        // Verify entity is no longer managed
        assertFalse(entityManager.isEntityManaged("urn:ngsi-ld:Person:001"));
        
        // Subscription should still exist
        assertEquals(1, subscriptionManager.getActiveSubscriptionCount());
    }

    @Test
    @DisplayName("Should handle manager disposal")
    public void testManagerDisposal() {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            null,
            handler
        );
        
        // Register entity
        Person person = basicFactory.createPerson();
        person.setFirstName("John");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        assertEquals(1, subscriptionManager.getTotalSubscriptionCount());
        assertEquals(1, entityManager.getManagedEntityCount());
        
        // Dispose subscription manager
        subscriptionManager.dispose();
        
        assertEquals(0, subscriptionManager.getTotalSubscriptionCount());
        assertEquals(0, subscriptionManager.getActiveSubscriptionCount());
        
        // Dispose entity manager
        entityManager.dispose();
        
        assertEquals(0, entityManager.getManagedEntityCount());
    }

    @Test
    @DisplayName("Should handle subscription retrieval")
    public void testSubscriptionRetrieval() {
        TestNotificationHandler handler = new TestNotificationHandler();
        
        // Create subscription
        NGSISubscription subscription = subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",
            null,
            handler
        );
        
        // Retrieve subscription
        NGSISubscription retrieved = subscriptionManager.getSubscription("urn:ngsi-ld:Subscription:001");
        assertNotNull(retrieved);
        assertEquals(subscription, retrieved);
        
        // Test non-existent subscription
        assertNull(subscriptionManager.getSubscription("urn:ngsi-ld:Subscription:999"));
    }

    @Test
    @DisplayName("Should handle concurrent subscription operations")
    public void testConcurrentOperations() throws InterruptedException {
        int numThreads = 5;
        int subscriptionsPerThread = 10;
        CountDownLatch startLatch = new CountDownLatch(1);
        CountDownLatch completionLatch = new CountDownLatch(numThreads);
        AtomicInteger successCount = new AtomicInteger(0);
        
        // Create multiple threads that create subscriptions concurrently
        for (int i = 0; i < numThreads; i++) {
            final int threadId = i;
            Thread thread = new Thread(() -> {
                try {
                    startLatch.await();
                    
                    for (int j = 0; j < subscriptionsPerThread; j++) {
                        String subscriptionId = String.format("urn:ngsi-ld:Subscription:%d-%d", threadId, j);
                        TestNotificationHandler handler = new TestNotificationHandler();
                        
                        try {
                            subscriptionManager.createSubscription(subscriptionId, 
                                                                 "urn:ngsi-ld:Person:.*", 
                                                                 null, handler);
                            successCount.incrementAndGet();
                        } catch (Exception e) {
                            // Expected some conflicts in concurrent scenario
                        }
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } finally {
                    completionLatch.countDown();
                }
            });
            thread.start();
        }
        
        // Start all threads
        startLatch.countDown();
        
        // Wait for completion
        assertTrue(completionLatch.await(10, TimeUnit.SECONDS), 
                  "Concurrent operations should complete within timeout");
        
        // Verify some subscriptions were created successfully
        assertTrue(successCount.get() > 0, "At least some subscriptions should be created");
        assertEquals(successCount.get(), subscriptionManager.getTotalSubscriptionCount());
    }
}