/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.mcp.api;

import java.util.Dictionary;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.http.whiteboard.annotations.RequireHttpWhiteboard;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.modelcontextprotocol.json.jackson.JacksonMcpJsonMapper;
import io.modelcontextprotocol.json.schema.jackson.DefaultJsonSchemaValidator;
import io.modelcontextprotocol.server.McpAsyncServer;
import io.modelcontextprotocol.server.McpServer;
import io.modelcontextprotocol.server.transport.HttpServletStreamableServerTransportProvider;
import io.modelcontextprotocol.spec.McpSchema;
import jakarta.servlet.Servlet;

/**
 * 
 * @author ilenia
 * @since Dec 2, 2025
 */
@RequireHttpWhiteboard
public abstract class AbstractHttpMCPServer implements MCPServer {

	private ServiceRegistration<?> servletRegistration;
	protected McpAsyncServer mcpServer;
	protected BundleContext context;

	protected abstract Dictionary<String, Object> getServletProperties();
	
	protected abstract String getServerName();
	
	protected abstract String getServerVersion();
	
	protected abstract String getEndpointPath();
	
	protected abstract boolean hasToolCapability();
	
	protected abstract boolean hasPromptCapability();
	
	protected abstract boolean hasResourceCapability();
	

	protected void initializeMCPServer() {
		HttpServletStreamableServerTransportProvider transportProvider = HttpServletStreamableServerTransportProvider.builder()
				.jsonMapper(new JacksonMcpJsonMapper(new ObjectMapper()))
				.mcpEndpoint(getEndpointPath())
				.build();

		registerHttpTransportProvider(transportProvider, context);

		mcpServer = McpServer.async(transportProvider)
				.serverInfo(getServerName(), getServerVersion())
				.jsonMapper(new JacksonMcpJsonMapper(new ObjectMapper()))
				.jsonSchemaValidator(new DefaultJsonSchemaValidator())
				.capabilities(McpSchema.ServerCapabilities.builder()
						.tools(hasToolCapability())
						.resources(hasResourceCapability(), hasResourceCapability())
						.prompts(hasPromptCapability())
						.logging()
						.build())
				.tools(getTools())
				.prompts(getPrompts())
				.resources(getResources())
				.build();
	}

	protected void registerHttpTransportProvider(HttpServletStreamableServerTransportProvider transportProvider, BundleContext context) {
		servletRegistration = context.registerService(
				Servlet.class,
				transportProvider,
				getServletProperties()
				);
	}

	protected void unregisterHttpTransportProvider() {
		if(mcpServer != null) {
			mcpServer.close();
			mcpServer = null;
		}		
		if(servletRegistration != null) {
			servletRegistration.unregister();
			servletRegistration = null;
		}
	}
}
