/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.mcp.http.component;

import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer;
import org.eclipse.fennec.ai.mcp.api.MCPToolProvider;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.metatype.annotations.Designate;

import io.modelcontextprotocol.server.McpServerFeatures.AsyncPromptSpecification;
import io.modelcontextprotocol.server.McpServerFeatures.AsyncResourceSpecification;
import io.modelcontextprotocol.server.McpServerFeatures.AsyncToolSpecification;

/**
 * 
 * @author ilenia
 * @since Dec 2, 2025
 */
@Component(name = "HttpMCPServerComponent", configurationPid = "HttpMCPServerComponent", configurationPolicy = ConfigurationPolicy.REQUIRE)
@Designate(ocd = HttpMCPServerConfig.class)
public class HttpMCPServerComponent extends AbstractHttpMCPServer{
	
	
	
	private HttpMCPServerConfig config;
	private List<MCPToolProvider> toolProviders;

	@Activate
	public HttpMCPServerComponent(
			HttpMCPServerConfig config, 
			BundleContext context, 
			@Reference(name = "toolProviders", cardinality = ReferenceCardinality.AT_LEAST_ONE) 
			List<MCPToolProvider> toolProviders) {
		this.config = config;
		this.context = context;
		this.toolProviders = toolProviders;
		initializeMCPServer();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.MCPServer#getTools()
	 */
	@Override
	public List<AsyncToolSpecification> getTools() {
		return toolProviders.stream().map(p -> p.getMCPTools()).flatMap(List::stream).toList();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.MCPServer#getPrompts()
	 */
	@Override
	public List<AsyncPromptSpecification> getPrompts() {
		return Collections.emptyList();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.MCPServer#getResources()
	 */
	@Override
	public List<AsyncResourceSpecification> getResources() {
		return Collections.emptyList();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#getServletProperties()
	 */
	protected Dictionary<String, Object> getServletProperties() {
		Dictionary<String, Object> servletProps = new Hashtable<>();
        servletProps.put("osgi.http.whiteboard.servlet.pattern", config.osgi_http_whiteboard_servlet_pattern());
        servletProps.put("osgi.http.whiteboard.servlet.name", config.server_name());
        servletProps.put("osgi.http.whiteboard.servlet.asyncSupported", true);
        servletProps.put("osgi.http.whiteboard.target", config.osgi_http_whiteboard_target());
		return servletProps;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#getServerName()
	 */
	@Override
	protected String getServerName() {
		return config.server_name();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#getServerVersion()
	 */
	@Override
	protected String getServerVersion() {
		return config.server_version();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#hasToolCapability()
	 */
	@Override
	protected boolean hasToolCapability() {
		return config.has_tool_capability();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#hasPromptCapability()
	 */
	@Override
	protected boolean hasPromptCapability() {
		return config.has_prompt_capability();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#hasResourceCapability()
	 */
	@Override
	protected boolean hasResourceCapability() {
		return config.has_resource_capability();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.AbstractHttpMCPServer#getEndpointPath()
	 */
	@Override
	protected String getEndpointPath() {
		return config.osgi_http_whiteboard_servlet_pattern();
	};
	


}
