package org.eclipse.fennec.ai.mcp.structured.output.handler;

import static java.util.Objects.requireNonNull;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.fennec.ai.mcp.api.StructuredOutputHandler;
import org.eclipse.fennec.codec.options.CodecModuleOptions;
import org.eclipse.fennec.codec.options.CodecResourceOptions;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.Designate;

import tools.jackson.databind.ObjectMapper;

@Component(name = "StructuredOutputFileStorage", configurationPid = "StructuredOutputFileStorage", configurationPolicy = ConfigurationPolicy.REQUIRE)
@Designate(ocd = StructuredOutputFileStorageConfig.class)
public class StructuredOutputFileStorage implements StructuredOutputHandler {

	private static final Logger LOGGER = Logger.getLogger(StructuredOutputFileStorage.class.getName());
	private ResourceSet resourceSet;
	private StructuredOutputFileStorageConfig config;

	@Activate
	public StructuredOutputFileStorage(StructuredOutputFileStorageConfig config, 
			@Reference ResourceSet resourceSet) {
		requireNonNull(config.storage_folder(), "storage.folder property must be provided");
		requireNonNull(config.root_eclass_uri(), "root.eclass.uri property must be provided");
		this.config = config;
		this.resourceSet = resourceSet;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.StructuredOutputHandler#handleStructuredOutput(java.lang.Object)
	 */
	@Override
	public String handleStructuredOutput(Map<String, Object> propertyMap) {
		EObject loadedEObject = loadEObject(config.root_eclass_uri(), propertyMap);
		String fileName = config.storage_folder().concat(UUID.randomUUID().toString()).concat(config.file_extension());
		Resource saveRes = resourceSet.createResource(URI.createFileURI(fileName));
		saveRes.getContents().add(loadedEObject);
		try {
			saveRes.save(null);
			return String.format("Successfully saved Strcutured Output to file %s", fileName);
		} catch (IOException e) {
			LOGGER.severe(String.format("IOExcpetion when trying to save structured output EObject of EClass %s", config.root_eclass_uri()));
			e.printStackTrace();
			return String.format("Error while saving Strcutured Output to file %s", fileName);
		}

	}

	private EObject loadEObject(String classUri, Map<String, Object> propertyMap)  {
		try {
			String jsonString = mapToJsonString(propertyMap);
			InputStream inputStream = stringToInputStream(jsonString);
			Resource resource = resourceSet.createResource(URI.createURI("temp.json"));
			EObject eClassEO = resourceSet.getEObject(URI.createURI(classUri), false);
			Map<String, Object> options = new HashMap<>();
			options.put(CodecResourceOptions.CODEC_ROOT_OBJECT, eClassEO);
			options.put(CodecModuleOptions.CODEC_MODULE_USE_NAMES_FROM_EXTENDED_METADATA, true);
			resource.load(inputStream, options);
			if(!resource.getContents().isEmpty() && EcoreUtil.getURI(resource.getContents().get(0).eClass()).toString().equals(classUri)) {
				return resource.getContents().get(0);
			}
			
		} catch (IOException e) {
			LOGGER.severe(String.format("IOExcpetion when trying to load structured output into known EObject of EClass %s", classUri));
			e.printStackTrace();
		}
		return null;
		
	}
	
	private InputStream stringToInputStream(String jsonString) {
	    // Wrap the string bytes in a ByteArrayInputStream, using UTF-8 encoding
	    return new ByteArrayInputStream(jsonString.getBytes(StandardCharsets.UTF_8));
	}
	
	private String mapToJsonString(Map<String, Object> propertyMap) throws IOException {
	    ObjectMapper mapper = new ObjectMapper();
	    
	    // Convert the Map into a JSON formatted String
	    String jsonString = mapper.writeValueAsString(propertyMap);
	    
	    // Note: The resulting JSON string needs to be wrapped if your EMF resource expects a root element.
	    // If your EMF model root object is mapped directly from the properties, this is often sufficient.
	    
	    return jsonString;
	}
}
