package org.eclipse.fennec.ai.mcp.tool.provider;

import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import org.eclipse.fennec.ai.mcp.api.MCPToolProvider;
import org.eclipse.fennec.ai.pdf.extractor.PDFService;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.modelcontextprotocol.json.jackson.JacksonMcpJsonMapper;
import io.modelcontextprotocol.server.McpAsyncServerExchange;
import io.modelcontextprotocol.server.McpServerFeatures;
import io.modelcontextprotocol.server.McpServerFeatures.AsyncToolSpecification;
import io.modelcontextprotocol.spec.McpSchema;
import io.modelcontextprotocol.spec.McpSchema.CallToolRequest;
import io.modelcontextprotocol.spec.McpSchema.Tool;
import reactor.core.publisher.Mono;

@Component(immediate = true, name = "PDFToolProvider")
public class PDFToolProvider implements MCPToolProvider{
	
	@Reference
	PDFService pdfExtractor;

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.MCPToolProvider#getMCPTools()
	 */
	@Override
	public List<AsyncToolSpecification> getMCPTools() {
		return List.of(getPDFPageExtractTool(), getPDFToCTool(), getPDFContentTool());
	}

	private McpServerFeatures.AsyncToolSpecification getPDFPageExtractTool() {

		String schema = """

				{
				   "type": "object",
				    "properties": {
				      "file_name": {
				        "type": "string",
				        "description": "The file name of the pdf document to look for, with the pdf extension (e.g. MyReport.pdf)"
				      },
				      "start_page": {
				        "type": "integer",
				        "description": "The number of the start page from which we want the pdf content to be extracted from"
				      },
				      "end_page": {
				        "type": "integer",
				        "description": "The number of the end page from which we want the pdf content to be extracted from"
				      }
				    },
				    "required": [
				      "file_name", "start_page", "end_page" 
				    ]


				}
				""";
		Tool tool = new McpSchema.Tool.Builder().name("extract_pdf_page_content").description("Get the content of the specified pdf document within the specified page range.").inputSchema(new JacksonMcpJsonMapper(new ObjectMapper()), schema).build();
		BiFunction<McpAsyncServerExchange, CallToolRequest, Mono<McpSchema.CallToolResult>> callHandler = (exchange, request) -> {

			Map<String, Object> arguments = request.arguments();
			String fileName = (String) arguments.get("file_name");
			int startPage = (int) arguments.get("start_page");
			int endPage = (int) arguments.get("end_page");	
			Mono<McpSchema.CallToolResult> mono = Mono.just(McpSchema.CallToolResult.builder().addTextContent(pdfExtractor.extractTextFromPages(fileName, startPage, endPage)).build());
			return mono;
		};
		var syncToolSpecification =  McpServerFeatures.AsyncToolSpecification.builder().tool(tool).callHandler(callHandler).build();
		return syncToolSpecification;
	}
	
	private McpServerFeatures.AsyncToolSpecification getPDFContentTool() {

		String schema = """

				{
				   "type": "object",
				    "properties": {
				      "file_name": {
				        "type": "string",
				        "description": "The file name of the pdf document to look for, with the pdf extension (e.g. MyReport.pdf)"
				      }
				    },
				    "required": [
				      "file_name"
				    ]


				}
				""";
		Tool tool = new McpSchema.Tool.Builder().name("extract_pdf_content").description("Get the entire content of the specified pdf document.").inputSchema(new JacksonMcpJsonMapper(new ObjectMapper()), schema).build();
		BiFunction<McpAsyncServerExchange, CallToolRequest, Mono<McpSchema.CallToolResult>> callHandler = (exchange, request) -> {

			Map<String, Object> arguments = request.arguments();
			String fileName = (String) arguments.get("file_name");	
			Mono<McpSchema.CallToolResult> mono = Mono.just(McpSchema.CallToolResult.builder().addTextContent(pdfExtractor.getPDFDocument(fileName)).build());
			return mono;
		};
		var syncToolSpecification =  McpServerFeatures.AsyncToolSpecification.builder().tool(tool).callHandler(callHandler).build();
		return syncToolSpecification;
	}
	
	private McpServerFeatures.AsyncToolSpecification getPDFToCTool() {

		String schema = """

				{
				   "type": "object",
				    "properties": {
				      "file_name": {
				        "type": "string",
				        "description": "The file name of the pdf document to look for, with the pdf extension (e.g. MyReport.pdf)"
				      }
				    },
				    "required": [
				      "file_name"
				    ]


				}
				""";
		Tool tool = new McpSchema.Tool.Builder().name("extract_pdf_toc").description("Get the Table of Content of the specified pdf document.").inputSchema(new JacksonMcpJsonMapper(new ObjectMapper()), schema).build();
		BiFunction<McpAsyncServerExchange, CallToolRequest, Mono<McpSchema.CallToolResult>> callHandler = (exchange, request) -> {

			Map<String, Object> arguments = request.arguments();
			String fileName = (String) arguments.get("file_name");	
			Mono<McpSchema.CallToolResult> mono = Mono.just(McpSchema.CallToolResult.builder().addTextContent(pdfExtractor.extractTableOfContents(fileName)).build());
			return mono;
		};
		var syncToolSpecification =  McpServerFeatures.AsyncToolSpecification.builder().tool(tool).callHandler(callHandler).build();
		return syncToolSpecification;
	}
	
}
