/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.mcp.tool.provider;

import static java.util.Objects.requireNonNull;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import org.eclipse.fennec.ai.mcp.api.MCPToolProvider;
import org.eclipse.fennec.ai.mcp.api.StructuredOutputHandler;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.metatype.annotations.Designate;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.modelcontextprotocol.json.jackson.JacksonMcpJsonMapper;
import io.modelcontextprotocol.server.McpAsyncServerExchange;
import io.modelcontextprotocol.server.McpServerFeatures;
import io.modelcontextprotocol.server.McpServerFeatures.AsyncToolSpecification;
import io.modelcontextprotocol.spec.McpSchema;
import io.modelcontextprotocol.spec.McpSchema.CallToolRequest;
import io.modelcontextprotocol.spec.McpSchema.Tool;
import reactor.core.publisher.Mono;

/**
 * 
 * @author ilenia
 * @since Dec 3, 2025
 */
@Component(name = "StructuredOutputTool", configurationPid = "StructuredOutputTool", configurationPolicy = ConfigurationPolicy.REQUIRE)
@Designate(ocd = StructuredOutputToolConfig.class)
public class StructuredOutputTool implements MCPToolProvider {
	
	@Reference(cardinality = ReferenceCardinality.MANDATORY)
	StructuredOutputHandler structuredOutputHandler;
	
	private final String structuredOutputSchema;
	private StructuredOutputToolConfig config;

	@Activate
	public StructuredOutputTool(StructuredOutputToolConfig config) throws IOException {
		requireNonNull(config.tool_name(), "tool.name property must be provided");
		requireNonNull(config.tool_description(), "tool.description property must be provided");
		requireNonNull(config.schema_folder_path(), "schema.folder.path property must be provided");
		requireNonNull(config.schema_file_name(), "schema.file.name property must be provided");
		this.config = config;
		structuredOutputSchema = loadOutputSchema(config.schema_folder_path().concat(config.schema_file_name()));
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.mcp.api.MCPToolProvider#getMCPTools()
	 */
	@Override
	public List<AsyncToolSpecification> getMCPTools() {
		return List.of(getStructuredOutputTool());
	}
	
	@SuppressWarnings("unchecked")
	private AsyncToolSpecification getStructuredOutputTool() {

		String schema = structuredOutputSchema;
		Tool tool = new McpSchema.Tool.Builder().name(config.tool_name()).description(config.tool_description()).inputSchema(new JacksonMcpJsonMapper(new ObjectMapper()), schema).build();
		BiFunction<McpAsyncServerExchange, CallToolRequest, Mono<McpSchema.CallToolResult>> callHandler = (exchange, request) -> {

			Map<String, Object> arguments = request.arguments();
			if(config.output_root_object() != null) arguments =  (Map<String, Object>) arguments.get(config.output_root_object());	
			Mono<McpSchema.CallToolResult> mono = Mono.just(McpSchema.CallToolResult.builder().addTextContent(structuredOutputHandler.handleStructuredOutput(arguments)).build());
			return mono;
		};
		var syncToolSpecification =  McpServerFeatures.AsyncToolSpecification.builder().tool(tool).callHandler(callHandler).build();
		return syncToolSpecification;
	}
	
	private String loadOutputSchema(String schemaPath) throws IOException {
		Path path = Paths.get(schemaPath);

	    // Check if the file exists before attempting to read it
	    if (!Files.exists(path)) {
	        throw new IOException("File not found at path: " + schemaPath);
	    }

	    // Read all bytes from the file into a String
	    return Files.readString(path, StandardCharsets.UTF_8);
	}

}
