/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.pdf.extractor;

import static java.util.Objects.requireNonNull;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.logging.Logger;

import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.encryption.AccessPermission;
import org.apache.pdfbox.pdmodel.interactive.documentnavigation.outline.PDDocumentOutline;
import org.apache.pdfbox.pdmodel.interactive.documentnavigation.outline.PDOutlineItem;
import org.apache.pdfbox.pdmodel.interactive.documentnavigation.outline.PDOutlineNode;
import org.apache.pdfbox.text.PDFTextStripper;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.metatype.annotations.Designate;

/**
 * 
 * @author ilenia
 * @since Nov 14, 2025
 */
@Component(name = "PDFService", service = PDFService.class, configurationPid = "PDFService", configurationPolicy = ConfigurationPolicy.REQUIRE)
@Designate(ocd = PDFServiceConfig.class)
public class PDFService {

	private static final Logger LOGGER = Logger.getLogger(PDFService.class.getName());
	private PDFServiceConfig config;

	@Activate
	public void activate(PDFServiceConfig config) {
		requireNonNull(config.data_folder(), "data.folder property must be provided");	
		this.config = config;
	}


	public String saveAsPDF(String fileName, InputStream fileInputStream) throws IOException{
		// --- 2. Determine File Path ---
		// Note: When using the simplified approach, you need another way to get the filename.
		// For this example, we'll use a UUID to ensure a unique name.
		String uniqueFileName = fileName + ".pdf";
		Path targetPath = Paths.get(config.data_folder(), uniqueFileName);


		// --- 3. Ensure Directory Exists (Crucial for stability) ---
		Files.createDirectories(targetPath.getParent());

		// --- 4. Write the Stream to File (The core logic) ---
		// Files.copy reads the entire InputStream and writes it to the target path.
		// StandardCopyOption.REPLACE_EXISTING is optional, used if a file with the same name might exist.
		long bytesWritten = Files.copy(
				fileInputStream, 
				targetPath, 
				StandardCopyOption.REPLACE_EXISTING
				);

		// --- 5. Return Success Response ---
		LOGGER.info("Saved " + bytesWritten + " bytes to: " + targetPath.toAbsolutePath());
		return targetPath.toAbsolutePath().toString();

	}
	
	public boolean reportExists(String fileName) {
		String uniqueFileName = fileName + ".pdf";
		Path path = Paths.get(config.data_folder(), uniqueFileName);
		if (!Files.exists(path)) return false;
		return true;
	}
	
	/**
     * Reads the entire content of a file from the disk into a byte array.
     * This is suitable for serving file content via a REST API.
     *
     */
    public byte[] loadFileFromDisk(String fileName) throws IOException {
        // 1. Create a Path object from the string path
    	String uniqueFileName = fileName + ".pdf";
		Path path = Paths.get(config.data_folder(), uniqueFileName);

        // 2. Check if the file exists before attempting to read
        if (!Files.exists(path)) {
            // Throw a specific exception for clarity
            throw new IOException("File not found at path: " + fileName);
        }

        // 3. Use Files.readAllBytes() to efficiently read all content
        // This is a simple and common way to load small to medium files.
        return Files.readAllBytes(path);
    }

	public String getPDFDocument(String fileName) {
		return extractTextFromPages(fileName, 1, 0);
	}


	/**
	 * Extracts text from a specific range of pages in a PDF document.
	 * * NOTE: PDFBox uses 1-based indexing for pages (Page 1 is the first page).
	 * * @param filePath The path to the PDF file.
	 * @param startPage The first page number to extract (inclusive, 1-based).
	 * @param endPage The last page number to extract (inclusive, 1-based).
	 * @return The extracted text from the specified page range.
	 */
	public String extractTextFromPages(String fileName, int startPage, int endPage) {

		String filePath = config.data_folder() + fileName;

		// Use try-with-resources to ensure the document is closed, even if an error occurs
		try (PDDocument document = Loader.loadPDF(new File(filePath))) {

			// 1. Get the total number of pages in the document
			int totalPages = document.getNumberOfPages();     
			if(endPage == 0) endPage = totalPages;

			// 2. Validate page range input
			if (startPage < 1 || startPage > totalPages || startPage > endPage) {
				// Return an error message or empty string, as appropriate for your tool
				return "Error: Invalid page range or start page exceeds end page. Document has " + totalPages + " pages.";
			}

			// Ensure the end page doesn't exceed the total pages
			if (endPage > totalPages) {
				endPage = totalPages;
			}

			if(document.isEncrypted()) {
				return String.format("Error: Cannot extract content from PDF %s because of encryption issues", fileName);
			}
			AccessPermission ap = document.getCurrentAccessPermission();
			if(!ap.canExtractContent()) {
				return String.format("Error: Cannot extract content from PDF %s because of permission issues", fileName);
			}

			// 3. Configure the PDFTextStripper
			PDFTextStripper stripper = new PDFTextStripper();

			// Set the range for text extraction
			stripper.setStartPage(startPage);
			stripper.setEndPage(endPage);

			stripper.setSortByPosition(true);
			// 4. Perform the extraction
			String extractedText = stripper.getText(document);

			return extractedText;
		} catch(IOException e) {
			e.printStackTrace();
			return String.format("Error: IOException when loading the PDF %s", fileName);
		}
		// Note: IOException is thrown for file loading or reading issues
	}

	public String extractTableOfContents(String fileName) {
		String pdfPath = config.data_folder() + fileName;

		// Use try-with-resources to ensure the document is closed
		try (PDDocument document = Loader.loadPDF(new File(pdfPath))) {

			// 1. Get the main document outline structure
			PDDocumentOutline outline = document.getDocumentCatalog().getDocumentOutline();

			if (outline != null) {
				

				// 2. Start recursion with the first bookmark item
				return printOutline(outline, document, 0);
			} else {
				LOGGER.warning("PDF does not contain a Document Outline (TOC/Bookmarks).");
				return String.format("PDF %s does not contain a Document Outline (TOC/Bookmarks).", pdfPath);
			}
		} catch(IOException e ) {
			e.printStackTrace();
			return String.format("IoExcpetion when extracting ToC for PDF %s", pdfPath);
		}

	}

	/**
	 * Recursively prints the outline items (bookmarks).
	 */
	private static String printOutline(PDOutlineNode item, PDDocument document, int level) throws IOException {

		// Iterate through all siblings at the current level
		PDOutlineItem currentItem = item.getFirstChild();
		StringBuilder sb = new StringBuilder();
		while (currentItem != null) {

			// Create indentation for hierarchical view
			String indentation = "  ".repeat(level);

			// Get the title of the bookmark
			String title = currentItem.getTitle();

			// Attempt to find the page number the bookmark links to
			String pageInfo = "";
			try {
				// This resolves the link action to an actual PDPage object
				int pageIndex = document.getPages().indexOf(currentItem.findDestinationPage(document));

				// PDF pages are 1-indexed for users, so add 1
				if (pageIndex >= 0) {
					pageInfo = " (Page " + (pageIndex + 1) + ")";
				}
			} catch (IOException | NullPointerException e) {
				e.printStackTrace();
				return String.format("IoExcpetion when extracting ToC for PDF %s");
			}

			
			sb.append(indentation + "- " + title + pageInfo);
			sb.append("\n");

			// Recursively call for children (sub-sections)
			printOutline(currentItem, document, level + 1);

			// Move to the next sibling
			currentItem = currentItem.getNextSibling();
			return sb.toString();
		}
		return "";
	}

}
