/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.persistence.geojson;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.fennec.models.geojson.GeoJsonObject;
import org.eclipse.fennec.models.geojson.GeoJsonPackage;
import org.eclipse.fennec.models.geojson.util.GeoJsonResourceImpl;
import org.eclipse.fennec.persistence.api.TypeConverter;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * GeoJson type converter
 * @author mark
 * @since 14.01.2025
 */
@Component
public class GeoJsonTypeConverter implements TypeConverter {
	
	@Reference
	private ResourceSet resourceSet;
	@Reference
	private GeoJsonPackage geoJsonPackage;

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.persistence.api.TypeConverter#getName()
	 */
	@Override
	public String getName() {
		return "geojson";
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.persistence.api.TypeConverter#convertValueToEMF(org.eclipse.emf.ecore.EClassifier, java.lang.Object)
	 */
	@Override
	public Object convertValueToEMF(EClassifier eDataType, Object value) {
		if (value instanceof String) {
			Resource resource = new GeoJsonResourceImpl(URI.createFileURI("tmp"));
			try (ByteArrayInputStream bais = new ByteArrayInputStream(value.toString().getBytes(StandardCharsets.UTF_8))) {
				resource.load(bais, null);
				if (resource.getContents().isEmpty()) {
					return null;
				} else {
					EObject result = resource.getContents().get(0);
					resource.getContents().clear();
					return result;
				}
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		return value;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.persistence.api.TypeConverter#convertEMFToValue(org.eclipse.emf.ecore.EClassifier, java.lang.Object)
	 */
	@Override
	public Object convertEMFToValue(EClassifier eDataType, Object emfValue) {
		if (!(emfValue instanceof GeoJsonObject)) {
			return emfValue;
		}
		GeoJsonObject geoJson = (GeoJsonObject) emfValue;
		Resource resource = new GeoJsonResourceImpl(URI.createFileURI("tmp"));
		resource.getContents().add(geoJson);
		try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
			resource.save(baos, null);
			resource.getContents().clear();
			return new String(baos.toByteArray(), StandardCharsets.UTF_8);
		} catch (IOException e) {
			e.printStackTrace();
		} 
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.persistence.api.TypeConverter#isConverterForType(org.eclipse.emf.ecore.EClassifier)
	 */
	@Override
	public boolean isConverterForType(EClassifier eDataType) {
		if (eDataType instanceof EClass eClass) {
			return geoJsonPackage.getGeoJsonObject().isSuperTypeOf(eClass) || geoJsonPackage.getGeoJsonObject().equals(eClass);
		}
		return false;
	}

}
