/*******************************************************************************
 * Copyright (c) 2010 Oracle.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution. 
 * The Eclipse Public License is available at
 *     http://www.eclipse.org/legal/epl-v10.html
 * and the Apache License v2.0 is available at 
 *     http://www.opensource.org/licenses/apache2.0.php.
 * You may elect to redistribute this code under either of these licenses.
 *
 * Contributors:
 *     mkeith - Gemini JPA sample 
 ******************************************************************************/
package org.eclipse.fennec.jpa.demo;

import static java.util.Objects.nonNull;
import static java.util.Objects.requireNonNull;
import static java.util.function.Predicate.not;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.fennec.jpa.demo.classloader.OSGiJPADynamicHelper;
import org.eclipse.persistence.dynamic.DynamicClassLoader;
import org.eclipse.persistence.dynamic.DynamicEntity;
import org.eclipse.persistence.dynamic.DynamicEnumBuilder;
import org.eclipse.persistence.dynamic.DynamicType;
import org.eclipse.persistence.dynamic.DynamicTypeBuilder;
import org.eclipse.persistence.internal.helper.DatabaseField;
import org.eclipse.persistence.jpa.dynamic.JPADynamicHelper;
import org.eclipse.persistence.mappings.DatabaseMapping;
import org.eclipse.persistence.mappings.DirectCollectionMapping;
import org.eclipse.persistence.mappings.DirectToFieldMapping;
import org.eclipse.persistence.mappings.ManyToManyMapping;
import org.eclipse.persistence.mappings.OneToOneMapping;
import org.eclipse.persistence.queries.ReadAllQuery;

import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;

/**
 * Gemini JPA sample client class
 * 
 * @author mkeith
 */
public class BasicModelClient {

	private final EntityManager em;
	private final JPADynamicHelper helper;
	private final DynamicClassLoader dcl;
	private final Map<EClass, Class<?>> dynamicClassMap = new ConcurrentHashMap<>();
	private final Map<EClass, DynamicTypeBuilder> dynamicTypeBuilderMap = new ConcurrentHashMap<>();

	/**
	 * Creates a new instance.
	 */
	public BasicModelClient(EntityManagerFactory emf) {
		em = emf.createEntityManager();
		helper = new OSGiJPADynamicHelper(em);
		dcl = helper.getDynamicClassLoader();
	}

	public void initEPackage(EPackage ePackage) {
		requireNonNull(ePackage);
		List<EClass> eClasses = ePackage.getEClassifiers().
				stream().
				filter(EClass.class::isInstance).
				map(EClass.class::cast).
				toList();
		// Build EClass Type without references
		eClasses.forEach(this::buildEClass);
		// Build EClass References
//		eClasses.forEach(this::buildEClassReferences);

		// Register dynamic types to create / update tables
		DynamicType[] dynamicTypes = dynamicTypeBuilderMap.
				values().
				stream().
				map(DynamicTypeBuilder::getType).toArray(size -> new DynamicType[size]);
		helper.addTypes(true /*create tables*/, true /*create constraints*/, dynamicTypes);
	}

	public void runDynamic(EObject...objects) {
		if (nonNull(objects) && objects.length > 0) {
			List<DynamicEntity> entities = new LinkedList<>();
			for (EObject eo : objects) {
				entities.addAll(create(eo));
			}
			em.getTransaction().begin();
			entities.forEach(em::persist);
			System.out.println("Peristed");
			em.getTransaction().commit();
			
		}
	}
	
	@SuppressWarnings("unchecked")
	public void readDynamic(EClass eClass) {
		requireNonNull(eClass);
		ReadAllQuery query = helper.newReadAllQuery(eClass.getName());
		EAttribute idAttribute = eClass.getEIDAttribute();
		if (nonNull(idAttribute)) {
			query.addAscendingOrdering(idAttribute.getName());
		}
		List<DynamicEntity> result = (List<DynamicEntity>)helper.getSession().executeQuery(query);
		System.out.println(String.format("Result-Size for '%s' is %s", eClass.getName(), result.size()));
		result.forEach(e->{
			eClass.getEStructuralFeatures().stream().filter(not(EStructuralFeature::isTransient)).forEach(f->{
				Object object = e.get(f.getName());
				if (nonNull(object)) {
					System.out.println(String.format("%s - %s: Value = %s", eClass.getName(), f.getName(), object));
				} else {
					System.out.println(String.format("%s - %s: NULL", eClass.getName(), f.getName()));
				}
				
			});
		});
		em.close();
	}

	@SuppressWarnings("unchecked")
	private List<DynamicEntity> create(EObject eObject) {
		requireNonNull(eObject);
		List<DynamicEntity> results = new LinkedList<>();
		Set<EObject> mapped = new HashSet<>();
		EClass eClass = eObject.eClass();
		DynamicTypeBuilder typeBuilder = dynamicTypeBuilderMap.get(eClass);
		DynamicEntity dynEntity = typeBuilder.getType().newDynamicEntity();
		mapped.add(eObject);
		eClass.getEStructuralFeatures().stream().filter(not(EStructuralFeature::isTransient)).forEach(f->{
			Object o = eObject.eGet(f);
			if (f instanceof EReference) {
				if (nonNull(o)) {
					if (f.isMany()) {
						Collection<EObject> manyRefs = (Collection<EObject>) o;
						results.addAll(manyRefs.stream().filter(not(mapped::contains)).map(eo->{
							mapped.add(eo);
							return eo;
						}).map(this::create).flatMap(List::stream).collect(Collectors.toList()));
					} else {
						EObject refO = (EObject) o;
						if (!mapped.contains(refO)) {
							mapped.add(eObject);
							results.addAll(create(refO));
						}
					}
				}
			} else {
				if (nonNull(o)) {
					dynEntity.set(f.getName(), o);
				}
			}
			results.add(dynEntity);
		});
		return results;
	}

	/**
	 * Build the basic EClass with ID fields and basic attributes
	 * @param eclass the {@link EClass}
	 */
	private void buildEClass(EClass eClass) {
		requireNonNull(eClass);
		requireNonNull(dcl);
		final Class<?> dynamicClass = dynamicClassMap.computeIfAbsent(eClass, c->{
			String name = c.getEPackage().getName() + "." + c.getName();
			Class<?> dynClass= dcl.createDynamicClass(name);
			return dynClass;
		});
		final DynamicTypeBuilder typeBuilder = dynamicTypeBuilderMap.computeIfAbsent(eClass, c->new DynamicTypeBuilder(dynamicClass,
				null /*no parent type*/, c.getName().toUpperCase()));
		EAttribute eidAttribute = eClass.getEIDAttribute();
		final String  pkName = nonNull(eidAttribute) ? eidAttribute.getName().toUpperCase() : null;
		if (nonNull(eidAttribute)) {
			typeBuilder.setPrimaryKeyFields(pkName);
			typeBuilder.configureSequencing("SEQ_" + eClass.getName(), eidAttribute.getName());
		}
		List<EAttribute> attributes = eClass.getEAttributes().stream().filter(not(EStructuralFeature::isTransient)).toList();
		// single value attributes
		attributes.stream().filter(not(EStructuralFeature::isMany)).filter(not(this::isEnum)).forEach(ea->{
			Class<?> typeClass = ea.getEAttributeType().getInstanceClass();
			DirectToFieldMapping mapping = typeBuilder.addDirectMapping(ea.getName(), typeClass, ea.getName().toUpperCase());
			mapping.setAttributeAccessor(EFeatureAccessor.create(ea));
		});
		// enum single value attributes
		attributes.stream().filter(not(EStructuralFeature::isMany)).filter(this::isEnum).forEach(ea->{
			Class<?> typeClass = ea.getEAttributeType().getInstanceClass();
			if (isEnum(ea)) {
				if (nonNull(typeClass)) {
					EEnum eEnum = (EEnum) ea.getEAttributeType();
					DynamicEnumBuilder enumBuilder = typeBuilder.addEnum(ea.getName(), typeClass.getName(), ea.getName().toUpperCase(), dcl);
					eEnum.getELiterals().forEach(literal->{
						enumBuilder.addEnumLiteral(literal.getLiteral());
					});
					return;
				} else {
					typeClass = String.class;
					DirectToFieldMapping mapping = typeBuilder.addDirectMapping(ea.getName(), typeClass, ea.getName().toUpperCase());
					mapping.setAttributeAccessor(EFeatureAccessor.create(ea));
				}
			}
			typeBuilder.addDirectMapping(ea.getName(), typeClass, ea.getName().toUpperCase());
		});
		// multi-valued value attributes
//		attributes.stream().filter(EStructuralFeature::isMany).forEach(ea->{
//			// TODO Do we need a enum ???
//			EClass c = ea.getEContainingClass();
//			String tableName = "REF_" + c.getName().toUpperCase() + "_" + ea.getName().toUpperCase();
//			String valueColumn = "VAL_" + ea.getName().toUpperCase();
//			DirectCollectionMapping mapping = typeBuilder.addDirectCollectionMapping(ea.getName(), tableName, valueColumn, Collection.class, pkName);
//			mapping.setAttributeAccessor(EFeatureAccessor.create(ea));
//		});
	}

	private boolean isEnum(EAttribute eAttribute) {
		requireNonNull(eAttribute);
		return eAttribute.getEAttributeType() instanceof EEnum;
	}

	private void buildEClassReferences(EClass eclass) {
		requireNonNull(eclass);
		List<EReference> references = eclass.getEReferences().stream().filter(not(EStructuralFeature::isTransient)).toList();
		references.stream().filter(EStructuralFeature::isMany).forEach(this::buildManyReference);
		references.stream().filter(not(EStructuralFeature::isMany)).forEach(this::buildSingleReference);
	}

	private void buildManyReference(EReference eReference) {
		requireNonNull(eReference);
		boolean manyToMany = false;
		EClass eClass = eReference.getEContainingClass();
		EReference opposite = eReference.getEOpposite();
		if (nonNull(opposite)) {
			manyToMany = true;
		}
		EClass referenceType = eReference.getEReferenceType();
		DynamicTypeBuilder typeBuilder = dynamicTypeBuilderMap.get(eClass);
		DynamicTypeBuilder refTypeBuilder = dynamicTypeBuilderMap.get(referenceType);
		DatabaseMapping mapping;
		if (manyToMany) {
			String mappingTable = eClass.getName() + "_" + eReference.getName() + "__" + referenceType.getName();
			ManyToManyMapping m2Mmapping = new ManyToManyMapping();
			m2Mmapping.setAttributeName(eReference.getName());
			m2Mmapping.setReferenceClass(refTypeBuilder.getType().getJavaClass());
			m2Mmapping.setRelationTableName(mappingTable.toUpperCase());

			for (DatabaseField sourcePK : typeBuilder.getType().getDescriptor().getPrimaryKeyFields()) {
				m2Mmapping.addSourceRelationKeyFieldName(sourcePK.getName(), sourcePK.getQualifiedName());
			}
			DynamicType refType = refTypeBuilder.getType();
			for (DatabaseField targetPK : refType.getDescriptor().getPrimaryKeyFields()) {
				String relField = targetPK.getName();
				if (m2Mmapping.getSourceRelationKeyFieldNames().contains(relField)) {
					relField = refType.getName() + "_" + relField;
				}
				m2Mmapping.addTargetRelationKeyFieldName(relField, targetPK.getQualifiedName());
			}
			m2Mmapping.useTransparentList();
			mapping = typeBuilder.addMapping(m2Mmapping);

		} else {
			mapping = typeBuilder.addOneToManyMapping(eReference.getName(), refTypeBuilder.getType(), eReference.getName().toUpperCase());
		}
		mapping.setAttributeAccessor(EFeatureAccessor.create(eReference));
		typeBuilder.addManyToManyMapping(eReference.getName(), refTypeBuilder.getType(), eReference.getName().toUpperCase());
	}

	private void buildSingleReference(EReference eReference) {
		requireNonNull(eReference);
		EClass eClass = eReference.getEContainingClass();
		EClass referenceType = eReference.getEReferenceType();
		DynamicTypeBuilder typeBuilder = dynamicTypeBuilderMap.get(eClass);
		DynamicTypeBuilder refTypeBuilder = dynamicTypeBuilderMap.get(referenceType);
		OneToOneMapping mapping = typeBuilder.addOneToOneMapping(eReference.getName(), refTypeBuilder.getType(), eReference.getName().toUpperCase());
		mapping.setAttributeAccessor(EFeatureAccessor.create(eReference));
	}

	/**
	 * Returns the Java Type for a given {@link EDataType}
	 * @param dataType the Ecore {@link EDataType}
	 * @return the Java type
	 * @TODO Add plugable type converter here
	 */
	protected Class<?> getType(EDataType dataType) {
		requireNonNull(dataType);
		return dataType.getInstanceClass();
	}
}