/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.persistence.repository;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static java.util.Objects.requireNonNull;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.EcoreUtil.Copier;
import org.eclipse.fennec.persistence.eclipselink.copying.ECopier;
import org.eclipse.persistence.descriptors.ClassDescriptor;
import org.eclipse.persistence.jpa.JpaHelper;
import org.eclipse.persistence.sessions.server.Server;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.repository.DefaultEMFRepository;
import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;

import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;
import jakarta.persistence.TypedQuery;

/**
 * Basic JPA repository implementation 
 * @author Jürgen Albert
 * @since 13 Jan 2025
 */
@Component(name="fennec.jpa.JPARepository", 
service = EMFRepository.class, 
immediate = true,
configurationPolicy = ConfigurationPolicy.REQUIRE)
public class JPARepository extends DefaultEMFRepository {

	@Reference(name="entityManager")
	private EntityManagerFactory emf;

	@Reference
	private ResourceSetFactory rs;

	private EntityManager entityManager;
	private boolean sharedEM = false;
	private Server server;

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#activate(java.util.Map)
	 */
	@Override
	@Activate
	public void activate(Map<String, ?> properties) {
		super.activate(properties);
		String shared = (String) properties.get("sharedEM");
		sharedEM = nonNull(shared) && Boolean.valueOf(shared);
		server = JpaHelper.getServerSession(emf);
		entityManager = emf.createEntityManager(properties);
	}

	@Deactivate
	public void deactivate() {
		if (nonNull(entityManager)) {
			entityManager.close();
		}
		super.deactivate();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getResourceSetFactory()
	 */
	@Override
	public ResourceSetFactory getResourceSetFactory() {
		return rs;
	}
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getResourceSet()
	 */
	@Override
	public ResourceSet getResourceSet() {
		// TODO Auto-generated method stub
		return super.getResourceSet();
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#setIDs(org.eclipse.emf.ecore.EObject)
	 */
	@Override
	protected void setIDs(EObject rootObject) {
		// TODO Auto-generated method stub

	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#save(org.eclipse.emf.ecore.EObject)
	 */
	@Override
	public void save(EObject object) {
		save(object, (Map<?, ?>)null);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#save(org.eclipse.emf.ecore.EObject, java.util.Map)
	 */
	@Override
	public void save(EObject object, Map<?, ?> options) {
		save(Collections.singleton(object));
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#save(org.eclipse.emf.ecore.EObject[])
	 */
	@Override
	public void save(EObject... objects) {
		save(Arrays.asList(objects));
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#save(java.util.Collection)
	 */
	@Override
	public void save(Collection<EObject> objects) {
		// We need to copy the content for dynamic EObjects
		Map<EObject, EObject> copy = doCopy(objects);

		if (!sharedEM &&  
				nonNull(entityManager) && 
				entityManager.isOpen()) {
			System.out.println("close entity manager " + sharedEM);
			entityManager.close();
			entityManager = emf.createEntityManager();
		}

		entityManager.getTransaction().begin();

		// Handle both INSERT and UPDATE properly
		Map<EObject, EObject> mergedEntities = new LinkedHashMap<>();
		for (Map.Entry<EObject, EObject> entry : copy.entrySet()) {
			EObject original = entry.getKey();
			EObject copyEntity = entry.getValue();

			// Check if entity already exists by trying to find it
			String id = EcoreUtil.getID(copyEntity);
			EObject existing = isNull(id) ? null :  entityManager.find(copyEntity.getClass(), id);

			if (isNull(existing)) {
				// New entity - use persist() to generate IDs properly
				entityManager.persist(copyEntity);
				mergedEntities.put(original, copyEntity);
			} else {
				// Existing entity - copy changes to the managed entity
				ECopier copier = new ECopier(existing, null);
				copier.setCopyContainments(false);
				copier.setMergeContainments(true);
				copier.setCopyFunction(this::createEclipselinkEObject);
				copier.copy(copyEntity);
//				entityManager.merge(copyEntity);
				mergedEntities.put(original, existing);
			}
		}

		entityManager.getTransaction().commit();

		// Copy the generated/updated IDs back to the original objects
		mergedEntities.entrySet().stream().filter(e->e.getKey() instanceof DynamicEObjectImpl).forEach(e->{
			String id = EcoreUtil.getID(e.getValue());
			if (nonNull(id)) {
				EcoreUtil.setID(e.getKey(), id);
			}
		});
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getEObject(org.eclipse.emf.ecore.EClass, java.lang.Object, java.util.Map)
	 */
	@Override
	public <T extends EObject> T getEObject(EClass eClass, Object id, Map<?, ?> options) {
		return super.getEObject(eClass, id);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getEObject(org.eclipse.emf.ecore.EClass, java.lang.Object)
	 */
	@Override
	public <T extends EObject> T getEObject(EClass eClass, Object id) {
		return getEObject(eClass.getName(), id);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getEObject(java.lang.String, java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> T getEObject(String eClassName, Object id) {
		ClassDescriptor classDescriptor = server.getDescriptorForAlias(eClassName);
		requireNonNull(classDescriptor);
		// Use the same EntityManager instance to keep entities managed
		Class<?> clazz = classDescriptor.getJavaClass();
		return (T) entityManager.find(clazz, id);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getAllEObjects(org.eclipse.emf.ecore.EClass, java.util.Map)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> List<T> getAllEObjects(EClass eClass, Map<?, ?> options) {
		Long limit = (Long) options.get("limit");
		Long skip = (Long) options.get("skip");
		if (isNull(limit) || limit < 1) {
			limit = 100L;
		}
		if (isNull(skip) || skip < 0) {
			skip = 0l;
		}
		requireNonNull(eClass);
		ClassDescriptor classDescriptor = server.getDescriptorForAlias(eClass.getName());
		requireNonNull(classDescriptor);

		// Use the same EntityManager instance to keep entities managed
		Class<?> clazz = classDescriptor.getJavaClass();
		TypedQuery<?> query = entityManager.createQuery(String.format("SELECT e FROM %s e", eClass.getName()), clazz);
		query.setFirstResult(skip.intValue());
		query.setMaxResults(limit.intValue());
		List<?> result = query.getResultList();
		if (isNull(result) || result.isEmpty()) {
			return Collections.emptyList();
		}
		return (List<T>) result;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.emf.repository.DefaultEMFRepository#getAllEObjects(org.eclipse.emf.ecore.EClass)
	 */
	@Override
	public <T extends EObject> List<T> getAllEObjects(EClass eClass) {
		return getAllEObjects(eClass, Collections.emptyMap());
	}

	private Map<EObject,EObject> doCopy(Collection<EObject> sources) {
		Map<EObject, EObject> mappingCache = new LinkedHashMap<>();
		List<Copier> copiers =new ArrayList<>(sources.size());
		// first copy the attributes
		ECopier copier = null;
		for (EObject source : sources) {
			EObject target = source;
			if (source instanceof DynamicEObjectImpl) {
				target = createEclipselinkEObject(source);
				copier = new ECopier(target, null);
				copier.setCopyContainments(true);
				copier.setCopyFunction(this::createEclipselinkEObject);
				copiers.add(copier);

				copier.copy(source);
			} else {
				// Even for non-DynamicEObjectImpl, we need to convert any contained DynamicEObjectImpl
				copier = new ECopier(target, null);
				copier.setCopyContainments(true);
				copier.setCopyFunction(this::createEclipselinkEObject);
				copiers.add(copier);

				copier.copy(source);
			}  
			mappingCache.put(source, target);
		}
		//		if (copier != null) {
		//			copier.putAll(mappingCache);
		//			copier.copyReferences();
		//		}
		// then copy the references
		for (Copier _copier : copiers) {
			_copier.putAll(mappingCache);
			_copier.copyReferences();
		}
		return mappingCache;
	}

	private EObject doCopyInto(EObject source, EObject target) {
		// first copy the attributes
		ECopier copier = null;
		if (source instanceof DynamicEObjectImpl) {
			target = createEclipselinkEObject(source);
			copier = new ECopier(target, null);
			copier.setCopyContainments(false);

			copier.copy(source);
		}  
		return target;
	}

	private EObject createEclipselinkEObject(EObject source) {
		EClass eClass = source.eClass();
		ClassDescriptor descriptor = server.getDescriptorForAlias(eClass.getName());
		return (EObject) descriptor.getInstantiationPolicy().buildNewInstance();
	}

}
