/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.civitas.handler.mqtt;

import java.util.Arrays;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.civitas.handler.mqtt.helper.MqttReceiverHelper;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.osgi.constants.EMFNamespaces;
import org.gecko.osgi.messaging.Message;
import org.gecko.osgi.messaging.MessagingService;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.metatype.annotations.AttributeDefinition;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;
import org.osgi.service.typedevent.TypedEventBus;
import org.osgi.util.pushstream.PushStream;

/**
 * 
 * @author ilenia
 * @since Sep 26, 2025
 */
@Designate(ocd = MqttReceiver.Config.class)
@Component(name = "MqttReceiver", configurationPid = "MqttReceiverConfig", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class MqttReceiver {
	
	@Reference
	private MessagingService messaging;
	
	@Reference
	TypedEventBus typedEventBus;
	
	private static final Logger LOGGER = Logger.getLogger(MqttReceiver.class.getName());
	
	private Config config;
	@Reference(target = "("+EMFNamespaces.EMF_MODEL_FILE_EXT +"=json)") 
	private ComponentServiceObjects<ResourceSet> rsFactory;
	@Reference(name = "mqtt.service", target = "(id=full)") 
	private MessagingService messagingService;

	private PushStream<Message> subscription;
	
	@ObjectClassDefinition(name = "MqttReceiver Configuration")
	@interface Config {

		@AttributeDefinition(name = "MQTT Topic", description = "The MQTT topic where to listen to")
		String mqtt_topic();
		
		@AttributeDefinition(name = "Payload EClass URI", description = "The EClass URI of the object we expect in the MQTT message payload")
		String payload_eclassuri();
		
		@AttributeDefinition(name = "Forward Topic", description = "The topic where to publish the received EObject")
		String[] forward_topics();
		
		@AttributeDefinition(name = "Print Payload", description = "Whether or not to print the payload content")
		boolean print_payload() default false;
	}
	
	@Activate
	public void activate (Config config) {
		this.config = config;
		try {
			subscription = messagingService.subscribe(config.mqtt_topic());
			subscription.forEach(this::handleMessage);			
		} catch (Exception e) {
			LOGGER.log(Level.SEVERE, String.format("Exception while subscribing to topic %s", config.mqtt_topic()), e);
		}
	}
	
	@Deactivate
	public void deactivate() {
		subscription.close();
	}
	
	
	private void handleMessage(Message msg) {
		if(msg == null) {
			LOGGER.warning(String.format("Message is null! Cannot extract anything!"));
			return;
		}
		LOGGER.info(String.format("Got Msg for topic %s", msg.topic()));
		EObject content = extractPayload(msg);
		if(config.print_payload()) printPayload(content);
		Arrays.asList(config.forward_topics()).forEach(t -> typedEventBus.deliver(t, EcoreUtil.copy(content)));
	}

	private EObject extractPayload(Message message) {		
		byte[] content = message.payload().array();
		LOGGER.log(Level.SEVERE, String.format("Extract payload %s", new String(content)));
		
		try {
			return MqttReceiverHelper
					.loadResource(rsFactory, URI.createFileURI(UUID.randomUUID().toString().concat(".json")), 
							"application/json", content, 
							config.payload_eclassuri());
		} catch(Exception e) {
			LOGGER.log(Level.SEVERE, String.format("IOException while reading payload from MQTT"), e);
			return null;
		}
	}
	
	private void printPayload(EObject content) {		
		try {
			MqttReceiverHelper
					.printResource(rsFactory, URI.createFileURI(UUID.randomUUID().toString().concat(".json")), 
							"application/json", content);
		} catch(Exception e) {
			LOGGER.log(Level.SEVERE, String.format("IOException while printing payload from MQTT"), e);
		}
	}

}
