package org.gecko.emf.ngsi;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EContentAdapter;

/**
 * A change tracker for NGSI entities that monitors property changes using EContentAdapter.
 * This class holds the current EObject instance and tracks changes when new instances arrive.
 * 
 * @author Generated with Claude Code
 */
public class NGSIEntityChangeTracker extends EContentAdapter {
    
    /**
     * Listener interface for NGSI entity changes
     */
    public interface NGSIChangeListener {
        /**
         * Called when a property value changes
         * @param entity the NGSI entity that changed
         * @param propertyName the name of the property that changed
         * @param oldValue the old value
         * @param newValue the new value
         */
        void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue);
        
        /**
         * Called when a property is added
         * @param entity the NGSI entity that changed
         * @param propertyName the name of the property that was added
         * @param newValue the new value
         */
        void onPropertyAdded(EObject entity, String propertyName, Object newValue);
        
        /**
         * Called when a property is removed
         * @param entity the NGSI entity that changed
         * @param propertyName the name of the property that was removed
         * @param oldValue the old value
         */
        void onPropertyRemoved(EObject entity, String propertyName, Object oldValue);
    }
    
    private EObject currentEntity;
    private final List<NGSIChangeListener> listeners = new CopyOnWriteArrayList<>();
    
    /**
     * Creates a new NGSI entity change tracker
     */
    public NGSIEntityChangeTracker() {
        super();
    }
    
    /**
     * Sets the current NGSI entity instance to track
     * @param entity the entity to track
     */
    public void setCurrentEntity(EObject entity) {
        // Remove adapter from previous entity
        if (currentEntity != null) {
            currentEntity.eAdapters().remove(this);
        }
        
        this.currentEntity = entity;
        
        // Add adapter to new entity
        if (entity != null) {
            entity.eAdapters().add(this);
        }
    }
    
    /**
     * Gets the current NGSI entity instance
     * @return the current entity
     */
    public EObject getCurrentEntity() {
        return currentEntity;
    }
    
    /**
     * Updates the current entity with a new instance and detects changes
     * @param newEntity the new entity instance
     */
    public void updateEntity(EObject newEntity) {
        if (currentEntity != null && newEntity != null) {
            detectAndNotifyChanges(currentEntity, newEntity);
        }
        setCurrentEntity(newEntity);
    }
    
    /**
     * Adds a change listener
     * @param listener the listener to add
     */
    public void addChangeListener(NGSIChangeListener listener) {
        if (listener != null) {
            listeners.add(listener);
        }
    }
    
    /**
     * Removes a change listener
     * @param listener the listener to remove
     */
    public void removeChangeListener(NGSIChangeListener listener) {
        listeners.remove(listener);
    }
    
    @Override
    public void notifyChanged(Notification notification) {
        super.notifyChanged(notification);
        
        if (notification.getNotifier() == currentEntity) {
            handleNotification(notification);
        }
    }
    
    /**
     * Handles EMF notifications for property changes
     * @param notification the EMF notification
     */
    private void handleNotification(Notification notification) {
        Object feature = notification.getFeature();
        String propertyName = getPropertyName(feature);
        
        if (propertyName == null) {
            return;
        }
        
        Object oldValue = notification.getOldValue();
        Object newValue = notification.getNewValue();
        
        switch (notification.getEventType()) {
            case Notification.SET:
                notifyPropertyChanged(currentEntity, propertyName, oldValue, newValue);
                break;
            case Notification.ADD:
                notifyPropertyAdded(currentEntity, propertyName, newValue);
                break;
            case Notification.REMOVE:
                notifyPropertyRemoved(currentEntity, propertyName, oldValue);
                break;
            case Notification.UNSET:
                notifyPropertyRemoved(currentEntity, propertyName, oldValue);
                break;
        }
    }
    
    /**
     * Detects changes between old and new entity instances
     * @param oldEntity the old entity
     * @param newEntity the new entity
     */
    private void detectAndNotifyChanges(EObject oldEntity, EObject newEntity) {
        if (oldEntity.eClass() != newEntity.eClass()) {
            return; // Different types, can't compare
        }
        
        // Check all attributes
        for (EAttribute attribute : oldEntity.eClass().getEAllAttributes()) {
            Object oldValue = oldEntity.eGet(attribute);
            Object newValue = newEntity.eGet(attribute);
            
            if (!Objects.equals(oldValue, newValue)) {
                notifyPropertyChanged(oldEntity, attribute.getName(), oldValue, newValue);
            }
        }
        
        // Check all references
        for (EReference reference : oldEntity.eClass().getEAllReferences()) {
            Object oldValue = oldEntity.eGet(reference);
            Object newValue = newEntity.eGet(reference);
            
            if (!Objects.equals(oldValue, newValue)) {
                notifyPropertyChanged(oldEntity, reference.getName(), oldValue, newValue);
            }
        }
    }
    
    /**
     * Gets the property name from an EMF feature
     * @param feature the EMF feature
     * @return the property name or null if not applicable
     */
    private String getPropertyName(Object feature) {
        if (feature instanceof EAttribute) {
            return ((EAttribute) feature).getName();
        } else if (feature instanceof EReference) {
            return ((EReference) feature).getName();
        }
        return null;
    }
    
    /**
     * Notifies all listeners of a property change
     */
    private void notifyPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
        for (NGSIChangeListener listener : listeners) {
            try {
                listener.onPropertyChanged(entity, propertyName, oldValue, newValue);
            } catch (Exception e) {
                // Log error but continue with other listeners
                System.err.println("Error notifying listener: " + e.getMessage());
            }
        }
    }
    
    /**
     * Notifies all listeners of a property addition
     */
    private void notifyPropertyAdded(EObject entity, String propertyName, Object newValue) {
        for (NGSIChangeListener listener : listeners) {
            try {
                listener.onPropertyAdded(entity, propertyName, newValue);
            } catch (Exception e) {
                System.err.println("Error notifying listener: " + e.getMessage());
            }
        }
    }
    
    /**
     * Notifies all listeners of a property removal
     */
    private void notifyPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
        for (NGSIChangeListener listener : listeners) {
            try {
                listener.onPropertyRemoved(entity, propertyName, oldValue);
            } catch (Exception e) {
                System.err.println("Error notifying listener: " + e.getMessage());
            }
        }
    }
    
    /**
     * Clears all listeners and removes the adapter from the current entity
     */
    public void dispose() {
        listeners.clear();
        if (currentEntity != null) {
            currentEntity.eAdapters().remove(this);
            currentEntity = null;
        }
    }
}