
package org.gecko.emf.ngsi;

import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.ngsi.NGSIEntity;
import org.gecko.emf.ngsi.NGSIFactory;

/**
 * Manager class for NGSI entities that maintains current instances and tracks changes.
 * 
 * <p>This class provides the mechanism to hold current EObject instances and detect changes
 * when new instances arrive. It now includes persistent storage capabilities through
 * {@link NGSIStorageHandler} integration.</p>
 * 
 * <p>Key features:</p>
 * <ul>
 *   <li>Entity registration and change tracking</li>
 *   <li>Persistent storage with versioning</li>
 *   <li>History tracking and retrieval</li>
 *   <li>Subscription manager integration</li>
 *   <li>Automatic state persistence</li>
 * </ul>
 * 
 * @author Generated with Claude Code
 */
public class NGSIEntityManager {
    
    /**
     * Map of entity IDs to their current instances and change trackers
     */
    private final Map<String, EntityHolder> entities = new ConcurrentHashMap<>();
    
    /**
     * List of subscription managers that need to be notified of entity lifecycle events
     */
    private final List<NGSISubscriptionManager> subscriptionManagers = new CopyOnWriteArrayList<>();
    
    /**
     * Holds an entity instance with its change tracker
     */
    private static class EntityHolder {
        private final NGSIEntityChangeTracker changeTracker;
        private EObject currentEntity;
        
        public EntityHolder(EObject entity) {
            this.changeTracker = new NGSIEntityChangeTracker();
            this.currentEntity = entity;
            this.changeTracker.setCurrentEntity(entity);
        }
        
        public NGSIEntityChangeTracker getChangeTracker() {
            return changeTracker;
        }
        
        public EObject getCurrentEntity() {
            return currentEntity;
        }
        
        public void updateEntity(EObject newEntity) {
            this.currentEntity = newEntity;
            this.changeTracker.updateEntity(newEntity);
        }
        
        public void dispose() {
            this.changeTracker.dispose();
        }
    }
    
    /**
     * Registers a new NGSI entity or updates an existing one
     * @param entityId the unique identifier for the entity
     * @param entity the entity instance
     */
    public void registerEntity(String entityId, EObject entity) {
        if (entityId == null || entity == null) {
            throw new IllegalArgumentException("Entity ID and entity cannot be null");
        }
        
        EntityHolder holder = entities.get(entityId);
        boolean isNewEntity = holder == null;
        
        if (isNewEntity) {
            // New entity
            holder = new EntityHolder(entity);
            entities.put(entityId, holder);
            
            // Notify subscription managers of new entity
            for (NGSISubscriptionManager subscriptionManager : subscriptionManagers) {
                subscriptionManager.onEntityRegistered(entityId);
            }
        } else {
            // Update existing entity
            holder.updateEntity(entity);
        }
        
    }
    
    /**
     * Gets the current instance of an entity
     * @param entityId the entity ID
     * @return the current entity instance or null if not found
     */
    public EObject getCurrentEntity(String entityId) {
        EntityHolder holder = entities.get(entityId);
        return holder != null ? holder.getCurrentEntity() : null;
    }
    
    /**
     * Adds a change listener to an entity
     * @param entityId the entity ID
     * @param listener the change listener
     */
    public void addChangeListener(String entityId, NGSIEntityChangeTracker.NGSIChangeListener listener) {
        EntityHolder holder = entities.get(entityId);
        if (holder != null) {
            holder.getChangeTracker().addChangeListener(listener);
        }
    }
    
    /**
     * Removes a change listener from an entity
     * @param entityId the entity ID
     * @param listener the change listener to remove
     */
    public void removeChangeListener(String entityId, NGSIEntityChangeTracker.NGSIChangeListener listener) {
        EntityHolder holder = entities.get(entityId);
        if (holder != null) {
            holder.getChangeTracker().removeChangeListener(listener);
        }
    }
    
    /**
     * Gets the change tracker for an entity
     * @param entityId the entity ID
     * @return the change tracker or null if entity not found
     */
    public NGSIEntityChangeTracker getChangeTracker(String entityId) {
        EntityHolder holder = entities.get(entityId);
        return holder != null ? holder.getChangeTracker() : null;
    }
    
    /**
     * Removes an entity from management
     * @param entityId the entity ID
     */
    public void unregisterEntity(String entityId) {
        EntityHolder holder = entities.remove(entityId);
        if (holder != null) {
            // Notify subscription managers of entity removal
            for (NGSISubscriptionManager subscriptionManager : subscriptionManagers) {
                subscriptionManager.onEntityUnregistered(entityId);
            }
            
            holder.dispose();
        }
    }
    
    /**
     * Checks if an entity is currently being managed
     * @param entityId the entity ID
     * @return true if the entity is managed, false otherwise
     */
    public boolean isEntityManaged(String entityId) {
        return entities.containsKey(entityId);
    }
    
    /**
     * Gets the number of entities currently being managed
     * @return the number of managed entities
     */
    public int getManagedEntityCount() {
        return entities.size();
    }
    
    /**
     * Registers a subscription manager to receive entity lifecycle notifications.
     * 
     * @param subscriptionManager the subscription manager to register
     * @throws IllegalArgumentException if subscriptionManager is null
     */
    public void registerSubscriptionManager(NGSISubscriptionManager subscriptionManager) {
        if (subscriptionManager == null) {
            throw new IllegalArgumentException("Subscription manager cannot be null");
        }
        
        subscriptionManagers.add(subscriptionManager);
        
        // Notify the subscription manager of all existing entities
        for (String entityId : entities.keySet()) {
            subscriptionManager.onEntityRegistered(entityId);
        }
    }
    
    /**
     * Unregisters a subscription manager from receiving entity lifecycle notifications.
     * 
     * @param subscriptionManager the subscription manager to unregister
     * @return true if the subscription manager was found and removed, false otherwise
     */
    public boolean unregisterSubscriptionManager(NGSISubscriptionManager subscriptionManager) {
        boolean removed = subscriptionManagers.remove(subscriptionManager);
        
        if (removed) {
            // Notify the subscription manager of all entity removals
            for (String entityId : entities.keySet()) {
                subscriptionManager.onEntityUnregistered(entityId);
            }
        }
        
        return removed;
    }
    
    /**
     * Gets all managed entity IDs.
     * 
     * @return set of managed entity IDs
     */
    public java.util.Set<String> getManagedEntityIds() {
        return java.util.Collections.unmodifiableSet(entities.keySet());
    }
    
    /**
     * Disposes all managed entities and clears the manager
     */
    public void dispose() {
        // Notify all subscription managers that all entities are being removed
        for (String entityId : entities.keySet()) {
            for (NGSISubscriptionManager subscriptionManager : subscriptionManagers) {
                subscriptionManager.onEntityUnregistered(entityId);
            }
        }
        
        for (EntityHolder holder : entities.values()) {
            holder.dispose();
        }
        entities.clear();
        subscriptionManagers.clear();
    }
    
    /**
     * Creates an NGSIEntity from an entity ID and EObject.
     * 
     * @param entityId the entity ID
     * @param entity the EObject to wrap
     * @return the created NGSI entity
     */
    private NGSIEntity createNGSIEntity(String entityId, EObject entity) {
        NGSIEntity ngsiEntity = NGSIFactory.eINSTANCE.createNGSIEntity();
        
        ngsiEntity.setId(entityId);
        ngsiEntity.setWrappedObject(entity);
        
        // Set entity type based on EClass name
        if (entity.eClass() != null) {
            ngsiEntity.setType(entity.eClass().getName());
        }
        
        // Set timestamps
        java.util.Date now = new java.util.Date();
        if (ngsiEntity.getCreatedAt() == null) {
            ngsiEntity.setCreatedAt(now);
        }
        ngsiEntity.setModifiedAt(now);
        
        return ngsiEntity;
    }
}