/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.emf.ngsi;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import org.eclipse.emf.ecore.EObject;

/**
 * Manager for NGSI-LD subscriptions that handles subscription lifecycle and notification delivery.
 * 
 * <p>This class provides comprehensive subscription management capabilities including:</p>
 * <ul>
 *   <li>Subscription registration and lifecycle management</li>
 *   <li>Entity type and attribute filtering</li>
 *   <li>Pattern-based entity ID matching</li>
 *   <li>Asynchronous notification delivery</li>
 *   <li>Integration with {@link NGSIEntityManager} for change tracking</li>
 * </ul>
 * 
 * <p>The subscription manager acts as a bridge between the NGSI entity change tracking
 * system and external notification endpoints, ensuring that subscribers receive
 * relevant change notifications based on their subscription criteria.</p>
 * 
 * <h3>Usage Example:</h3>
 * <pre>
 * NGSISubscriptionManager subscriptionManager = new NGSISubscriptionManager();
 * NGSIEntityManager entityManager = new NGSIEntityManager();
 * 
 * // Connect the managers
 * subscriptionManager.attachToEntityManager(entityManager);
 * 
 * // Create a subscription for temperature sensor entities
 * NGSISubscription subscription = subscriptionManager.createSubscription(
 *     "urn:ngsi-ld:Subscription:001",
 *     "urn:ngsi-ld:TemperatureSensor:.*",
 *     Arrays.asList("temperature", "humidity"),
 *     notificationHandler
 * );
 * </pre>
 * 
 * @author Generated with Claude Code
 * @since 1.0.0
 * 
 * @see NGSIEntityManager
 * @see NGSIEntityChangeTracker
 * @see NGSISubscription
 */
public class NGSISubscriptionManager {

    /**
     * Interface for handling subscription notifications.
     * 
     * <p>Implementations of this interface receive notifications when entities
     * matching subscription criteria change. The notification includes details
     * about the subscription, the changed entity, and the specific changes.</p>
     */
    public interface NGSINotificationHandler {
        /**
         * Called when an entity change matches a subscription.
         * 
         * @param subscription the subscription that matched
         * @param entityId the ID of the entity that changed
         * @param entity the changed entity
         * @param changedProperty the name of the property that changed
         * @param oldValue the previous value
         * @param newValue the new value
         */
        void onNotification(NGSISubscription subscription, String entityId, EObject entity, 
                          String changedProperty, Object oldValue, Object newValue);
    }

    /**
     * Internal representation of a subscription with its metadata and filtering criteria.
     */
    private static class SubscriptionEntry {
        private final NGSISubscription subscription;
        private final String subscriptionId;
        private final Pattern entityIdPattern;
        private final Set<String> monitoredAttributes;
        private final NGSINotificationHandler notificationHandler;
        private final long createdAt;
        private boolean active;

        /**
         * Creates a new subscription entry.
         * 
         * @param subscription the NGSI subscription model object
         * @param subscriptionId unique identifier for the subscription
         * @param entityIdPattern regex pattern for matching entity IDs
         * @param monitoredAttributes set of attribute names to monitor (null means all)
         * @param notificationHandler handler for notifications
         */
        public SubscriptionEntry(NGSISubscription subscription, String subscriptionId, 
                               Pattern entityIdPattern, Set<String> monitoredAttributes,
                               NGSINotificationHandler notificationHandler) {
            this.subscription = subscription;
            this.subscriptionId = subscriptionId;
            this.entityIdPattern = entityIdPattern;
            this.monitoredAttributes = monitoredAttributes != null ? 
                new CopyOnWriteArraySet<>(monitoredAttributes) : null;
            this.notificationHandler = notificationHandler;
            this.createdAt = System.currentTimeMillis();
            this.active = true;
        }

        public NGSISubscription getSubscription() { return subscription; }
        public String getSubscriptionId() { return subscriptionId; }
        public Pattern getEntityIdPattern() { return entityIdPattern; }
        public Set<String> getMonitoredAttributes() { return monitoredAttributes; }
        public NGSINotificationHandler getNotificationHandler() { return notificationHandler; }
        public long getCreatedAt() { return createdAt; }
        public boolean isActive() { return active; }
        public void setActive(boolean active) { this.active = active; }

        /**
         * Checks if this subscription matches the given entity ID and attribute.
         * 
         * @param entityId the entity ID to check
         * @param attribute the attribute name to check
         * @return true if the subscription matches
         */
        public boolean matches(String entityId, String attribute) {
            if (!active) {
                return false;
            }
            
            // Check entity ID pattern
            if (entityIdPattern != null && !entityIdPattern.matcher(entityId).matches()) {
                return false;
            }
            
            // Check monitored attributes (null means monitor all attributes)
            if (monitoredAttributes != null && !monitoredAttributes.contains(attribute)) {
                return false;
            }
            
            return true;
        }
    }

    /**
     * Change listener that bridges entity changes to subscription notifications.
     */
    private class SubscriptionChangeListener implements NGSIEntityChangeTracker.NGSIChangeListener {
        private final String entityId;

        public SubscriptionChangeListener(String entityId) {
            this.entityId = entityId;
        }

        @Override
        public void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
            processEntityChange(entityId, entity, propertyName, oldValue, newValue);
        }

        @Override
        public void onPropertyAdded(EObject entity, String propertyName, Object newValue) {
            processEntityChange(entityId, entity, propertyName, null, newValue);
        }

        @Override
        public void onPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
            processEntityChange(entityId, entity, propertyName, oldValue, null);
        }
    }

    // Core data structures
    private final Map<String, SubscriptionEntry> subscriptions = new ConcurrentHashMap<>();
    private final Map<String, List<SubscriptionChangeListener>> entityListeners = new ConcurrentHashMap<>();
    private final ExecutorService notificationExecutor = Executors.newCachedThreadPool(r -> {
        Thread t = new Thread(r, "NGSI-Subscription-Notifier");
        t.setDaemon(true);
        return t;
    });

    private NGSIEntityManager entityManager;
    private volatile boolean disposed = false;

    /**
     * Creates a new NGSI subscription manager.
     */
    public NGSISubscriptionManager() {
        // Initialize with default configuration
    }

    /**
     * Attaches this subscription manager to an entity manager for change tracking.
     * 
     * @param entityManager the entity manager to attach to
     * @throws IllegalArgumentException if entityManager is null
     */
    public void attachToEntityManager(NGSIEntityManager entityManager) {
        if (entityManager == null) {
            throw new IllegalArgumentException("Entity manager cannot be null");
        }
        this.entityManager = entityManager;
    }

    /**
     * Creates a new subscription for monitoring entity changes.
     * 
     * @param subscriptionId unique identifier for the subscription
     * @param entityIdPattern regex pattern for matching entity IDs (null matches all)
     * @param monitoredAttributes list of attribute names to monitor (null monitors all)
     * @param notificationHandler handler for notifications
     * @return the created subscription
     * @throws IllegalArgumentException if required parameters are null or invalid
     */
    public NGSISubscription createSubscription(String subscriptionId, String entityIdPattern, 
                                             List<String> monitoredAttributes,
                                             NGSINotificationHandler notificationHandler) {
        if (subscriptionId == null || subscriptionId.trim().isEmpty()) {
            throw new IllegalArgumentException("Subscription ID cannot be null or empty");
        }
        if (notificationHandler == null) {
            throw new IllegalArgumentException("Notification handler cannot be null");
        }
        if (subscriptions.containsKey(subscriptionId)) {
            throw new IllegalArgumentException("Subscription with ID " + subscriptionId + " already exists");
        }

        // Create NGSI subscription model object
        NGSISubscription subscription = NGSIFactory.eINSTANCE.createNGSISubscription();
        
        // Compile entity ID pattern
        Pattern pattern = null;
        if (entityIdPattern != null && !entityIdPattern.trim().isEmpty()) {
            try {
                pattern = Pattern.compile(entityIdPattern);
            } catch (Exception e) {
                throw new IllegalArgumentException("Invalid entity ID pattern: " + entityIdPattern, e);
            }
        }

        // Create subscription entry
        Set<String> attributeSet = monitoredAttributes != null ? 
            new CopyOnWriteArraySet<>(monitoredAttributes) : null;
        SubscriptionEntry entry = new SubscriptionEntry(subscription, subscriptionId, 
                                                       pattern, attributeSet, notificationHandler);
        
        subscriptions.put(subscriptionId, entry);
        
        // If entity manager is available, set up listeners for existing entities
        if (entityManager != null) {
            setupListenersForExistingEntities(entry);
        }

        return subscription;
    }

    /**
     * Activates or deactivates a subscription.
     * 
     * @param subscriptionId the subscription ID
     * @param active true to activate, false to deactivate
     * @return true if the subscription was found and updated, false otherwise
     */
    public boolean setSubscriptionActive(String subscriptionId, boolean active) {
        SubscriptionEntry entry = subscriptions.get(subscriptionId);
        if (entry != null) {
            entry.setActive(active);
            return true;
        }
        return false;
    }

    /**
     * Removes a subscription.
     * 
     * @param subscriptionId the subscription ID to remove
     * @return true if the subscription was found and removed, false otherwise
     */
    public boolean removeSubscription(String subscriptionId) {
        SubscriptionEntry entry = subscriptions.remove(subscriptionId);
        if (entry != null) {
            // Clean up listeners for this subscription
            cleanupListenersForSubscription(entry);
            return true;
        }
        return false;
    }

    /**
     * Gets information about a subscription.
     * 
     * @param subscriptionId the subscription ID
     * @return the subscription or null if not found
     */
    public NGSISubscription getSubscription(String subscriptionId) {
        SubscriptionEntry entry = subscriptions.get(subscriptionId);
        return entry != null ? entry.getSubscription() : null;
    }

    /**
     * Gets all active subscription IDs.
     * 
     * @return set of active subscription IDs
     */
    public Set<String> getActiveSubscriptionIds() {
        return subscriptions.entrySet().stream()
            .filter(entry -> entry.getValue().isActive())
            .map(Map.Entry::getKey)
            .collect(java.util.stream.Collectors.toSet());
    }

    /**
     * Called when a new entity is registered in the entity manager.
     * This method sets up change listeners for the new entity.
     * 
     * @param entityId the entity ID
     */
    public void onEntityRegistered(String entityId) {
        if (entityManager == null || disposed) {
            return;
        }

        List<SubscriptionChangeListener> listeners = new CopyOnWriteArrayList<>();
        
        // Create a listener for this entity
        SubscriptionChangeListener listener = new SubscriptionChangeListener(entityId);
        listeners.add(listener);
        
        // Add listener to entity manager
        entityManager.addChangeListener(entityId, listener);
        
        // Store listener reference
        entityListeners.put(entityId, listeners);
    }

    /**
     * Called when an entity is unregistered from the entity manager.
     * This method cleans up change listeners for the entity.
     * 
     * @param entityId the entity ID
     */
    public void onEntityUnregistered(String entityId) {
        List<SubscriptionChangeListener> listeners = entityListeners.remove(entityId);
        if (listeners != null && entityManager != null) {
            for (SubscriptionChangeListener listener : listeners) {
                entityManager.removeChangeListener(entityId, listener);
            }
        }
    }

    /**
     * Processes entity changes and triggers relevant subscriptions.
     * 
     * @param entityId the entity ID
     * @param entity the changed entity
     * @param propertyName the changed property name
     * @param oldValue the old value
     * @param newValue the new value
     */
    private void processEntityChange(String entityId, EObject entity, String propertyName, 
                                   Object oldValue, Object newValue) {
        if (disposed) {
            return;
        }

        // Find matching subscriptions
        List<SubscriptionEntry> matchingSubscriptions = new CopyOnWriteArrayList<>();
        for (SubscriptionEntry entry : subscriptions.values()) {
            if (entry.matches(entityId, propertyName)) {
                matchingSubscriptions.add(entry);
            }
        }

        // Send notifications asynchronously
        if (!matchingSubscriptions.isEmpty()) {
            notificationExecutor.submit(() -> {
                for (SubscriptionEntry entry : matchingSubscriptions) {
                    try {
                        entry.getNotificationHandler().onNotification(
                            entry.getSubscription(), entityId, entity, 
                            propertyName, oldValue, newValue);
                    } catch (Exception e) {
                        System.err.println("Error in subscription notification handler: " + e.getMessage());
                        // Continue with other subscriptions
                    }
                }
            });
        }
    }

    /**
     * Sets up listeners for existing entities when a new subscription is created.
     * 
     * @param entry the subscription entry
     */
    private void setupListenersForExistingEntities(SubscriptionEntry entry) {
        if (entityManager == null) {
            return;
        }

        // Note: This would require an enhancement to NGSIEntityManager to provide
        // a way to iterate over existing entities. For now, we rely on
        // onEntityRegistered being called for new entities.
    }

    /**
     * Cleans up listeners for a removed subscription.
     * 
     * @param entry the subscription entry
     */
    private void cleanupListenersForSubscription(SubscriptionEntry entry) {
        // Since we use a shared listener per entity, we don't need to remove
        // listeners here. The filtering happens in processEntityChange.
    }

    /**
     * Disposes the subscription manager and cleans up resources.
     */
    public void dispose() {
        disposed = true;
        
        // Clean up all listeners
        if (entityManager != null) {
            for (Map.Entry<String, List<SubscriptionChangeListener>> entry : entityListeners.entrySet()) {
                String entityId = entry.getKey();
                List<SubscriptionChangeListener> listeners = entry.getValue();
                for (SubscriptionChangeListener listener : listeners) {
                    entityManager.removeChangeListener(entityId, listener);
                }
            }
        }
        
        // Clear data structures
        subscriptions.clear();
        entityListeners.clear();
        
        // Shutdown notification executor
        notificationExecutor.shutdown();
    }

    /**
     * Gets the number of active subscriptions.
     * 
     * @return the number of active subscriptions
     */
    public int getActiveSubscriptionCount() {
        return (int) subscriptions.values().stream()
            .filter(SubscriptionEntry::isActive)
            .count();
    }

    /**
     * Gets the total number of subscriptions (active and inactive).
     * 
     * @return the total number of subscriptions
     */
    public int getTotalSubscriptionCount() {
        return subscriptions.size();
    }
}