package org.gecko.emf.ngsi.example;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.ngsi.NGSIEntityChangeTracker;
import org.gecko.emf.ngsi.NGSIEntityManager;

/**
 * Example demonstrating how to use the NGSI entity change tracking mechanism.
 * This shows how to hold current EObject instances and track property changes
 * when new instances arrive.
 * 
 * @author Generated with Claude Code
 */
public class NGSIChangeTrackingExample {
    
    /**
     * Example NGSI change listener that logs changes
     */
    private static class LoggingChangeListener implements NGSIEntityChangeTracker.NGSIChangeListener {
        
        @Override
        public void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
            System.out.println("Property changed on entity " + entity.eClass().getName() + 
                             ": " + propertyName + " = " + oldValue + " -> " + newValue);
        }
        
        @Override
        public void onPropertyAdded(EObject entity, String propertyName, Object newValue) {
            System.out.println("Property added to entity " + entity.eClass().getName() + 
                             ": " + propertyName + " = " + newValue);
        }
        
        @Override
        public void onPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
            System.out.println("Property removed from entity " + entity.eClass().getName() + 
                             ": " + propertyName + " (was " + oldValue + ")");
        }
    }
    
    /**
     * Example NGSI change listener that could trigger NGSI notifications
     */
    private static class NGSINotificationListener implements NGSIEntityChangeTracker.NGSIChangeListener {
        
        @Override
        public void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
            // In a real implementation, this would trigger NGSI change notifications
            sendNGSIChangeNotification(entity, propertyName, oldValue, newValue);
        }
        
        @Override
        public void onPropertyAdded(EObject entity, String propertyName, Object newValue) {
            // In a real implementation, this would trigger NGSI add notifications
            sendNGSIAddNotification(entity, propertyName, newValue);
        }
        
        @Override
        public void onPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
            // In a real implementation, this would trigger NGSI remove notifications
            sendNGSIRemoveNotification(entity, propertyName, oldValue);
        }
        
        private void sendNGSIChangeNotification(EObject entity, String propertyName, Object oldValue, Object newValue) {
            // TODO: Implement NGSI change notification
            System.out.println("NGSI: Would send change notification for " + propertyName);
        }
        
        private void sendNGSIAddNotification(EObject entity, String propertyName, Object newValue) {
            // TODO: Implement NGSI add notification
            System.out.println("NGSI: Would send add notification for " + propertyName);
        }
        
        private void sendNGSIRemoveNotification(EObject entity, String propertyName, Object oldValue) {
            // TODO: Implement NGSI remove notification
            System.out.println("NGSI: Would send remove notification for " + propertyName);
        }
    }
    
    /**
     * Demonstrates basic usage of the change tracking mechanism
     */
    public static void demonstrateBasicUsage() {
        // Create the entity manager
        NGSIEntityManager entityManager = new NGSIEntityManager();
        
        // Create change listeners
        LoggingChangeListener loggingListener = new LoggingChangeListener();
        NGSINotificationListener ngsiListener = new NGSINotificationListener();
        
        // Example entity ID
        String entityId = "urn:ngsi-ld:Entity:example001";
        
        // TODO: Replace with actual NGSIEntity creation
        // EObject initialEntity = createExampleNGSIEntity();
        // entityManager.registerEntity(entityId, initialEntity);
        
        // Add listeners
        entityManager.addChangeListener(entityId, loggingListener);
        entityManager.addChangeListener(entityId, ngsiListener);
        
        // Simulate receiving a new entity instance
        // TODO: Replace with actual updated entity
        // EObject updatedEntity = createUpdatedNGSIEntity();
        // entityManager.registerEntity(entityId, updatedEntity);
        
        // The change listeners will automatically be notified of differences
        
        // Clean up
        entityManager.dispose();
    }
    
    /**
     * Demonstrates advanced usage with multiple entities
     */
    public static void demonstrateAdvancedUsage() {
        NGSIEntityManager entityManager = new NGSIEntityManager();
        
        // Create different listeners for different entity types
        NGSIEntityChangeTracker.NGSIChangeListener temperatureListener = new NGSIEntityChangeTracker.NGSIChangeListener() {
            @Override
            public void onPropertyChanged(EObject entity, String propertyName, Object oldValue, Object newValue) {
                if ("temperature".equals(propertyName)) {
                    System.out.println("Temperature changed from " + oldValue + " to " + newValue);
                    // Could trigger temperature-specific logic here
                }
            }
            
            @Override
            public void onPropertyAdded(EObject entity, String propertyName, Object newValue) {
                if ("temperature".equals(propertyName)) {
                    System.out.println("Temperature sensor activated: " + newValue);
                }
            }
            
            @Override
            public void onPropertyRemoved(EObject entity, String propertyName, Object oldValue) {
                if ("temperature".equals(propertyName)) {
                    System.out.println("Temperature sensor deactivated");
                }
            }
        };
        
        // Register multiple entities
        String[] entityIds = {
            "urn:ngsi-ld:TemperatureSensor:001",
            "urn:ngsi-ld:TemperatureSensor:002",
            "urn:ngsi-ld:HumiditySensor:001"
        };
        
        for (String entityId : entityIds) {
            // TODO: Create actual entities
            // EObject entity = createEntityForId(entityId);
            // entityManager.registerEntity(entityId, entity);
            
            // Add appropriate listeners
            if (entityId.contains("Temperature")) {
                entityManager.addChangeListener(entityId, temperatureListener);
            }
        }
        
        System.out.println("Managing " + entityManager.getManagedEntityCount() + " entities");
        
        // Clean up
        entityManager.dispose();
    }
    
    /**
     * Example main method showing usage
     */
    public static void main(String[] args) {
        System.out.println("=== Basic Usage Example ===");
        demonstrateBasicUsage();
        
        System.out.println("\n=== Advanced Usage Example ===");
        demonstrateAdvancedUsage();
    }
}