/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.emf.ngsi.example;

import java.util.Arrays;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.ngsi.NGSIEntityManager;
import org.gecko.emf.ngsi.NGSISubscription;
import org.gecko.emf.ngsi.NGSISubscriptionManager;
import org.gecko.emf.osgi.example.model.basic.BasicFactory;
import org.gecko.emf.osgi.example.model.basic.Person;
import org.gecko.emf.osgi.example.model.basic.Address;

/**
 * Example demonstrating how to use the NGSI subscription system for monitoring
 * entity changes and receiving notifications.
 * 
 * <p>This example shows:</p>
 * <ul>
 *   <li>Setting up entity and subscription managers</li>
 *   <li>Creating subscriptions with different filtering criteria</li>
 *   <li>Registering entities and monitoring changes</li>
 *   <li>Handling subscription notifications</li>
 * </ul>
 * 
 * <h3>Usage Scenarios:</h3>
 * <ul>
 *   <li>IoT device monitoring</li>
 *   <li>Real-time data synchronization</li>
 *   <li>Event-driven data processing</li>
 *   <li>Change auditing and logging</li>
 * </ul>
 * 
 * @author Generated with Claude Code
 * @since 1.0.0
 */
public class NGSISubscriptionExample {

    /**
     * Sample notification handler that logs changes to the console.
     */
    private static class LoggingNotificationHandler implements NGSISubscriptionManager.NGSINotificationHandler {
        
        private final String handlerName;
        
        public LoggingNotificationHandler(String handlerName) {
            this.handlerName = handlerName;
        }
        
        @Override
        public void onNotification(NGSISubscription subscription, String entityId, EObject entity, 
                                 String changedProperty, Object oldValue, Object newValue) {
            System.out.printf("[%s] Entity %s changed property '%s' from '%s' to '%s'%n", 
                             handlerName, entityId, changedProperty, oldValue, newValue);
        }
    }

    /**
     * Demonstrates basic subscription functionality.
     */
    public static void runBasicSubscriptionExample() {
        System.out.println("=== Basic Subscription Example ===");
        
        // Create managers
        NGSIEntityManager entityManager = new NGSIEntityManager();
        NGSISubscriptionManager subscriptionManager = new NGSISubscriptionManager();
        
        // Connect the managers
        subscriptionManager.attachToEntityManager(entityManager);
        entityManager.registerSubscriptionManager(subscriptionManager);
        
        // Create a subscription for all Person entities
        LoggingNotificationHandler handler = new LoggingNotificationHandler("PersonMonitor");
        NGSISubscription subscription = subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:001",
            "urn:ngsi-ld:Person:.*",  // Monitor all Person entities
            null,                     // Monitor all attributes
            handler
        );
        
        System.out.println("Created subscription: " + subscription);
        
        // Create a person entity
        BasicFactory factory = BasicFactory.eINSTANCE;
        Person person = factory.createPerson();
        person.setFirstName("John");
        person.setTest("30");
        
        System.out.println("Registering person entity...");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Make some changes
        System.out.println("Updating person's first name...");
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        System.out.println("Updating person's age...");
        person.setTest("31");
        entityManager.registerEntity("urn:ngsi-ld:Person:001", person);
        
        // Wait for async notifications
        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        
        // Clean up
        subscriptionManager.dispose();
        entityManager.dispose();
        
        System.out.println("Basic example completed.\n");
    }

    /**
     * Demonstrates attribute filtering in subscriptions.
     */
    public static void runAttributeFilteringExample() {
        System.out.println("=== Attribute Filtering Example ===");
        
        // Create managers
        NGSIEntityManager entityManager = new NGSIEntityManager();
        NGSISubscriptionManager subscriptionManager = new NGSISubscriptionManager();
        
        // Connect the managers
        subscriptionManager.attachToEntityManager(entityManager);
        entityManager.registerSubscriptionManager(subscriptionManager);
        
        // Create a subscription that only monitors the firstName attribute
        LoggingNotificationHandler handler = new LoggingNotificationHandler("FirstNameMonitor");
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:002",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("firstName"),  // Only monitor firstName
            handler
        );
        
        // Create a person entity
        BasicFactory factory = BasicFactory.eINSTANCE;
        Person person = factory.createPerson();
        person.setFirstName("Bob");
        person.setTest("25");
        
        System.out.println("Registering person entity...");
        entityManager.registerEntity("urn:ngsi-ld:Person:002", person);
        
        // Update firstName (should trigger notification)
        System.out.println("Updating monitored attribute (firstName)...");
        person.setFirstName("Alice");
        entityManager.registerEntity("urn:ngsi-ld:Person:002", person);
        
        // Update test attribute (should NOT trigger notification)
        System.out.println("Updating non-monitored attribute (test)...");
        person.setTest("26");
        entityManager.registerEntity("urn:ngsi-ld:Person:002", person);
        
        // Wait for async notifications
        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        
        // Clean up
        subscriptionManager.dispose();
        entityManager.dispose();
        
        System.out.println("Attribute filtering example completed.\n");
    }

    /**
     * Demonstrates entity ID pattern matching.
     */
    public static void runEntityPatternExample() {
        System.out.println("=== Entity Pattern Matching Example ===");
        
        // Create managers
        NGSIEntityManager entityManager = new NGSIEntityManager();
        NGSISubscriptionManager subscriptionManager = new NGSISubscriptionManager();
        
        // Connect the managers
        subscriptionManager.attachToEntityManager(entityManager);
        entityManager.registerSubscriptionManager(subscriptionManager);
        
        // Create a subscription that only monitors temperature sensor entities
        LoggingNotificationHandler handler = new LoggingNotificationHandler("TemperatureMonitor");
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:003",
            "urn:ngsi-ld:TemperatureSensor:.*",  // Only temperature sensors
            null,
            handler
        );
        
        BasicFactory factory = BasicFactory.eINSTANCE;
        
        // Create a person entity (should NOT match)
        Person person = factory.createPerson();
        person.setFirstName("John");
        
        System.out.println("Registering person entity (should not match pattern)...");
        entityManager.registerEntity("urn:ngsi-ld:Person:003", person);
        
        person.setFirstName("Jane");
        entityManager.registerEntity("urn:ngsi-ld:Person:003", person);
        
        // Create a temperature sensor entity (should match)
        Person sensor = factory.createPerson(); // Using Person as sensor placeholder
        sensor.setFirstName("23.5");
        
        System.out.println("Registering temperature sensor entity (should match pattern)...");
        entityManager.registerEntity("urn:ngsi-ld:TemperatureSensor:001", sensor);
        
        System.out.println("Updating temperature sensor...");
        sensor.setFirstName("24.1");
        entityManager.registerEntity("urn:ngsi-ld:TemperatureSensor:001", sensor);
        
        // Wait for async notifications
        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        
        // Clean up
        subscriptionManager.dispose();
        entityManager.dispose();
        
        System.out.println("Entity pattern example completed.\n");
    }

    /**
     * Demonstrates multiple subscriptions for the same entity.
     */
    public static void runMultipleSubscriptionsExample() {
        System.out.println("=== Multiple Subscriptions Example ===");
        
        // Create managers
        NGSIEntityManager entityManager = new NGSIEntityManager();
        NGSISubscriptionManager subscriptionManager = new NGSISubscriptionManager();
        
        // Connect the managers
        subscriptionManager.attachToEntityManager(entityManager);
        entityManager.registerSubscriptionManager(subscriptionManager);
        
        // Create multiple subscriptions for different purposes
        LoggingNotificationHandler nameHandler = new LoggingNotificationHandler("NameTracker");
        LoggingNotificationHandler ageHandler = new LoggingNotificationHandler("AgeTracker");
        LoggingNotificationHandler addressHandler = new LoggingNotificationHandler("AddressTracker");
        
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:004",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("firstName"),
            nameHandler
        );
        
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:005",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("test"),
            ageHandler
        );
        
        subscriptionManager.createSubscription(
            "urn:ngsi-ld:Subscription:006",
            "urn:ngsi-ld:Person:.*",
            Arrays.asList("address"),
            addressHandler
        );
        
        System.out.println("Created 3 subscriptions for different attributes");
        
        // Create a person entity
        BasicFactory factory = BasicFactory.eINSTANCE;
        Person person = factory.createPerson();
        person.setFirstName("Charlie");
        person.setTest("30");
        
        Address address = factory.createAddress();
        address.setStreet("123 Main St");
        person.setAddress(address);
        
        System.out.println("Registering person entity...");
        entityManager.registerEntity("urn:ngsi-ld:Person:004", person);
        
        // Update firstName (should trigger nameHandler only)
        System.out.println("Updating firstName...");
        person.setFirstName("David");
        entityManager.registerEntity("urn:ngsi-ld:Person:004", person);
        
        // Update age (should trigger ageHandler only)
        System.out.println("Updating age...");
        person.setTest("31");
        entityManager.registerEntity("urn:ngsi-ld:Person:004", person);
        
        // Update address (should trigger addressHandler only)
        System.out.println("Updating address...");
        Address newAddress = factory.createAddress();
        newAddress.setStreet("456 Oak Ave");
        person.setAddress(newAddress);
        entityManager.registerEntity("urn:ngsi-ld:Person:004", person);
        
        // Wait for async notifications
        try {
            Thread.sleep(100);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        
        // Clean up
        subscriptionManager.dispose();
        entityManager.dispose();
        
        System.out.println("Multiple subscriptions example completed.\n");
    }

    /**
     * Main method to run all examples.
     */
    public static void main(String[] args) {
        System.out.println("NGSI Subscription System Examples");
        System.out.println("==================================");
        
        runBasicSubscriptionExample();
        runAttributeFilteringExample();
        runEntityPatternExample();
        runMultipleSubscriptionsExample();
        
        System.out.println("All examples completed successfully!");
    }
}