/*
 * Created on 28/ott/2011
 * Copyright 2011 by Andrea Vacondio (andrea.vacondio@gmail.com).
 *
 * This file is part of the Sejda source code
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.sejda.model.task;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.sejda.model.input.TaskSource;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import static java.util.Objects.nonNull;
import static org.sejda.commons.util.RequireUtils.requireNotNullArg;

/**
 * An set of metadata related to the task the event is notifying about.
 *
 * @author Andrea Vacondio
 */
public class NotifiableTaskMetadata implements Serializable {

    private static final long serialVersionUID = -6423865557633949211L;
    /**
     * Null object pattern
     */
    public static final NotifiableTaskMetadata NULL = new NullNotifiableTaskMetadata();
    private UUID taskIdentifier;
    private String qualifiedName;
    private List<File> taskOutput = new ArrayList<>();
    private List<File> skippedOutput = new ArrayList<>();
    private String currentSource;

    private NotifiableTaskMetadata() {
        // empty constructor
    }

    public NotifiableTaskMetadata(Task<?> task) {
        requireNotNullArg(task, "No task given, unable to create notifiable metadata.");
        this.taskIdentifier = UUID.randomUUID();
        this.qualifiedName = task.getClass().getName();
    }

    /**
     * @return the identifier of the task the event is notifying about.
     */
    public UUID getTaskIdentifier() {
        return taskIdentifier;
    }

    /**
     * @return the qualified name of the task the event is notifying about.
     */
    public String getQualifiedName() {
        return qualifiedName;
    }

    public void addTaskOutput(File output) {
        if (nonNull(output)) {
            taskOutput.add(output);
        }
    }

    public void addSkippedOutput(File skipped) {
        if (nonNull(skipped)) {
            skippedOutput.add(skipped);
        }
    }

    public void setCurrentSource(TaskSource<?> source) {
        this.currentSource = source.getName();
    }

    public void clearCurrentSource() {
        this.currentSource = "";
    }

    public String getCurrentSource() {
        return this.currentSource;
    }

    /**
     * @return the list of files generated by the task. This is empty if the task is set to write to an output stream
     */
    public List<File> taskOutput() {
        return Collections.unmodifiableList(taskOutput);
    }

    /**
     * @return the list of output files that where skipped because already existing (depending on the {@link org.sejda.model.output.ExistingOutputPolicy})
     */
    public List<File> skippedOutput() {
        return Collections.unmodifiableList(skippedOutput);
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder().append(taskIdentifier).append(qualifiedName).toHashCode();
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof NotifiableTaskMetadata meta)) {
            return false;
        }
        return new EqualsBuilder().append(taskIdentifier, meta.taskIdentifier).append(qualifiedName, meta.qualifiedName)
                .isEquals();
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE).append("taskIdentifier", taskIdentifier)
                .append("qualifiedName", qualifiedName).toString();
    }

    /**
     * Null object pattern providing empty behavior.
     *
     * @author Andrea Vacondio
     */
    private static class NullNotifiableTaskMetadata extends NotifiableTaskMetadata {

        private static final long serialVersionUID = 6788562820506828221L;

        @Override
        public UUID getTaskIdentifier() {
            return null;
        }

        @Override
        public String getQualifiedName() {
            return StringUtils.EMPTY;
        }
    }
}
