/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.chat.completion.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService;
import org.eclipse.fennec.ai.chat.completion.helper.ChatCompletionHelper;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.Candidate;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.Content;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.GeminiChatCompletionFactory;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.GeminiChatCompletionPackage;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.GeminiChatCompletionRequest;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.GeminiChatCompletionResponse;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.GenerationConfig;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.Part;
import org.eclipse.fennec.ai.chat.completion.model.geminichatcompletion.SystemInstruction;
import org.eclipse.fennec.ai.jsonschema.model.jsonschema.JsonSchema;
import org.eclipse.fennec.codec.options.CodecModuleOptions;
import org.eclipse.fennec.codec.options.CodecResourceOptions;
import org.eclipse.fennec.qvt.osgi.api.ModelTransformator;
import org.gecko.emf.json.constants.EMFJs;
import org.gecko.emf.osgi.constants.EMFNamespaces;
import org.gecko.emf.osgi.constants.EMFUriHandlerConstants;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;

/**
 * 
 * @author ilenia
 * @since Jun 10, 2025
 */
@Component(name = "GeminiChatCompletionService", configurationPid = "GeminiChatCompletionService", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class GeminiChatCompletionServiceImpl implements ChatCompletionService {

	@Reference(target="(transformator.id=ecorejsonschema)", cardinality = ReferenceCardinality.OPTIONAL)
	ModelTransformator qvtTransformator;

	private static final Logger LOGGER = Logger.getLogger(GeminiChatCompletionServiceImpl.class.getName());

	private ResourceSet resSet;
	private String url;

	@Activate
	public GeminiChatCompletionServiceImpl(@Reference(cardinality = ReferenceCardinality.MANDATORY, target="("+EMFNamespaces.EMF_MODEL_FILE_EXT + "=json)") 
	ResourceSet resSet, Map<String, Object> properties) {
		this.resSet = resSet;
		url = (String) properties.getOrDefault("url", null);
		Objects.requireNonNull(url, "You must provide a url in the configuration for the GeminiChatCompletionService");		
		LOGGER.info("Gemini Chat Completion Service configured with URL: " + url);
	}

	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService#complete(org.eclipse.emf.ecore.EClass, java.lang.String, java.lang.String)
	 */
	@Override
	public EObject complete(EClass resultEClass, String systemMsg, String userMsg) {
		//		1. convert EClass to JsonSchema using mmt
		//		2. send request to AI
		//		3. convert back result to EClass
		JsonSchema schema = qvtTransformator.doTransformation(resultEClass);
		GeminiChatCompletionRequest request = createChatCompletionRequest(schema, systemMsg, userMsg);
		GeminiChatCompletionResponse response = sendChatCompletionRequest(request);
		if(response != null) {
			if(response.getCandidates().isEmpty()) { 
				LOGGER.severe(String.format("No candidates in completion response. Cannot continue!"));
			} else {
				Candidate candidate = response.getCandidates().get(0);
				if(candidate.getContent() != null && !candidate.getContent().getParts().isEmpty() && candidate.getContent().getParts().get(0).getText() != null) {
					Part part = candidate.getContent().getParts().get(0);					
					String completion = part.getText();
					return ChatCompletionHelper.convertToEClass(completion, resultEClass, resSet);
				} else {
					LOGGER.severe(String.format("Completion did not produce any content. Cannot continue!"));
				}
			}			
		}
		return null;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService#complete(java.lang.String, java.lang.String)
	 */
	@Override
	public String complete(String systemMsg, String userMsg) {
		GeminiChatCompletionRequest request = createChatCompletionRequest(systemMsg, userMsg);
		GeminiChatCompletionResponse response = sendChatCompletionRequest(request);
		if(response != null) {
			if(response.getCandidates().isEmpty()) { 
				LOGGER.severe(String.format("No candidates in completion response. Cannot continue!"));
			} else {
				Candidate candidate = response.getCandidates().get(0);
				if(candidate.getContent() != null && !candidate.getContent().getParts().isEmpty() && candidate.getContent().getParts().get(0).getText() != null) {
					Part part = candidate.getContent().getParts().get(0);					
					String completion = part.getText();
					return completion;
				} else {
					LOGGER.severe(String.format("Completion did not produce any content. Cannot continue!"));
				}
			}			
		}
		return null;
	}

	private GeminiChatCompletionRequest createChatCompletionRequest(JsonSchema jsonSchema, String systemMsg, String userMsg) {
		GeminiChatCompletionRequest request = GeminiChatCompletionFactory.eINSTANCE.createGeminiChatCompletionRequest();

		Content content = GeminiChatCompletionFactory.eINSTANCE.createContent();
		content.setRole("user");
		Part part = GeminiChatCompletionFactory.eINSTANCE.createPart();
		part.setText(userMsg);
		content.getParts().add(part);

		SystemInstruction systemInstruction = GeminiChatCompletionFactory.eINSTANCE.createSystemInstruction();
		part = GeminiChatCompletionFactory.eINSTANCE.createPart();
		part.setText(systemMsg);
		systemInstruction.getParts().add(part);

		GenerationConfig config = GeminiChatCompletionFactory.eINSTANCE.createGenerationConfig();
		config.setResponseMimeType("application/json");
		config.setResponseJsonSchema(jsonSchema);

		request.getContents().add(content);
		request.setSystemInstruction(systemInstruction);
		request.setGenerationConfig(config);
		return request;		
	}

	private GeminiChatCompletionRequest createChatCompletionRequest(String systemMsg, String userMsg) {
		GeminiChatCompletionRequest request = GeminiChatCompletionFactory.eINSTANCE.createGeminiChatCompletionRequest();

		Content content = GeminiChatCompletionFactory.eINSTANCE.createContent();
		content.setRole("user");
		Part part = GeminiChatCompletionFactory.eINSTANCE.createPart();
		part.setText(userMsg);
		content.getParts().add(part);

		SystemInstruction systemInstruction = GeminiChatCompletionFactory.eINSTANCE.createSystemInstruction();
		part = GeminiChatCompletionFactory.eINSTANCE.createPart();
		part.setText(systemMsg);
		systemInstruction.getParts().add(part);

		GenerationConfig config = GeminiChatCompletionFactory.eINSTANCE.createGenerationConfig();
		config.setResponseMimeType("application/json");

		request.getContents().add(content);
		request.setSystemInstruction(systemInstruction);
		request.setGenerationConfig(config);
		return request;		
	}
	
	private GeminiChatCompletionResponse sendChatCompletionRequest(GeminiChatCompletionRequest request) {

		Resource responseRes = resSet.createResource(URI.createURI(UUID.randomUUID().toString()+".json"), "application/json");
		Map<String, Object> options = getRequestOptions(responseRes);
		ChatCompletionHelper.sendChatCompletionRequest(url, request, options, resSet);
		if(!responseRes.getContents().isEmpty()) {
			if(responseRes.getContents().get(0) instanceof GeminiChatCompletionResponse response) {
				return response;
			} else {
				LOGGER.severe(String.format("Response object is not of expected type GeminiChatCompletionResponse"));
			}
		} else {
			LOGGER.severe(String.format("Response does NOT contain any object"));
		}
		return null;		
	}

	private Map<String, Object> getRequestOptions(Resource responseRes) {

		Map<String, Object> options = new HashMap<>();
		Map<String, Object> headers = new HashMap<>();		
		headers.put("Accept", "application/json");
		headers.put("Content-Type", "application/json");	
		headers.put("Method", "POST");
		options.put(EMFUriHandlerConstants.OPTION_HTTP_METHOD, "POST");	
		options.put(EMFUriHandlerConstants.OPTION_HTTP_HEADERS, headers);
		options.put(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE, responseRes);
		options.put(EMFJs.OPTION_SERIALIZE_TYPE, false);
		options.put(EMFJs.OPTION_SERIALIZE_DEFAULT_VALUE, true);
		options.put(CodecModuleOptions.CODEC_MODULE_SERIALIZE_TYPE, false);
		options.put(CodecModuleOptions.CODEC_MODULE_SERIALIZE_DEFAULT_VALUE, true);

		//		This option fixes the issue with the read timeout exception. 
		options.put(URIConverter.OPTION_TIMEOUT, 0);

		Map<String, Object> responseOptions = new HashMap<>();
		responseOptions.put(CodecResourceOptions.CODEC_ROOT_OBJECT, GeminiChatCompletionPackage.Literals.GEMINI_CHAT_COMPLETION_RESPONSE);
		responseOptions.put(EMFJs.OPTION_ROOT_ELEMENT, GeminiChatCompletionPackage.Literals.GEMINI_CHAT_COMPLETION_RESPONSE);
		responseOptions.put("Accepts", "application/json");
		options.put(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE_OPTIONS, responseOptions);
		return options;
	}

	

}
