/*
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *      Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.ai.chat.completion.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService;
import org.eclipse.fennec.ai.chat.completion.helper.ChatCompletionHelper;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.ChatCompletionFactory;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.ChatCompletionPackage;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.ChatCompletionRequest;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.ChatCompletionResponse;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.Choice;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.Message;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.ResponseFormat;
import org.eclipse.fennec.ai.chat.completion.model.chatcompletion.Schema;
import org.eclipse.fennec.ai.jsonschema.model.jsonschema.JsonSchema;
import org.eclipse.fennec.codec.options.CodecModuleOptions;
import org.eclipse.fennec.codec.options.CodecResourceOptions;
import org.eclipse.fennec.qvt.osgi.api.ModelTransformator;
import org.gecko.emf.json.constants.EMFJs;
import org.gecko.emf.osgi.constants.EMFNamespaces;
import org.gecko.emf.osgi.constants.EMFUriHandlerConstants;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;

@Component(name = "OpenAIChatCompletionService", configurationPid = "OpenAIChatCompletionService", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class OpenAIChatCompletionServiceImpl implements ChatCompletionService{	
	
	@Reference(target="(transformator.id=ecorejsonschema)", cardinality = ReferenceCardinality.OPTIONAL)
	ModelTransformator qvtTransformator;
	
	private static final Logger LOGGER = Logger.getLogger(OpenAIChatCompletionServiceImpl.class.getName());
	
	private ResourceSet resSet;
	private String url;
	private String apiKey;
	private String model;

	@Activate
	public OpenAIChatCompletionServiceImpl(@Reference(cardinality = ReferenceCardinality.MANDATORY, target="("+EMFNamespaces.EMF_MODEL_FILE_EXT + "=json)") 
	ResourceSet resSet, Map<String, Object> properties) {
		this.resSet = resSet;
		url = (String) properties.getOrDefault("url", null);
		Objects.requireNonNull(url, "You must provide a url in the configuration for the OpenAIChatCompletionService");
		apiKey = (String) properties.getOrDefault("api.key", null);
		Objects.requireNonNull(apiKey, "You must provide an API Key in the configuration for the OpenAIChatCompletionService");
		model = (String) properties.getOrDefault("model", "gpt-4o-2024-08-06");
	}
	
	
	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService#complete(org.eclipse.emf.ecore.EClass, java.lang.String, java.lang.String)
	 */
	@Override
	public EObject complete(EClass resultEClass, String systemMsg, String userMsg) {
//		1. convert EClass to JsonSchema using mmt
//		2. send request to AI
//		3. convert back result to EClass
		JsonSchema schema = qvtTransformator.doTransformation(resultEClass);
		ChatCompletionRequest request = createChatCompletionRequest(schema, systemMsg, userMsg);
		ChatCompletionResponse response = sendChatCompletionRequest(request);
		if(response != null) {
			if(response.getChoices().isEmpty()) { 
				LOGGER.severe(String.format("No choices in completion response. Cannot continue!"));
			} else {
				Choice choice = response.getChoices().get(0);
				if(choice.getMessage() != null && choice.getMessage().getContent() != null) {
					String completion = choice.getMessage().getContent();
					return ChatCompletionHelper.convertToEClass(completion, resultEClass, resSet);
				} else {
					LOGGER.severe(String.format("Completion did not produce any content. Cannot continue!"));
				}
			}			
		}
		return null;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.ai.chat.completion.api.ChatCompletionService#complete(java.lang.String, java.lang.String)
	 */
	@Override
	public String complete(String systemMsg, String userMsg) {
		throw new UnsupportedOperationException("Currently not implemented");
	}
	
	private Map<String, Object> getRequestOptions(Resource responseRes ) {
		
		Map<String, Object> options = new HashMap<>();
		Map<String, Object> headers = new HashMap<>();		
		headers.put("Accept", "application/json");
		headers.put("Content-Type", "application/json");	
		headers.put("Authorization", "Bearer " + apiKey);	
		headers.put("Method", "POST");
		options.put(EMFUriHandlerConstants.OPTION_HTTP_METHOD, "POST");	
		options.put(EMFUriHandlerConstants.OPTION_HTTP_HEADERS, headers);
		options.put(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE, responseRes);
		options.put(EMFJs.OPTION_SERIALIZE_TYPE, false);
		options.put(EMFJs.OPTION_SERIALIZE_DEFAULT_VALUE, true);
		options.put(CodecModuleOptions.CODEC_MODULE_SERIALIZE_TYPE, false);
		options.put(CodecModuleOptions.CODEC_MODULE_SERIALIZE_DEFAULT_VALUE, true);
		
//		This option fixes the issue with the read timeout exception. 
		options.put(URIConverter.OPTION_TIMEOUT, 0);
		
		Map<String, Object> responseOptions = new HashMap<>();
		responseOptions.put(CodecResourceOptions.CODEC_ROOT_OBJECT, ChatCompletionPackage.Literals.CHAT_COMPLETION_RESPONSE);
		responseOptions.put(EMFJs.OPTION_ROOT_ELEMENT, ChatCompletionPackage.Literals.CHAT_COMPLETION_RESPONSE);
		responseOptions.put("Accepts", "application/json");
		options.put(EMFUriHandlerConstants.OPTIONS_EXPECTED_RESPONSE_RESOURCE_OPTIONS, responseOptions);
		return options;
	}

	private ChatCompletionResponse sendChatCompletionRequest(ChatCompletionRequest request) {		
		Resource responseRes = resSet.createResource(URI.createURI(UUID.randomUUID().toString()+".json"), "application/json");
		Map<String, Object> options = getRequestOptions(responseRes);
		ChatCompletionHelper.sendChatCompletionRequest(url, request, options, resSet);
			if(!responseRes.getContents().isEmpty()) {
				if(responseRes.getContents().get(0) instanceof ChatCompletionResponse response) {
					return response;
				} else {
					LOGGER.severe(String.format("Response object is not of expected type ChatCompletionResponse"));
				}
			} else {
				LOGGER.severe(String.format("Response does NOT contain any object"));
			}
		return null;		
	}

	private ChatCompletionRequest createChatCompletionRequest(JsonSchema jsonSchema, String systemMsg, String userMsg) {
		ChatCompletionRequest request = ChatCompletionFactory.eINSTANCE.createChatCompletionRequest();
		request.setModel(model);
		Message message = ChatCompletionFactory.eINSTANCE.createMessage();
		message.setContent(userMsg);
		message.setRole("user");
		request.getMessages().add(message);
		message = ChatCompletionFactory.eINSTANCE.createMessage();
		message.setContent(systemMsg);
		message.setRole("system");
		request.getMessages().add(message);
		ResponseFormat responseFormat = ChatCompletionFactory.eINSTANCE.createResponseFormat();
		Schema schema = ChatCompletionFactory.eINSTANCE.createSchema();
		schema.setName("response_schema");
		schema.setSchema(jsonSchema);
		responseFormat.setJsonSchema(schema);		
		responseFormat.setType("json_schema");
		request.setResponseFormat(responseFormat);
		return request;		
	}


	
}
