/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.eclipse.fennec.apisix.api;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Eclipse Modeling Fennec annotation to configure APISIX route registration for JAX-RS resources.
 * When applied to a JAX-RS resource class, the Eclipse Fennec APISIX adapter will automatically
 * register/unregister routes based on the service lifecycle using the Eclipse Modeling Framework (EMF)
 * for type-safe APISIX configuration.
 * 
 * <p>Eclipse Modeling Fennec provides seamless integration between OSGi JAX-RS services and APISIX API Gateway,
 * enabling automatic route discovery and configuration management.</p>
 * 
 * @author Mark Hoffmann
 * @since 16.09.2025
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
public @interface ApiSixRoute {

    /**
     * The external URI pattern for the route (e.g., "/api/v1/users/*")
     * If not specified, will be derived from the JAX-RS @Path annotation
     */
    String uri() default "";
    
    /**
     * The route ID in APISIX. If not specified, will be generated based
     * on the service class name.
     */
    String routeId() default "";
    
    /**
     * Priority of the route (higher values have higher priority)
     */
    int priority() default 0;
    
    /**
     * Whether to enable path rewriting. When true, the external URI
     * will be rewritten to match the internal service path.
     */
    boolean enablePathRewrite() default true;
    
    /**
     * Custom path rewrite pattern. If specified, overrides the default
     * rewriting behavior. Format: ["^/external/(.*)$", "/internal/$1"]
     */
    String[] pathRewritePattern() default {};
    
    /**
     * Additional plugins to enable for this route (e.g., "rate-limit", "cors")
     */
    String[] plugins() default {};
    
    /**
     * Whether this route should be automatically registered when the
     * service becomes available
     */
    boolean autoRegister() default true;
    
}
