/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.apisix.tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.eclipse.fennec.apisix.ApiSixClient;
import org.eclipse.fennec.apisix.api.ApiSixRoute;
import org.eclipse.fennec.model.apisix.Route;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.annotations.RequireConfigurationAdmin;
import org.osgi.service.jakartars.runtime.JakartarsServiceRuntime;
import org.osgi.service.jakartars.runtime.dto.ApplicationDTO;
import org.osgi.service.jakartars.runtime.dto.ResourceDTO;
import org.osgi.service.jakartars.runtime.dto.RuntimeDTO;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;
import org.osgi.util.tracker.ServiceTracker;

import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.Response;

/**
 * Integration tests for APISIX route registration using mocked ApiSixClient
 */
@RequireEMF
@RequireJakartarsWhiteboard
@RequireConfigurationAdmin
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
public class ApiSixAdapterIntegrationTest {

	@Mock
	private ApiSixClient mockApiSixClient;

	private ServiceRegistration<ApiSixClient> mockClientRegistration;

	@InjectService
	ClientBuilder clientBuilder;

	@BeforeEach
	public void setup(@InjectBundleContext BundleContext ctx) {
		// Register the mock ApiSixClient with higher service ranking
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("service.ranking", 1000); // Higher than default (0)

		mockClientRegistration = ctx.registerService(ApiSixClient.class, mockApiSixClient, props);

		// Reset mock for clean state
		reset(mockApiSixClient);
	}

	@AfterEach
	public void cleanup() {
		if (mockClientRegistration != null) {
			mockClientRegistration.unregister();
		}
	}

	@Test
	public void testRouteRegistrationWithAnnotation(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create a test JAX-RS resource with ApiSixRoute annotation
		TestResource testResource = new TestResource();

		// Register the test resource as a JAX-RS service
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", true);
		props.put("osgi.jakartars.name", "test");


		ResourceAware resourceAware = ResourceAware.create(ctx, "test");

		ServiceRegistration<Object> resourceRegistration = ctx.registerService(Object.class, testResource, props);

		checkAvailability(clientBuilder, resourceAware, "/rest/test/hello", "Hello from test!");
		// Verify that createRoute was called on our mock
		ArgumentCaptor<Route> routeCaptor = ArgumentCaptor.forClass(Route.class);
		verify(mockApiSixClient, times(1)).createRoute(routeCaptor.capture());

		// Verify the route details
		Route capturedRoute = routeCaptor.getValue();
		assertNotNull(capturedRoute);
		assertEquals("auto-testresource", capturedRoute.getId());
		assertTrue(capturedRoute.getUri().contains("/test/hello"));

		resourceRegistration.unregister();

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
		
		// Give the ApiSixAdapter time to process the change
		Thread.sleep(1000);
		
		// Verify that deleteRoute was called
		verify(mockApiSixClient, times(1)).deleteRoute("auto-testresource");


	}

	@Test
	public void testRouteRegistrationWithoutAnnotation(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create a test JAX-RS resource WITHOUT ApiSixRoute annotation
		TestResourceWithoutAnnotation testResource = new TestResourceWithoutAnnotation();

		// Register the test resource as a JAX-RS service
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", true);
		props.put("osgi.jakartars.name", "test-no-annotation");

		ResourceAware resourceAware = ResourceAware.create(ctx, "test-no-annotation");

		ServiceRegistration<Object> resourceRegistration = ctx.registerService(Object.class, testResource, props);

		checkAvailability(clientBuilder, resourceAware, "/rest/test/noannotation", "Hello without annotation!");

		// Verify that createRoute was NOT called (no annotation)
		verify(mockApiSixClient, times(0)).createRoute(any(Route.class));
		
		resourceRegistration.unregister();

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
	}

	@Test
	public void testEndpointAccessibility(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		assertNotNull(clientBuilder);
		// Create a test JAX-RS resource
		TestResource testResource = new TestResource();

		// Register the test resource as a JAX-RS service
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", true);
		props.put("osgi.jakartars.name", "test-endpoint");

		ResourceAware resourceAware = ResourceAware.create(ctx, "test-endpoint");

		ServiceRegistration<Object> resourceRegistration = ctx.registerService(Object.class, testResource, props);

		checkAvailability(clientBuilder, resourceAware, "/rest/test/hello", "Hello from test!");

		resourceRegistration.unregister();

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
	}

	/**
	 * @param clientBuilder
	 * @throws InterruptedException 
	 */
	private void checkAvailability(ClientBuilder clientBuilder, ResourceAware resourceAware, String path, String content) throws InterruptedException {

		boolean resourceFound = resourceAware.waitForResource(5, TimeUnit.SECONDS);
		assertTrue(resourceFound, "Resource should be registered in Jakarta REST runtime within 5 seconds");
		// Test that the endpoint is accessible (not 404)
		Client client = clientBuilder.build();
		WebTarget webTarget = client.target("http://localhost:8085").path(path);
		// Verify the response content
		try {
			checkUrlForResponse(webTarget, 200, content);
		} finally {
			client.close();
		}
	}

	private static void checkUrlForResponse(WebTarget webTarget, int responseCode, String text) {
		assertNotNull(webTarget);
		assertNotNull(text);
		try {
			// Call the endpoint - this assumes the REST whiteboard is running on default port
			// The exact URL depends on your whiteboard configuration
			Response response = webTarget.request()
					.get();

			// Verify the endpoint returns a successful response (not 404)
			assertTrue(response.getStatus() == responseCode, 
					"Endpoint should be accessible, got status: " + response.getStatus());

			// Verify the response content
			String responseText = response.readEntity(String.class);
			assertEquals(text, responseText);

			response.close();

		} catch (Exception e) {
			// If we can't reach the endpoint, fail the test
			throw new AssertionError("Failed to call endpoint: " + e.getMessage(), e);
		}
	}

	public static class ResourceAware {

		private final ServiceTracker<JakartarsServiceRuntime, JakartarsServiceRuntime> runtimeTracker;
		private final CountDownLatch addLatch;
		private final CountDownLatch removeLatch;

		public static ResourceAware create(BundleContext ctx, String resourceName) {
			assertNotNull(ctx);
			assertNotNull(resourceName);
			return new ResourceAware(ctx, resourceName);
		}

		private ResourceAware(BundleContext ctx, String resourceName) {
			addLatch = new CountDownLatch(1);
			removeLatch = new CountDownLatch(1);
			runtimeTracker = new ServiceTracker<JakartarsServiceRuntime, JakartarsServiceRuntime>(ctx, JakartarsServiceRuntime.class, null) {

				@Override
				public JakartarsServiceRuntime addingService(
						org.osgi.framework.ServiceReference<JakartarsServiceRuntime> reference) {
					JakartarsServiceRuntime runtime = super.addingService(reference);
					checkForResourceRegistration(runtime, false);
					return runtime;
				}

				@Override
				public void modifiedService(
						org.osgi.framework.ServiceReference<JakartarsServiceRuntime> reference, 
						JakartarsServiceRuntime service) {
					super.modifiedService(reference, service);
					checkForResourceRegistration(service, false);
				}

				/* 
				 * (non-Javadoc)
				 * @see org.osgi.util.tracker.ServiceTracker#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
				 */
				@Override
				public void removedService(ServiceReference<JakartarsServiceRuntime> reference,
						JakartarsServiceRuntime service) {
					super.removedService(reference, service);
					checkForResourceRegistration(service, true);
				}

				private void checkForResourceRegistration(JakartarsServiceRuntime runtime, boolean removal) {
					if (runtime != null) {
						RuntimeDTO runtimeDTO = runtime.getRuntimeDTO();
						if (runtimeDTO != null && runtimeDTO.applicationDTOs != null) {
							ApplicationDTO appDTO = runtimeDTO.defaultApplication;
							if (appDTO.resourceDTOs != null) {
								boolean found = false;
								for (ResourceDTO resourceDTO : appDTO.resourceDTOs) {
									// Check if this is our test resource by looking for the test path
									if (resourceName.equals(resourceDTO.name)) {
										found = true;
										if (!removal) {
											addLatch.countDown();
											return;
										}
									}
								}
								if (removal && !found) {
									removeLatch.countDown();
									return;
								} else if (!removal && found) {
									addLatch.countDown();
									return;
								}
							} else if (removal) {
								// No resources at all means it was removed
								removeLatch.countDown();
								return;
							}
						}
					}
				}
			};

			runtimeTracker.open(true);
		}

		boolean waitForResource(long value, TimeUnit unit) throws InterruptedException {
			try {
				return addLatch.await(value, unit);
			} finally {
				runtimeTracker.close();
			}
		}

		boolean waitForResourceRemoval(long value, TimeUnit unit) throws InterruptedException {
			try {
				return removeLatch.await(value, unit);
			} finally {
				runtimeTracker.close();
			}
		}


	}

	// Test JAX-RS resource WITH ApiSixRoute annotation
	@Path("/test/hello")
	@ApiSixRoute(routeId = "auto-testresource", uri = "/test/hello/*")
	public static class TestResource {

		@GET
		public Response hello() {
			return Response.ok("Hello from test!").build();
		}
	}

	// Test JAX-RS resource WITHOUT ApiSixRoute annotation
	@Path("/test/noannotation")
	public static class TestResourceWithoutAnnotation {

		@GET
		public Response hello() {
			return Response.ok("Hello without annotation!").build();
		}
	}
}
