/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.apisix.impl;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.fennec.apisix.ApiSixClient;
import org.eclipse.fennec.apisix.api.ApiSixRoute;
import org.eclipse.fennec.model.apisix.ApisixFactory;
import org.eclipse.fennec.model.apisix.LoadBalancerType;
import org.eclipse.fennec.model.apisix.ProxyRewritePlugin;
import org.eclipse.fennec.model.apisix.Route;
import org.eclipse.fennec.model.apisix.Upstream;
import org.eclipse.fennec.model.apisix.UpstreamNode;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.jakartars.runtime.JakartarsServiceRuntime;
import org.osgi.service.jakartars.runtime.dto.ApplicationDTO;
import org.osgi.service.jakartars.runtime.dto.ResourceDTO;
import org.osgi.service.jakartars.runtime.dto.RuntimeDTO;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.Path;

/**
 * OSGi component that automatically registers and unregisters APISIX routes
 * for JAX-RS resources annotated with {@link ApiSixRoute}.
 * 
 * This adapter monitors the Jakarta REST Whiteboard runtime service for changes
 * and manages APISIX route configurations automatically based on registered resources.
 * 
 * @author Mark Hoffmann
 * @since 16.09.2025
 */
@Component
@RequireJakartarsWhiteboard
public class ApiSixAdapter {

    private static final Logger logger = LoggerFactory.getLogger(ApiSixAdapter.class);
    
    @Reference
    private ApiSixClient apiSixClient;
    
    private AtomicReference<JakartarsServiceRuntime> jakartarsRuntimeRef = new AtomicReference<>();
    private final Map<String, Object> runtimeProperties = new ConcurrentHashMap<>();
    
    private BundleContext bundleContext;
    private final Map<Long, Route> registeredRoutes = new ConcurrentHashMap<>();
    private long lastChangeCount = -1;
    
    @Activate
    void activate(BundleContext context) {
        this.bundleContext = context;
        logger.info("APISIX Adapter activated - monitoring Jakarta REST runtime for route registration");
        // Process initial state
        processRuntimeChanges();
    }
    
    @Deactivate
    void deactivate() {
        unregisterAllRoutes();
        logger.info("APISIX Adapter deactivated");
    }
    
    @Reference(cardinality = ReferenceCardinality.OPTIONAL, policy = ReferencePolicy.DYNAMIC , unbind = "removeRuntimeService", updated = "modifyRuntimeService")
    public void addRuntimeService(JakartarsServiceRuntime jakartarsRuntime, Map<String, Object> properties) {
    	jakartarsRuntimeRef.compareAndSet(null, jakartarsRuntime);
    	runtimeProperties.putAll(properties);
    	processRuntimeChanges();
    }
    
    public void modifyRuntimeService(JakartarsServiceRuntime jakartarsRuntime, Map<String, Object> properties) {
    	if (jakartarsRuntime.equals(jakartarsRuntimeRef.get())) {
    		runtimeProperties.putAll(properties);
    		processRuntimeChanges();
    	}
    }
    
    public void removeRuntimeService(JakartarsServiceRuntime jakartarsRuntime) {
    	if (jakartarsRuntimeRef.compareAndSet(jakartarsRuntime, null)) {
    		unregisterAllRoutes();
    	}
    }

	/**
	 * Clean up all routes
	 */
	private void unregisterAllRoutes() {
		registeredRoutes.values().forEach(route -> {
			try {
				apiSixClient.deleteRoute(route.getId());
				logger.info("Removed route {} due to runtime service removal", route.getId());
			} catch (Exception e) {
				logger.warn("Failed to remove route {} during runtime service removal: {}", route.getId(), e.getMessage());
			}
		});
		registeredRoutes.clear();
	}
    
    /**
     * Processes all resources from the Jakarta REST runtime and manages APISIX routes accordingly
     */
    private void processRuntimeChanges() {
        try {
            JakartarsServiceRuntime runtime = jakartarsRuntimeRef.get();
            if (runtime == null) {
                logger.debug("No Jakarta REST runtime service available yet");
                return;
            }
            
            // Check if change count has actually changed
            Long currentChangeCount = (Long) runtimeProperties.get("service.changecount");
            if (currentChangeCount != null && currentChangeCount.equals(lastChangeCount)) {
                logger.debug("Change count unchanged ({}), skipping processing", currentChangeCount);
                return;
            }
            
            RuntimeDTO runtimeDTO = runtime.getRuntimeDTO();
            if (runtimeDTO == null) {
                logger.warn("No runtime DTO available");
                return;
            }
            
            // Track current resource service IDs
            Set<Long> currentServiceIds = new HashSet<>();
             
            // Process all applications
            processApplication(currentServiceIds, runtimeDTO.defaultApplication);
            for (ApplicationDTO applicationDTO : runtimeDTO.applicationDTOs) {
            	// Process all resources
            	processApplication(currentServiceIds, applicationDTO);
            	
            }
            
            // Remove routes for resources that are no longer present
            Set<Long> removedServiceIds = new HashSet<>(registeredRoutes.keySet());
            removedServiceIds.removeAll(currentServiceIds);
            
            for (Long serviceId : removedServiceIds) {
                removeRoute(serviceId);
            }
            
            lastChangeCount = currentChangeCount;
                
        } catch (Exception e) {
            logger.error("Failed to process Jakarta REST runtime changes: {}", e.getMessage(), e);
        }
    }

	/**
	 * @param currentServiceIds
	 * @param applicationDTO
	 * @param basePath
	 */
	private void processApplication(Set<Long> currentServiceIds, ApplicationDTO applicationDTO) {
		String basePath = applicationDTO.base;
		for (ResourceDTO resourceDTO : applicationDTO.resourceDTOs) {
			currentServiceIds.add(resourceDTO.serviceId);
			
			if (!registeredRoutes.containsKey(resourceDTO.serviceId)) {
				// New resource - check if it needs APISIX route
				processNewResource(resourceDTO, basePath);
			}
		}
	}
    
    /**
     * Processes a new resource discovered from the runtime DTO
     */
    private void processNewResource(ResourceDTO resourceDTO, String applicationBasePath) {
        try {
            // Get service reference from service ID
            ServiceReference<?> serviceRef = null;
            try {
                ServiceReference<?>[] refs = bundleContext.getAllServiceReferences(null, null);
                for (ServiceReference<?> ref : refs) {
                    if (ref.getProperty("service.id").equals(resourceDTO.serviceId)) {
                        serviceRef = ref;
                        break;
                    }
                }
            } catch (Exception e) {
                logger.warn("Failed to search for service reference with ID {}: {}", resourceDTO.serviceId, e.getMessage());
                return;
            }
            
            if (serviceRef == null) {
                logger.warn("No service reference found for service ID: {}", resourceDTO.serviceId);
                return;
            }
            
            // Get the service instance
            Object service = bundleContext.getService(serviceRef);
            if (service == null) {
                logger.warn("No service instance found for service ID: {}", resourceDTO.serviceId);
                return;
            }
            
            try {
                Class<?> serviceClass = service.getClass();
                
                // Check for ApiSixRoute annotation
                ApiSixRoute routeAnnotation = serviceClass.getAnnotation(ApiSixRoute.class);
                if (routeAnnotation == null) {
                    return; // Not an APISIX-managed resource
                }
                
                // Check for Path annotation
                Path pathAnnotation = serviceClass.getAnnotation(Path.class);
                if (pathAnnotation == null) {
                    logger.warn("Class {} has @ApiSixRoute but no @Path annotation - skipping", serviceClass.getName());
                    return;
                }
                
                if (!routeAnnotation.autoRegister()) {
                    logger.debug("Auto-registration disabled for {}", serviceClass.getName());
                    return;
                }
                
                // Create and register the route
                Route route = createRoute(serviceClass, routeAnnotation, pathAnnotation, applicationBasePath);
                apiSixClient.createRoute(route);
                registeredRoutes.put(resourceDTO.serviceId, route);
                
                logger.info("Registered APISIX route: {} (ID: {}) for service ID: {}", 
                           route.getUri(), route.getId(), resourceDTO.serviceId);
                           
            } finally {
                bundleContext.ungetService(serviceRef);
            }
            
        } catch (Exception e) {
            logger.error("Failed to process new resource with service ID {}: {}", resourceDTO.serviceId, e.getMessage(), e);
        }
    }
    
    /**
     * Removes a route for the given service ID
     */
    private void removeRoute(Long serviceId) {
        Route route = registeredRoutes.remove(serviceId);
        if (route != null) {
            try {
                apiSixClient.deleteRoute(route.getId());
                logger.info("Unregistered APISIX route: {} (ID: {}) for service ID: {}", 
                           route.getUri(), route.getId(), serviceId);
            } catch (Exception e) {
                logger.error("Failed to unregister APISIX route {} for service ID {}: {}", 
                           route.getId(), serviceId, e.getMessage(), e);
            }
        }
    }
    
    /**
     * Creates an ECore Route model from the annotations using runtime service information
     */
    private Route createRoute(Class<?> resourceClass, ApiSixRoute routeAnnotation, Path pathAnnotation, String applicationBasePath) {
        
        ApisixFactory factory = ApisixFactory.eINSTANCE;
        Route route = factory.createRoute();
        
        // Set route ID
        String routeId = routeAnnotation.routeId().isEmpty() 
            ? generateRouteId(resourceClass) 
            : routeAnnotation.routeId();
        route.setId(routeId);
        
        // Set external URI
        String externalUri = routeAnnotation.uri().isEmpty()
            ? generateExternalUri(applicationBasePath, pathAnnotation.value())
            : routeAnnotation.uri();
        route.setUri(externalUri);
        
        // Set priority if specified
        if (routeAnnotation.priority() > 0) {
            route.setPriority(routeAnnotation.priority());
        }
        
        // Set route name
        route.setName(resourceClass.getSimpleName());
        
        // Create upstream configuration from runtime service
        Upstream upstream = createUpstreamFromRuntime();
        route.setUpstream(upstream);
        
        // Add proxy rewrite plugin if enabled
        if (routeAnnotation.enablePathRewrite()) {
            ProxyRewritePlugin proxyRewrite = factory.createProxyRewritePlugin();
            proxyRewrite.setName("proxy-rewrite");
            proxyRewrite.setEnabled(true);
            
            if (routeAnnotation.pathRewritePattern().length == 2) {
                // Custom rewrite pattern
                proxyRewrite.getRegexUri().add(routeAnnotation.pathRewritePattern()[0]);
                proxyRewrite.getRegexUri().add(routeAnnotation.pathRewritePattern()[1]);
            } else {
                // Generate default rewrite pattern
                String internalUri = (applicationBasePath != null ? applicationBasePath : "") + pathAnnotation.value();
                String[] pattern = generateDefaultRewritePattern(externalUri, internalUri);
                proxyRewrite.getRegexUri().add(pattern[0]);
                proxyRewrite.getRegexUri().add(pattern[1]);
            }
            
            route.getPlugins().add(proxyRewrite);
        }
        
        return route;
    }
    
    private String generateRouteId(Class<?> resourceClass) {
        return "auto-" + resourceClass.getSimpleName().toLowerCase();
    }
    
    private String generateExternalUri(String applicationBasePath, String resourcePath) {
        // Construct: applicationBasePath + resourcePath + /*
        // Example: "/rest" + "/v1/users" + "/*" = "/rest/v1/users/*"
        String combinedPath = (applicationBasePath != null ? applicationBasePath : "") + resourcePath;
        
        // Ensure it ends with /* for wildcard matching
        if (!combinedPath.endsWith("/*")) {
            combinedPath = combinedPath + "/*";
        }
        
        // Clean up double slashes
        combinedPath = combinedPath.replaceAll("//+", "/");
        
        return combinedPath;
    }
    
    private String[] generateDefaultRewritePattern(String externalUri, String internalUri) {
        // Convert /api/v1/users/* to ["^/api/v1/users/(.*)$", "/mac/rest/v1/users/$1"]
        String externalBase = externalUri.replace("/*", "");
        String internalBase = internalUri.replace("/*", "");
        
        return new String[] {
            "^" + externalBase + "/(.*)$", 
            internalBase + "/$1"
        };
    }
    
    /**
     * Creates upstream configuration from Jakarta REST Whiteboard runtime service
     */
    private Upstream createUpstreamFromRuntime() {
        ApisixFactory factory = ApisixFactory.eINSTANCE;
        Upstream upstream = factory.createUpstream();
        upstream.setType(LoadBalancerType.ROUNDROBIN);
        
        // Get endpoint information from runtime service
        String[] endpoints = getEndpointsFromRuntime();
        
        for (String endpoint : endpoints) {
            UpstreamNode node = createUpstreamNodeFromEndpoint(factory, endpoint);
            if (node != null) {
                upstream.getNodes().add(node);
            }
        }
        
        // Fallback if no endpoints found
        if (upstream.getNodes().isEmpty()) {
            logger.warn("No endpoints found from runtime service, using fallback configuration");
            UpstreamNode fallbackNode = factory.createUpstreamNode();
            fallbackNode.setHost("backend");
            fallbackNode.setPort(8085);
            fallbackNode.setWeight(1);
            upstream.getNodes().add(fallbackNode);
        }
        
        return upstream;
    }
    
    /**
     * Extracts endpoint URLs from the Jakarta REST runtime service
     */
    private String[] getEndpointsFromRuntime() {
        try {
            Object endpoints = runtimeProperties.get("osgi.jakartars.endpoint");
                
            if (endpoints instanceof String[]) {
                return (String[]) endpoints;
            } else if (endpoints instanceof String) {
                return new String[] { (String) endpoints };
            }
        } catch (Exception e) {
            logger.warn("Failed to get endpoints from runtime service: {}", e.getMessage());
        }
        
        return new String[0];
    }
    
    /**
     * Creates an upstream node from an endpoint URL (e.g., "http://localhost:8085/mac/rest")
     */
    private UpstreamNode createUpstreamNodeFromEndpoint(ApisixFactory factory, String endpoint) {
        try {
            java.net.URI uri = java.net.URI.create(endpoint);
            java.net.URL url = uri.toURL();
            
            UpstreamNode node = factory.createUpstreamNode();
            node.setHost(url.getHost());
            node.setPort(url.getPort() == -1 ? (url.getProtocol().equals("https") ? 443 : 80) : url.getPort());
            node.setWeight(1);
            
            return node;
        } catch (Exception e) {
            logger.warn("Failed to parse endpoint URL '{}': {}", endpoint, e.getMessage());
            return null;
        }
    }
}