/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.apisix.impl;

import java.util.concurrent.TimeUnit;

import org.eclipse.fennec.apisix.ApiSixClient;
import org.eclipse.fennec.model.apisix.Route;
import org.gecko.emf.json.annotation.RequireEMFJson;
import org.gecko.emf.rest.annotations.RequireEMFMessageBodyReaderWriter;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.AttributeDefinition;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.ext.MessageBodyWriter;

/**
 * JAX-RS client for communicating with the APISIX Admin API.
 * Handles route creation, deletion, and management operations using
 * the OSGi Jakarta RS ClientBuilder service.
 * 
 * @author Mark Hoffmann
 * @since 16.09.2025
 */
@RequireEMFJson
@RequireEMFMessageBodyReaderWriter
@Component(name = "APISIXClient", configurationPolicy = ConfigurationPolicy.REQUIRE)
@Designate(ocd = ApiSixClientImpl.Config.class)
public class ApiSixClientImpl implements ApiSixClient {

    private static final Logger logger = LoggerFactory.getLogger(ApiSixClientImpl.class);
    
    @ObjectClassDefinition(name = "APISIX Client Configuration")
    @interface Config {
        @AttributeDefinition(name = "APISIX Admin URL", description = "URL of the APISIX Admin API")
        String adminUrl() default "http://localhost:9180";
        
        @AttributeDefinition(name = "API Key", description = "API Key for APISIX Admin API")
        String apiKey() default "edd1c9f034335f136f87ad84b625c8f1";
        
        @AttributeDefinition(name = "Request Timeout", description = "HTTP request timeout in seconds")
        int timeoutSeconds() default 10;
        
        @AttributeDefinition(name = "Default Upstream Host", description = "Default upstream host for routes")
        String defaultUpstreamHost() default "backend";
        
        @AttributeDefinition(name = "Default Upstream Port", description = "Default upstream port for routes")
        int defaultUpstreamPort() default 8085;
    }
    @Reference(target = "(osgi.jakartars.name=EMFEObjectMessagebodyReaderWriter)")
    MessageBodyWriter<?> emfWriter;
    
    @Reference
    private ClientBuilder clientBuilder;
    
    private Client client;
    private String adminUrl;
    private String apiKey;
    
    @Activate
    void activate(Config config) {
        this.adminUrl = config.adminUrl();
        this.apiKey = config.apiKey();
        
        this.client = clientBuilder
            .connectTimeout(config.timeoutSeconds(), TimeUnit.SECONDS)
            .readTimeout(config.timeoutSeconds(), TimeUnit.SECONDS)
            .register(emfWriter)
            .build();
            
        logger.info("APISIX Client activated - Admin URL: {}", adminUrl);
    }
    
    @Deactivate
    void deactivate() {
        if (client != null) {
            client.close();
        }
        logger.info("APISIX Client deactivated");
    }
    
    /* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.apisix.ApiSixClient#createRoute(org.eclipse.fennec.model.apisix.Route)
	 */
    @Override
	public void createRoute(Route route) {
        try {
            String url = adminUrl + "/apisix/admin/routes/" + route.getId();
            
            try (Response response = client.target(url)
                    .request(MediaType.APPLICATION_JSON)
                    .header("X-API-KEY", apiKey)
                    .put(Entity.json(route))) {
                    
                if (response.getStatus() < 200 || response.getStatus() >= 300) {
                    String errorBody = response.readEntity(String.class);
                    throw new RuntimeException("Failed to create route " + route.getId() + 
                                             ": HTTP " + response.getStatus() + " - " + errorBody);
                }
                
                logger.debug("Created APISIX route {} successfully", route.getId());
            }
        } catch (Exception e) {
            logger.error("Failed to create APISIX route {}: {}", route.getId(), e.getMessage(), e);
            throw new RuntimeException("Route creation failed", e);
        }
    }
    
    /* 
	 * (non-Javadoc)
	 * @see org.eclipse.fennec.apisix.ApiSixClient#deleteRoute(java.lang.String)
	 */
    @Override
	public void deleteRoute(String routeId) {
        String url = adminUrl + "/apisix/admin/routes/" + routeId;
        
        try (Response response = client.target(url)
                .request()
                .header("X-API-KEY", apiKey)
                .delete()) {
                
            if (response.getStatus() < 200 || response.getStatus() >= 300) {
                String errorBody = response.readEntity(String.class);
                throw new RuntimeException("Failed to delete route " + routeId + 
                                         ": HTTP " + response.getStatus() + " - " + errorBody);
            }
            
            logger.debug("Deleted APISIX route {} successfully", routeId);
        } catch (Exception e) {
            logger.error("Failed to delete APISIX route {}: {}", routeId, e.getMessage(), e);
            throw new RuntimeException("Route deletion failed", e);
        }
    }
    
}