/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.jakarta.runtime.whiteboard.api.util;

import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class for safely looking up OSGi services and their properties.
 * 
 * <p>This helper avoids the problems of holding onto ServiceReferences which can
 * become stale when services are unregistered. Instead, it provides fresh lookups
 * on demand.</p>
 * 
 * @author Mark Hoffmann
 * @since 05.10.2025
 */
public class ServiceLookupHelper {
    
    private static final Logger logger = LoggerFactory.getLogger(ServiceLookupHelper.class);
    
    /**
     * Result of a service lookup operation
     */
    public static class ServiceLookupResult {
        private final Object serviceInstance;
        private final Map<String, Object> serviceProperties;
        
        public ServiceLookupResult(Object serviceInstance, Map<String, Object> serviceProperties) {
            this.serviceInstance = serviceInstance;
            this.serviceProperties = serviceProperties != null ? Map.copyOf(serviceProperties) : Map.of();
        }
        
        public Object getServiceInstance() {
            return serviceInstance;
        }
        
        public Map<String, Object> getServiceProperties() {
            return serviceProperties;
        }
        
        public Object getServiceProperty(String key) {
            return serviceProperties.get(key);
        }
        
        public boolean getBooleanServiceProperty(String key, boolean defaultValue) {
            Object value = serviceProperties.get(key);
            if (value instanceof Boolean) {
                return (Boolean) value;
            } else if (value instanceof String) {
                return Boolean.parseBoolean((String) value);
            }
            return defaultValue;
        }
    }
    
    /**
     * Looks up a service by its service ID and returns both the service instance
     * and its properties. This method performs a fresh lookup each time to avoid
     * stale reference issues.
     * 
     * @param bundleContext the bundle context to use for service lookup
     * @param serviceId the service ID to look for
     * @return the lookup result, or empty result if service not found
     */
    public static ServiceLookupResult lookupService(BundleContext bundleContext, long serviceId) {
        if (bundleContext == null) {
            logger.warn("BundleContext is null, cannot lookup service {}", serviceId);
            return new ServiceLookupResult(null, Map.of());
        }
        
        try {
            ServiceReference<?>[] refs = bundleContext.getAllServiceReferences(null, null);
            if (refs == null) {
                logger.debug("No service references found");
                return new ServiceLookupResult(null, Map.of());
            }
            
            for (ServiceReference<?> ref : refs) {
                Object refServiceId = ref.getProperty("service.id");
                if (refServiceId != null && refServiceId.equals(serviceId)) {
                    Object service = bundleContext.getService(ref);
                    
                    // Extract service properties
                    Map<String, Object> properties = new HashMap<>();
                    String[] keys = ref.getPropertyKeys();
                    if (keys != null) {
                        for (String key : keys) {
                            properties.put(key, ref.getProperty(key));
                        }
                    }
                    
                    // Note: We don't unget the service here as it might be used by listeners
                    // This is a design trade-off for simplicity - in a production system,
                    // you might want a more sophisticated service tracking mechanism
                    return new ServiceLookupResult(service, properties);
                }
            }
        } catch (Exception e) {
            logger.warn("Failed to lookup service with ID {}: {}", serviceId, e.getMessage());
        }
        
        return new ServiceLookupResult(null, Map.of());
    }
    
    /**
     * Convenience method to check if a service has a specific boolean property set to true
     * 
     * @param bundleContext the bundle context to use for service lookup
     * @param serviceId the service ID to look for
     * @param propertyKey the property key to check
     * @return true if the property exists and is set to true, false otherwise
     */
    public static boolean hasServiceProperty(BundleContext bundleContext, long serviceId, String propertyKey) {
        ServiceLookupResult result = lookupService(bundleContext, serviceId);
        return result.getBooleanServiceProperty(propertyKey, false);
    }
}