/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.jakarta.runtime.whiteboard.dto;

import java.util.List;
import java.util.Map;

import org.osgi.service.jakartars.runtime.dto.ApplicationDTO;
import org.osgi.service.jakartars.runtime.dto.ResourceDTO;
import org.osgi.service.jakartars.runtime.dto.RuntimeDTO;

/**
 * Event object containing information about Jakarta REST runtime changes.
 * 
 * <p>This event provides access to the current runtime state as well as
 * the specific resources that were added or removed.</p>
 * 
 * @author Mark Hoffmann
 * @since 04.10.2025
 */
public class RuntimeChangeEvent {

    private final RuntimeDTO runtimeDTO;
    private final Map<String, Object> runtimeProperties;
    private final List<ResourceInfo> addedResources;
    private final List<ResourceInfo> removedResources;
    private final List<ResourceInfo> modifiedResources;
    private final long changeCount;

    public RuntimeChangeEvent(RuntimeDTO runtimeDTO, 
                            Map<String, Object> runtimeProperties,
                            List<ResourceInfo> addedResources,
                            List<ResourceInfo> removedResources,
                            List<ResourceInfo> modifiedResources,
                            long changeCount) {
        this.runtimeDTO = runtimeDTO;
        this.runtimeProperties = Map.copyOf(runtimeProperties);
        this.addedResources = List.copyOf(addedResources != null ? addedResources : List.of());
        this.removedResources = List.copyOf(removedResources != null ? removedResources : List.of());
        this.modifiedResources = List.copyOf(modifiedResources != null ? modifiedResources : List.of());
        this.changeCount = changeCount;
    }

    /**
     * @return the current runtime DTO snapshot
     */
    public RuntimeDTO getRuntimeDTO() {
        return runtimeDTO;
    }

    /**
     * @return the runtime service properties
     */
    public Map<String, Object> getRuntimeProperties() {
        return runtimeProperties;
    }

    /**
     * @return resources that were added in this change
     */
    public List<ResourceInfo> getAddedResources() {
        return addedResources;
    }

    /**
     * @return resources that were removed in this change
     */
    public List<ResourceInfo> getRemovedResources() {
        return removedResources;
    }

    /**
     * @return resources that were modified in this change
     */
    public List<ResourceInfo> getModifiedResources() {
        return modifiedResources;
    }

    /**
     * @return the change count from the runtime service
     */
    public long getChangeCount() {
        return changeCount;
    }
    
    /**
     * Extracts endpoint URLs from the Jakarta REST runtime service properties.
     * 
     * @return Array of endpoint URLs, or empty array if none found
     */
    public String[] getEndpoints() {
        try {
            Object endpoints = runtimeProperties.get("osgi.jakartars.endpoint");
                
            if (endpoints instanceof String[]) {
                return (String[]) endpoints;
            } else if (endpoints instanceof String) {
                return new String[] { (String) endpoints };
            }
        } catch (Exception e) {
            // Ignore exceptions and return empty array
        }
        
        return new String[0];
    }
    
    /**
     * Gets the primary server URL from the first endpoint, or returns a default.
     * 
     * @return The primary server URL for OpenAPI server definitions
     */
    public String getPrimaryServerUrl() {
        String[] endpoints = getEndpoints();
        if (endpoints.length > 0) {
            return endpoints[0];
        }
        
        // Fallback to a default server URL
        return "http://localhost:8080";
    }

    /**
     * Information about a specific resource and its context.
     */
    public static class ResourceInfo {
        private final ResourceDTO resourceDTO;
        private final ApplicationDTO applicationDTO;

        public ResourceInfo(ResourceDTO resourceDTO, ApplicationDTO applicationDTO) {
            this.resourceDTO = resourceDTO;
            this.applicationDTO = applicationDTO;
        }

        /**
         * @return the resource DTO
         */
        public ResourceDTO getResourceDTO() {
            return resourceDTO;
        }

        /**
         * @return the application DTO containing this resource
         */
        public ApplicationDTO getApplicationDTO() {
            return applicationDTO;
        }

        /**
         * @return the service ID
         */
        public long getServiceId() {
            return resourceDTO.serviceId;
        }

        /**
         * @return the application base path
         */
        public String getApplicationBasePath() {
            return applicationDTO.base;
        }
        
        /**
         * @return the application name
         */
        public String getApplicationName() {
            return applicationDTO.name;
        }

        /**
         * Checks if this resource has meaningful differences from another resource info.
         * Used to detect modifications beyond just service ID comparison.
         */
        public boolean hasDifferencesFrom(ResourceInfo other) {
            if (other == null || this.resourceDTO.serviceId != other.resourceDTO.serviceId) {
                return true;
            }
            
            // Compare application base paths
            if (!java.util.Objects.equals(this.applicationDTO.base, other.applicationDTO.base)) {
                return true;
            }
            
            // Compare resource DTOs - this is where we'd detect changes in resource configuration
            // For now, we'll do a simple comparison, but this could be enhanced
            return !java.util.Objects.deepEquals(this.resourceDTO, other.resourceDTO);
        }
    }
}