/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.jakarta.runtime.whiteboard.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.fennec.jakarta.runtime.whiteboard.api.JakartaRuntimeChangeListener;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent.ResourceInfo;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.jakartars.runtime.JakartarsServiceRuntime;
import org.osgi.service.jakartars.runtime.dto.ApplicationDTO;
import org.osgi.service.jakartars.runtime.dto.ResourceDTO;
import org.osgi.service.jakartars.runtime.dto.RuntimeDTO;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * OSGi component that tracks Jakarta REST runtime changes and dispatches
 * events to registered listeners using the whiteboard pattern.
 * 
 * <p>This tracker monitors the Jakarta REST runtime service for changes
 * and notifies all registered {@link JakartaRuntimeChangeListener} services
 * about resource additions, removals, and runtime configuration changes.</p>
 * 
 * @author Mark Hoffmann
 * @since 04.10.2025
 */
@Component
@RequireJakartarsWhiteboard
public class JakartaRuntimeWhiteboardTracker {

    private static final Logger logger = LoggerFactory.getLogger(JakartaRuntimeWhiteboardTracker.class);
    
    private final AtomicReference<JakartarsServiceRuntime> jakartarsRuntimeRef = new AtomicReference<>();
    private final Map<String, Object> runtimeProperties = new ConcurrentHashMap<>();
    private final List<JakartaRuntimeChangeListener> listeners = new CopyOnWriteArrayList<>();
    
    private final Map<Long, ResourceInfo> knownResources = new ConcurrentHashMap<>();
    private long lastChangeCount = -1;
    
    @Activate
    public void activate() {
        logger.info("Jakarta REST Runtime Whiteboard Tracker activated");
        processRuntimeChanges();
    }
    
    @Deactivate
    public void deactivate() {
        listeners.clear();
        knownResources.clear();
        logger.info("Jakarta REST Runtime Whiteboard Tracker deactivated");
    }
    
    @Reference(cardinality = ReferenceCardinality.OPTIONAL, policy = ReferencePolicy.DYNAMIC, 
               unbind = "removeRuntimeService", updated = "modifyRuntimeService")
    public void addRuntimeService(JakartarsServiceRuntime jakartarsRuntime, Map<String, Object> properties) {
        jakartarsRuntimeRef.compareAndSet(null, jakartarsRuntime);
        runtimeProperties.putAll(properties);
        processRuntimeChanges();
    }
    
    public void modifyRuntimeService(JakartarsServiceRuntime jakartarsRuntime, Map<String, Object> properties) {
        if (jakartarsRuntime.equals(jakartarsRuntimeRef.get())) {
            runtimeProperties.putAll(properties);
            processRuntimeChanges();
        }
    }
    
    public void removeRuntimeService(JakartarsServiceRuntime jakartarsRuntime) {
        if (jakartarsRuntimeRef.compareAndSet(jakartarsRuntime, null)) {
            notifyRuntimeRemoved();
        }
    }
    
    @Reference(cardinality = ReferenceCardinality.MULTIPLE, policy = ReferencePolicy.DYNAMIC,
               unbind = "removeListener")
    public void addListener(JakartaRuntimeChangeListener listener) {
        listeners.add(listener);
        logger.debug("Added Jakarta REST runtime change listener: {}", listener.getClass().getName());
    }
    
    public void removeListener(JakartaRuntimeChangeListener listener) {
        listeners.remove(listener);
        logger.debug("Removed Jakarta REST runtime change listener: {}", listener.getClass().getName());
    }
    
    /**
     * Processes all resources from the Jakarta REST runtime and dispatches change events
     */
    private void processRuntimeChanges() {
        try {
            JakartarsServiceRuntime runtime = jakartarsRuntimeRef.get();
            if (runtime == null) {
                logger.debug("No Jakarta REST runtime service available yet");
                return;
            }
            
            // Check if change count has actually changed
            Long currentChangeCount = (Long) runtimeProperties.get("service.changecount");
            if (currentChangeCount != null && currentChangeCount.equals(lastChangeCount)) {
                logger.debug("Change count unchanged ({}), skipping processing", currentChangeCount);
                return;
            }
            
            RuntimeDTO runtimeDTO = runtime.getRuntimeDTO();
            if (runtimeDTO == null) {
                logger.warn("No runtime DTO available");
                return;
            }
            
            // Track current and new resources
            Set<Long> currentServiceIds = new HashSet<>();
            List<ResourceInfo> newResources = new ArrayList<>();
            
            // Process all applications
            processApplication(currentServiceIds, newResources, runtimeDTO.defaultApplication);
            for (ApplicationDTO applicationDTO : runtimeDTO.applicationDTOs) {
                processApplication(currentServiceIds, newResources, applicationDTO);
            }
            
            // Determine added, removed, and modified resources
            List<ResourceInfo> addedResources = new ArrayList<>();
            List<ResourceInfo> removedResources = new ArrayList<>();
            List<ResourceInfo> modifiedResources = new ArrayList<>();
            
            // Find newly added and modified resources
            for (ResourceInfo resourceInfo : newResources) {
                ResourceInfo existingResource = knownResources.get(resourceInfo.getServiceId());
                if (existingResource == null) {
                    // Newly added resource
                    addedResources.add(resourceInfo);
                    knownResources.put(resourceInfo.getServiceId(), resourceInfo);
                } else if (resourceInfo.hasDifferencesFrom(existingResource)) {
                    // Modified resource
                    modifiedResources.add(resourceInfo);
                    knownResources.put(resourceInfo.getServiceId(), resourceInfo);
                }
                // If no differences, resource hasn't changed
            }
            
            // Find removed resources
            Set<Long> removedServiceIds = new HashSet<>(knownResources.keySet());
            removedServiceIds.removeAll(currentServiceIds);
            
            for (Long serviceId : removedServiceIds) {
                ResourceInfo resourceInfo = knownResources.remove(serviceId);
                if (resourceInfo != null) {
                    removedResources.add(resourceInfo);
                }
            }
            
            // Dispatch events
            if (!addedResources.isEmpty()) {
                RuntimeChangeEvent event = new RuntimeChangeEvent(runtimeDTO, runtimeProperties, 
                                                                addedResources, null, null, currentChangeCount);
                notifyResourcesAdded(event);
            }
            
            if (!removedResources.isEmpty()) {
                RuntimeChangeEvent event = new RuntimeChangeEvent(runtimeDTO, runtimeProperties, 
                                                                null, removedResources, null, currentChangeCount);
                notifyResourcesRemoved(event);
            }
            
            if (!modifiedResources.isEmpty()) {
                RuntimeChangeEvent event = new RuntimeChangeEvent(runtimeDTO, runtimeProperties, 
                                                                null, null, modifiedResources, currentChangeCount);
                notifyResourcesModified(event);
            }
            
            // Always notify of runtime changes if change count changed
            if (currentChangeCount != null && !currentChangeCount.equals(lastChangeCount)) {
                RuntimeChangeEvent event = new RuntimeChangeEvent(runtimeDTO, runtimeProperties, 
                                                                null, null, null, currentChangeCount);
                notifyRuntimeChanged(event);
            }
            
            lastChangeCount = currentChangeCount;
                
        } catch (Exception e) {
            logger.error("Failed to process Jakarta REST runtime changes: {}", e.getMessage(), e);
        }
    }
    
    /**
     * Processes resources from a single application
     */
    private void processApplication(Set<Long> currentServiceIds, List<ResourceInfo> newResources, 
                                   ApplicationDTO applicationDTO) {
        for (ResourceDTO resourceDTO : applicationDTO.resourceDTOs) {
            currentServiceIds.add(resourceDTO.serviceId);
            ResourceInfo resourceInfo = new ResourceInfo(resourceDTO, applicationDTO);
            newResources.add(resourceInfo);
        }
    }
    
    /**
     * Notifies all listeners about added resources
     */
    private void notifyResourcesAdded(RuntimeChangeEvent event) {
        for (JakartaRuntimeChangeListener listener : listeners) {
            try {
                listener.onResourcesAdded(event);
            } catch (Exception e) {
                logger.error("Error notifying listener {} about added resources: {}", 
                           listener.getClass().getName(), e.getMessage(), e);
            }
        }
    }
    
    /**
     * Notifies all listeners about removed resources
     */
    private void notifyResourcesRemoved(RuntimeChangeEvent event) {
        for (JakartaRuntimeChangeListener listener : listeners) {
            try {
                listener.onResourcesRemoved(event);
            } catch (Exception e) {
                logger.error("Error notifying listener {} about removed resources: {}", 
                           listener.getClass().getName(), e.getMessage(), e);
            }
        }
    }
    
    /**
     * Notifies all listeners about modified resources
     */
    private void notifyResourcesModified(RuntimeChangeEvent event) {
        for (JakartaRuntimeChangeListener listener : listeners) {
            try {
                listener.onResourcesModified(event);
            } catch (Exception e) {
                logger.error("Error notifying listener {} about modified resources: {}", 
                           listener.getClass().getName(), e.getMessage(), e);
            }
        }
    }
    
    /**
     * Notifies all listeners about runtime changes
     */
    private void notifyRuntimeChanged(RuntimeChangeEvent event) {
        for (JakartaRuntimeChangeListener listener : listeners) {
            try {
                listener.onRuntimeChanged(event);
            } catch (Exception e) {
                logger.error("Error notifying listener {} about runtime changes: {}", 
                           listener.getClass().getName(), e.getMessage(), e);
            }
        }
    }
    
    /**
     * Notifies all listeners that the runtime was removed
     */
    private void notifyRuntimeRemoved() {
        List<ResourceInfo> allResources = new ArrayList<>(knownResources.values());
        knownResources.clear();
        
        if (!allResources.isEmpty()) {
            RuntimeChangeEvent event = new RuntimeChangeEvent(null, runtimeProperties, 
                                                            null, allResources, null, lastChangeCount);
            notifyResourcesRemoved(event);
        }
    }
}