/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.openapi.tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.jakartars.runtime.JakartarsServiceRuntime;
import org.osgi.service.jakartars.runtime.dto.ApplicationDTO;
import org.osgi.service.jakartars.runtime.dto.ResourceDTO;
import org.osgi.service.jakartars.runtime.dto.RuntimeDTO;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;
import org.osgi.util.tracker.ServiceTracker;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * Integration tests for OpenAPI generators with Jakarta REST runtime
 */
@RequireJakartarsWhiteboard
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public abstract class BaseOpenApiIntegrationTest {

	@InjectService
	protected ClientBuilder clientBuilder;

	private ServiceRegistration<Object> resourceRegistration;

	@BeforeEach
	public void setup(@InjectBundleContext BundleContext ctx) {
		// Reset any previous state
		cleanup();
	}

	@AfterEach
	public void cleanup() {
		if (resourceRegistration != null) {
			try {
				resourceRegistration.unregister();
			} catch (IllegalStateException e) {
				// Already unregistered, ignore
			}
			resourceRegistration = null;
		}
	}

	@Test
	public void testBasicOpenApiGeneratorIntegration(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create a test JAX-RS resource with OpenAPI enabled
		TestRestResource testResource = new TestRestResource();

		// Register the test resource as a JAX-RS service with openapi=true
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", Boolean.TRUE);
		props.put("osgi.jakartars.name", "openapi-test-basic");
		props.put("openapi", Boolean.TRUE); // Enable OpenAPI generation

		ResourceAware resourceAware = ResourceAware.create(ctx, "openapi-test-basic");
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");
		resourceRegistration = ctx.registerService(Object.class, testResource, props);

		// Wait for resource to be registered in Jakarta REST runtime
		checkResourceAvailability(clientBuilder, resourceAware, "/rest/test/basic/hello", "Hello from OpenAPI test!");

		// Check if basic OpenAPI spec is available
		checkOpenApiSpecAvailability(clientBuilder, "/rest/openapi/spec", "Basic OpenAPI");

		resourceRegistration.unregister();
		resourceRegistration = null;

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
	}

	public void testSwaggerOpenApiGeneratorIntegration(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create a test JAX-RS resource with Swagger annotations
		SwaggerTestResource testResource = new SwaggerTestResource();

		// Register the test resource as a JAX-RS service with openapi=true
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", Boolean.TRUE);
		props.put("osgi.jakartars.name", "openapi-test-swagger");
		props.put("openapi", Boolean.TRUE); // Enable OpenAPI generation

		ResourceAware resourceAware = ResourceAware.create(ctx, "openapi-test-swagger");
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");

		resourceRegistration = ctx.registerService(Object.class, testResource, props);

		// Wait for resource to be registered in Jakarta REST runtime
		checkResourceAvailability(clientBuilder, resourceAware, "/rest/test/swagger/users/123", "User: 123");

		// Check if Swagger OpenAPI spec is available (should have higher ranking)
		checkOpenApiSpecAvailability(clientBuilder, "/rest/openapi/spec", "Swagger Core");

		resourceRegistration.unregister();
		resourceRegistration = null;

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
	}

	@Test
	public void testOpenApiGeneratorWithoutFlag(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create a test JAX-RS resource WITHOUT openapi=true
		TestRestResource testResource = new TestRestResource();

		// Register the test resource as a JAX-RS service WITHOUT openapi flag
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("osgi.jakartars.resource", Boolean.TRUE);
		props.put("osgi.jakartars.name", "openapi-test-disabled");
		// Note: No "openapi" property - should default to false

		ResourceAware resourceAware = ResourceAware.create(ctx, "openapi-test-disabled");
		
		// OpenAPI specs should NOT be available when openapi flag is not set
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");

		resourceRegistration = ctx.registerService(Object.class, testResource, props);

		// Wait for resource to be registered in Jakarta REST runtime
		checkResourceAvailability(clientBuilder, resourceAware, "/rest/test/basic/hello", "Hello from OpenAPI test!");

		// OpenAPI specs should NOT be available when openapi flag is not set
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");

		resourceRegistration.unregister();
		resourceRegistration = null;

		resourceAware.waitForResourceRemoval(5, TimeUnit.SECONDS);
	}

	@Test
	public void testOpenApiSpecContentVerification(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// Create test resources with different paths and operations
		TestRestResource basicResource = new TestRestResource();
		SwaggerTestResource swaggerResource = new SwaggerTestResource();

		// Register basic resource with openapi=true
		Dictionary<String, Object> props1 = new Hashtable<>();
		props1.put("osgi.jakartars.resource", Boolean.TRUE);
		props1.put("osgi.jakartars.name", "content-test-basic");
		props1.put("openapi", Boolean.TRUE);

		ResourceAware resourceAware1 = ResourceAware.create(ctx, "content-test-basic");
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");
		
		ServiceRegistration<Object> registration1 = ctx.registerService(Object.class, basicResource, props1);

		// Wait for resource registration
		checkResourceAvailability(clientBuilder, resourceAware1, "/rest/test/basic/hello", "Hello from OpenAPI test!");

		// Verify OpenAPI spec content contains basic resource paths
		verifyOpenApiSpecContent(clientBuilder, "/rest/openapi/spec", spec -> {
			// Check for basic resource base path
			assertTrue(spec.contains("/test/basic"), 
					"OpenAPI spec should contain /test/basic base path");
			
			// Check for specific operation paths  
			assertTrue(spec.contains("get:/hello") || spec.contains("/hello"), 
					"OpenAPI spec should contain hello operation");
			assertTrue(spec.contains("get:/info") || spec.contains("/info"), 
					"OpenAPI spec should contain info operation");
			
			// Check for HTTP methods
			assertTrue(spec.contains("get:"), 
					"OpenAPI spec should contain GET operations");
			
			// Check for operation IDs or summaries related to our resource
			assertTrue(spec.contains("TestRestResource") || spec.contains("hello") || spec.contains("getInfo"), 
					"OpenAPI spec should reference our test resource methods");
			
			// Check for server URL from runtime endpoint (should contain port 8085 from test configuration)
			assertTrue(spec.contains(":8085") || spec.contains("8085"), 
					"OpenAPI spec should contain actual server URL with port 8085 from runtime endpoint");
			
			// Check for servers section
			assertTrue(spec.contains("servers:") || spec.contains("\"servers\""), 
					"OpenAPI spec should contain servers section");
			
			// Verify no duplicate path segments in server URL
			assertFalse(spec.contains("/rest/rest"), 
					"OpenAPI spec should not contain duplicate /rest path segments in server URL");
		});

		// Register second resource (Swagger) with openapi=true
		Dictionary<String, Object> props2 = new Hashtable<>();
		props2.put("osgi.jakartars.resource", Boolean.TRUE);
		props2.put("osgi.jakartars.name", "content-test-swagger");
		props2.put("openapi", Boolean.TRUE);

		ResourceAware resourceAware2 = ResourceAware.create(ctx, "content-test-swagger");
		ServiceRegistration<Object> registration2 = ctx.registerService(Object.class, swaggerResource, props2);

		// Wait for second resource registration
		checkResourceAvailability(clientBuilder, resourceAware2, "/rest/test/swagger/users/123", "User: 123");

		// Verify OpenAPI spec now contains both resources
		verifyOpenApiSpecContent(clientBuilder, "/rest/openapi/spec", spec -> {
			// Check for both basic and swagger resource base paths
			assertTrue(spec.contains("/test/basic"), 
					"OpenAPI spec should still contain basic resource base path");
			assertTrue(spec.contains("/test/swagger"), 
					"OpenAPI spec should contain swagger resource base path");
			
			// Check for basic resource operations
			assertTrue(spec.contains("get:/hello") || spec.contains("/hello"), 
					"OpenAPI spec should contain hello operation");
			
			// Check for swagger resource operations  
			assertTrue(spec.contains("get:/users") || spec.contains("/users"), 
					"OpenAPI spec should contain users operations");
			
			// Check for path parameters in swagger resource
			assertTrue(spec.contains("{id}") || spec.contains("pathParam") || spec.contains("PathParam"), 
					"OpenAPI spec should contain path parameter definitions");
			
			// Verify server URL is still correct with multiple resources
			assertTrue(spec.contains(":8085") || spec.contains("8085"), 
					"OpenAPI spec should contain actual server URL with port 8085 from runtime endpoint");
		});

		// Remove first resource and verify spec is updated
		registration1.unregister();
		resourceAware1.waitForResourceRemoval(5, TimeUnit.SECONDS);
		Thread.sleep(1000); // Give time for spec regeneration

		// Verify OpenAPI spec no longer contains basic resource
		verifyOpenApiSpecContent(clientBuilder, "/rest/openapi/spec", spec -> {
			// Should not contain basic resource paths anymore
			assertFalse(spec.contains("/test/basic"), 
					"OpenAPI spec should not contain basic resource paths after removal");
			
			// Should still contain swagger resource
			assertTrue(spec.contains("/test/swagger"), 
					"OpenAPI spec should still contain swagger resource paths");
			
			// Server URL should still be correct after resource removal
			assertTrue(spec.contains(":8085") || spec.contains("8085"), 
					"OpenAPI spec should still contain actual server URL with port 8085 after resource removal");
		});

		// Cleanup
		registration2.unregister();
		resourceAware2.waitForResourceRemoval(5, TimeUnit.SECONDS);
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");
	}

	@Test
	public void testOpenApiResourceLifecycle(@InjectBundleContext BundleContext ctx) throws InterruptedException {
		// First, ensure no OpenAPI resources are available
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");

		// Register first resource with openapi=true
		TestRestResource testResource1 = new TestRestResource();
		Dictionary<String, Object> props1 = new Hashtable<>();
		props1.put("osgi.jakartars.resource", Boolean.TRUE);
		props1.put("osgi.jakartars.name", "lifecycle-test-1");
		props1.put("openapi", Boolean.TRUE);

		ResourceAware resourceAware1 = ResourceAware.create(ctx, "lifecycle-test-1");
		ServiceRegistration<Object> registration1 = ctx.registerService(Object.class, testResource1, props1);

		// Wait for resource registration and verify OpenAPI is available
		checkResourceAvailability(clientBuilder, resourceAware1, "/rest/test/basic/hello", "Hello from OpenAPI test!");
		checkOpenApiSpecAvailability(clientBuilder, "/rest/openapi/spec", "Basic OpenAPI");

		// Register second resource with openapi=true to same application
		TestRestResource testResource2 = new TestRestResource();
		Dictionary<String, Object> props2 = new Hashtable<>();
		props2.put("osgi.jakartars.resource", Boolean.TRUE);
		props2.put("osgi.jakartars.name", "lifecycle-test-2");
		props2.put("openapi", Boolean.TRUE);

		ResourceAware resourceAware2 = ResourceAware.create(ctx, "lifecycle-test-2");
		ServiceRegistration<Object> registration2 = ctx.registerService(Object.class, testResource2, props2);

		// Wait for second resource and verify OpenAPI still available
		checkResourceAvailability(clientBuilder, resourceAware2, "/rest/test/basic/hello", "Hello from OpenAPI test!");
		checkOpenApiSpecAvailability(clientBuilder, "/rest/openapi/spec", "Basic OpenAPI");

		// Remove first resource - OpenAPI should still be available
		registration1.unregister();
		resourceAware1.waitForResourceRemoval(5, TimeUnit.SECONDS);
		
		// Give some time for OpenAPI to update
		Thread.sleep(1000);
		checkOpenApiSpecAvailability(clientBuilder, "/rest/openapi/spec", "Basic OpenAPI");

		// Remove second (last) resource - OpenAPI should become unavailable
		registration2.unregister();
		resourceAware2.waitForResourceRemoval(5, TimeUnit.SECONDS);
		
		// Give some time for OpenAPI resource cleanup
		Thread.sleep(2000);
		checkOpenApiSpecNotAvailable(clientBuilder, "/rest/openapi/spec");
	}

	/**
	 * Check if the registered resource endpoint is available
	 */
	private void checkResourceAvailability(ClientBuilder clientBuilder, ResourceAware resourceAware, String path, String expectedContent) throws InterruptedException {
		boolean resourceFound = resourceAware.waitForResource(100, TimeUnit.SECONDS);
		assertTrue(resourceFound, "Resource should be registered in Jakarta REST runtime within 10 seconds");

		// Test that the endpoint is accessible
		Client client = clientBuilder.build();
		WebTarget webTarget = client.target("http://localhost:8085").path(path);
		try {
			checkUrlForResponse(webTarget, 200, expectedContent);
		} finally {
			client.close();
		}
	}

	/**
	 * Check if OpenAPI specification is available
	 */
	private void checkOpenApiSpecAvailability(ClientBuilder clientBuilder, String openApiPath, String expectedGeneratorInfo) {
		Client client = clientBuilder.build();
		WebTarget webTarget = client.target("http://localhost:8085").path(openApiPath);
		try {
			// Give some time for OpenAPI resource to be registered
			Thread.sleep(2000);

			Response response = webTarget.request().get();

			// Should return successful response
			assertTrue(response.getStatus() >= 200 && response.getStatus() < 300, 
					"OpenAPI spec should be available, got status: " + response.getStatus());

			String spec = response.readEntity(String.class);
			assertNotNull(spec);
			assertTrue(spec.length() > 0, "OpenAPI spec should not be empty");

			// Check for generator information in headers
			if (expectedGeneratorInfo != null) {
				String generator = response.getHeaderString("X-Generator");
				if (generator != null) {
					assertTrue(generator.contains(expectedGeneratorInfo), 
							"OpenAPI spec should be generated by " + expectedGeneratorInfo + ", got: " + generator);
				}
			}

			response.close();

		} catch (Exception e) {
			throw new AssertionError("Failed to check OpenAPI spec availability: " + e.getMessage(), e);
		} finally {
			client.close();
		}
	}

	/**
	 * Verify OpenAPI specification content using a custom verifier
	 */
	private void verifyOpenApiSpecContent(ClientBuilder clientBuilder, String openApiPath, 
			java.util.function.Consumer<String> contentVerifier) {
		Client client = clientBuilder.build();
		WebTarget webTarget = client.target("http://localhost:8085").path(openApiPath);
		try {
			// Give some time for OpenAPI resource to be registered
			Thread.sleep(2000);

			Response response = webTarget.request().get();

			// Should return successful response
			assertTrue(response.getStatus() >= 200 && response.getStatus() < 300, 
					"OpenAPI spec should be available, got status: " + response.getStatus());

			String spec = response.readEntity(String.class);
			assertNotNull(spec, "OpenAPI spec should not be null");
			assertTrue(spec.length() > 0, "OpenAPI spec should not be empty");

			// Run custom content verification
			contentVerifier.accept(spec);

			response.close();

		} catch (Exception e) {
			throw new AssertionError("Failed to verify OpenAPI spec content: " + e.getMessage(), e);
		} finally {
			client.close();
		}
	}

	/**
	 * Check that OpenAPI specification is NOT available
	 */
	private void checkOpenApiSpecNotAvailable(ClientBuilder clientBuilder, String openApiPath) {
		Client client = clientBuilder.build();
		WebTarget webTarget = client.target("http://localhost:8085").path(openApiPath);
		try {
			// Give some time for potential OpenAPI resource registration
			Thread.sleep(1000);

			Response response = webTarget.request().get();

			// Should return 404 or error status
			assertTrue(response.getStatus() >= 400, 
					"OpenAPI spec should NOT be available, got status: " + response.getStatus());

			response.close();

		} catch (Exception e) {
			// Expected - endpoint should not be available
		} finally {
			client.close();
		}
	}

	private static void checkUrlForResponse(WebTarget webTarget, int responseCode, String expectedContent) {
		assertNotNull(webTarget);
		assertNotNull(expectedContent);
		try {
			Response response = webTarget.request().get();

			// Verify the endpoint returns the expected response code
			assertEquals(responseCode, response.getStatus(), 
					"Endpoint should return " + responseCode + ", got status: " + response.getStatus());

			// Verify the response content
			String responseText = response.readEntity(String.class);
			assertEquals(expectedContent, responseText);

			response.close();

		} catch (Exception e) {
			throw new AssertionError("Failed to call endpoint: " + e.getMessage(), e);
		}
	}

	// Utility class to track resource registration/removal
	public static class ResourceAware {

		private final ServiceTracker<JakartarsServiceRuntime, JakartarsServiceRuntime> runtimeTracker;
		private final CountDownLatch addLatch;
		private final CountDownLatch removeLatch;

		public static ResourceAware create(BundleContext ctx, String resourceName) {
			assertNotNull(ctx);
			assertNotNull(resourceName);
			return new ResourceAware(ctx, resourceName);
		}

		private ResourceAware(BundleContext ctx, String resourceName) {
			addLatch = new CountDownLatch(1);
			removeLatch = new CountDownLatch(1);
			runtimeTracker = new ServiceTracker<JakartarsServiceRuntime, JakartarsServiceRuntime>(ctx, JakartarsServiceRuntime.class, null) {

				@Override
				public JakartarsServiceRuntime addingService(ServiceReference<JakartarsServiceRuntime> reference) {
					JakartarsServiceRuntime runtime = super.addingService(reference);
					checkForResourceRegistration(runtime, false);
					return runtime;
				}

				@Override
				public void modifiedService(ServiceReference<JakartarsServiceRuntime> reference, JakartarsServiceRuntime service) {
					super.modifiedService(reference, service);
					checkForResourceRegistration(service, false);
				}

				@Override
				public void removedService(ServiceReference<JakartarsServiceRuntime> reference, JakartarsServiceRuntime service) {
					super.removedService(reference, service);
					checkForResourceRegistration(service, true);
				}

				private void checkForResourceRegistration(JakartarsServiceRuntime runtime, boolean removal) {
					if (runtime != null) {
						RuntimeDTO runtimeDTO = runtime.getRuntimeDTO();
						if (runtimeDTO != null && runtimeDTO.applicationDTOs != null) {
							ApplicationDTO appDTO = runtimeDTO.defaultApplication;
							if (appDTO.resourceDTOs != null) {
								boolean found = false;
								for (ResourceDTO resourceDTO : appDTO.resourceDTOs) {
									if (resourceName.equals(resourceDTO.name)) {
										found = true;
										if (!removal) {
											addLatch.countDown();
											return;
										}
									}
								}
								if (removal && !found) {
									removeLatch.countDown();
									return;
								} else if (!removal && found) {
									addLatch.countDown();
									return;
								}
							} else if (removal) {
								removeLatch.countDown();
								return;
							}
						}
					}
				}
			};

			runtimeTracker.open(true);
		}

		boolean waitForResource(long value, TimeUnit unit) throws InterruptedException {
			try {
				return addLatch.await(value, unit);
			} finally {
				runtimeTracker.close();
			}
		}

		boolean waitForResourceRemoval(long value, TimeUnit unit) throws InterruptedException {
			try {
				return removeLatch.await(value, unit);
			} finally {
				runtimeTracker.close();
			}
		}
	}

	// Test JAX-RS resource for basic OpenAPI generation
	@Path("/test/basic")
	public static class TestRestResource {

		@GET
		@Path("/hello")
		@Produces(MediaType.TEXT_PLAIN)
		public Response hello() {
			return Response.ok("Hello from OpenAPI test!").build();
		}

		@GET
		@Path("/info")
		@Produces(MediaType.APPLICATION_JSON)
		public Response getInfo() {
			return Response.ok("{\"service\": \"OpenAPI Test\", \"version\": \"1.0\"}").build();
		}
	}

	// Test JAX-RS resource with Swagger annotations
	@Path("/test/swagger")
	@Tag(name = "Users", description = "User management operations")
	public static class SwaggerTestResource {

		@GET
		@Path("/users/{id}")
		@Produces(MediaType.TEXT_PLAIN)
		@Operation(
			summary = "Get user by ID", 
			description = "Retrieves a user by their unique identifier"
		)
		@ApiResponse(responseCode = "200", description = "User found")
		@ApiResponse(responseCode = "404", description = "User not found")
		public Response getUser(
			@Parameter(description = "User ID", required = true)
			@PathParam("id") String userId
		) {
			return Response.ok("User: " + userId).build();
		}

		@GET
		@Path("/users")
		@Produces(MediaType.APPLICATION_JSON)
		@Operation(
			summary = "List users", 
			description = "Retrieves a list of users with optional filtering"
		)
		public Response listUsers(
			@Parameter(description = "Maximum number of users to return")
			@QueryParam("limit") Integer limit
		) {
			return Response.ok("{\"users\": [], \"total\": 0}").build();
		}
	}
}