/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.openapi.impl;

import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

import org.eclipse.fennec.jakarta.runtime.whiteboard.api.JakartaRuntimeChangeListener;
import org.eclipse.fennec.jakarta.runtime.whiteboard.api.util.ServiceLookupHelper;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent.ResourceInfo;
import org.eclipse.fennec.openapi.config.OpenApiConfig;
import org.eclipse.fennec.openapi.resource.ApplicationOpenApiResource;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.metatype.annotations.Designate;

import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;

/**
 * OpenAPI integration that listens to Jakarta REST runtime changes
 * and generates OpenAPI documentation for discovered resources.
 * 
 * <p>This component acts as a pure change listener that monitors Jakarta REST 
 * applications and generates OpenAPI specifications. It creates separate 
 * JAX-RS resources for serving the OpenAPI specs per application.</p>
 * 
 * <p>The generator can output OpenAPI specs to:</p>
 * <ul>
 * <li>File system (when configured)</li>
 * <li>Per-application REST endpoints via dynamically created resources</li>
 * <li>Logger for debugging</li>
 * </ul>
 * 
 * @author Mark Hoffmann
 * @since 04.10.2025
 */
@Component(name = "BasicOpenApiGenerator",
    configurationPolicy = ConfigurationPolicy.REQUIRE,
    service = JakartaRuntimeChangeListener.class
)
@Designate(ocd = OpenApiConfig.class)
public class OpenApiGenerator implements JakartaRuntimeChangeListener {

    private static final Logger logger = Logger.getLogger(OpenApiGenerator.class.getName());
    
    // Per-application tracking
    private final Map<String, ApplicationInfo> applications = new ConcurrentHashMap<>();
    private final Map<String, ServiceRegistration<Object>> resourceRegistrations = new ConcurrentHashMap<>();
    
    private OpenApiConfig config;
    private BundleContext bundleContext;
    private String currentServerUrl = "http://localhost:8080"; // Default fallback
    
    /**
     * Information about a tracked Jakarta REST application
     */
    private static class ApplicationInfo {
        final String name;
        final String basePath;
        final Map<Long, OpenApiResource> resources = new ConcurrentHashMap<>();
        String currentSpec = "";
        
        ApplicationInfo(String name, String basePath) {
            this.name = name != null ? name : "default";
            this.basePath = basePath != null ? basePath : "";
        }
    }
    
    @Activate
    public void activate(BundleContext bundleContext, OpenApiConfig config) {
        this.bundleContext = bundleContext;
        this.config = config;
        logger.info("OpenAPI Generator activated - monitoring Jakarta REST resources");
        logger.info(String.format("Configuration: file output=%s, path=%s, title=%s", 
                   config.enableFileOutput(), config.outputFilePath(), config.apiTitle()));
    }
    
    @Deactivate
    public void deactivate() {
        // Unregister all OpenAPI resources
        for (ServiceRegistration<Object> registration : resourceRegistrations.values()) {
            try {
                registration.unregister();
            } catch (IllegalStateException e) {
                // Already unregistered, ignore
            }
        }
        resourceRegistrations.clear();
        applications.clear();
        logger.info("OpenAPI Generator deactivated - all application resources unregistered");
    }
    
    @Override
    public void onResourcesAdded(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getAddedResources()) {
            try {
                processNewResource(resourceInfo);
            } catch (Exception e) {
                logger.severe("Failed to process new resource for OpenAPI: " + e.getMessage());
            }
        }
    }
    
    @Override
    public void onResourcesRemoved(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getRemovedResources()) {
            removeResourceFromApplication(resourceInfo);
        }
    }
    
    @Override
    public void onResourcesModified(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getModifiedResources()) {
            try {
                // Re-process the modified resource
                processNewResource(resourceInfo);
            } catch (Exception e) {
                logger.severe("Failed to update modified resource in OpenAPI: " + e.getMessage());
            }
        }
    }
    
    @Override
    public void onRuntimeChanged(RuntimeChangeEvent event) {
        logger.fine("Jakarta REST runtime changed - change count: " + event.getChangeCount());
        
        // Update server URL from runtime endpoint information
        String newServerUrl = event.getPrimaryServerUrl();
        if (!newServerUrl.equals(currentServerUrl)) {
            currentServerUrl = newServerUrl;
            logger.info("Updated OpenAPI server URL to: " + currentServerUrl);
            
            // Regenerate all application specs with new server URL
            for (ApplicationInfo appInfo : applications.values()) {
                generateOpenApiSpecForApplication(appInfo);
            }
        }
    }
    
    
    /**
     * Processes a new resource and extracts OpenAPI information
     */
    private void processNewResource(ResourceInfo resourceInfo) {
        // Use ServiceLookupHelper to get fresh service instance and properties
        ServiceLookupHelper.ServiceLookupResult lookupResult = 
            ServiceLookupHelper.lookupService(bundleContext, resourceInfo.getServiceId());
        
        Object serviceInstance = lookupResult.getServiceInstance();
        if (serviceInstance == null) {
            logger.warning("No service instance available for service ID: " + resourceInfo.getServiceId());
            return;
        }
        
        // Check if OpenAPI generation is enabled for this service
        if (!lookupResult.getBooleanServiceProperty("openapi", false)) {
            logger.fine("OpenAPI generation disabled for service ID: " + resourceInfo.getServiceId());
            return;
        }
        
        Class<?> serviceClass = serviceInstance.getClass();
        
        // Check for Path annotation
        jakarta.ws.rs.Path classPath = serviceClass.getAnnotation(jakarta.ws.rs.Path.class);
        if (classPath == null) {
            logger.fine("Class " + serviceClass.getName() + " has no @Path annotation - skipping OpenAPI generation");
            return;
        }
        
        // Get or create application info
        String appName = resourceInfo.getApplicationName();
        String appBasePath = resourceInfo.getApplicationBasePath();
        String appKey = getApplicationKey(appName, appBasePath);
        
        ApplicationInfo appInfo = applications.computeIfAbsent(appKey, 
            k -> {
                ApplicationInfo info = new ApplicationInfo(appName, appBasePath);
                // Create and register OpenAPI resource for this application
                createOpenApiResourceForApplication(info);
                return info;
            });
        
        // Create resource entry
        OpenApiResource apiResource = new OpenApiResource();
        apiResource.serviceId = resourceInfo.getServiceId();
        apiResource.resourceClass = serviceClass;
        apiResource.basePath = appBasePath;
        apiResource.resourcePath = classPath.value();
        apiResource.operations = extractOperations(serviceClass);
        
        // Add to application
        appInfo.resources.put(resourceInfo.getServiceId(), apiResource);
        
        // Regenerate spec for this application
        generateOpenApiSpecForApplication(appInfo);
        
        logger.info("Added resource to OpenAPI spec for app '" + appName + "': " + serviceClass.getSimpleName() + " (service ID: " + resourceInfo.getServiceId() + ")");
    }
    
    /**
     * Removes a resource from its application and handles cleanup
     */
    private void removeResourceFromApplication(ResourceInfo resourceInfo) {
        String appName = resourceInfo.getApplicationName();
        String appBasePath = resourceInfo.getApplicationBasePath();
        String appKey = getApplicationKey(appName, appBasePath);
        
        ApplicationInfo appInfo = applications.get(appKey);
        if (appInfo != null) {
            // Check if this resource was actually contributing to OpenAPI
            boolean wasTracked = appInfo.resources.containsKey(resourceInfo.getServiceId());
            
            if (wasTracked) {
                appInfo.resources.remove(resourceInfo.getServiceId());
                logger.info("Removed resource from OpenAPI spec for app '" + appName + "': service ID " + resourceInfo.getServiceId());
                
                // If no resources left in application, unregister OpenAPI resource
                if (appInfo.resources.isEmpty()) {
                    unregisterOpenApiResourceForApplication(appKey);
                    applications.remove(appKey);
                    logger.info("Removed OpenAPI resource for empty application: " + appName);
                } else {
                    // Regenerate spec for remaining resources
                    generateOpenApiSpecForApplication(appInfo);
                }
            }
        }
    }
    
    /**
     * Creates a unique key for an application
     */
    private String getApplicationKey(String name, String basePath) {
        return (name != null ? name : "default") + "|" + (basePath != null ? basePath : "");
    }
    
    /**
     * Creates and registers an OpenAPI resource for an application
     */
    private void createOpenApiResourceForApplication(ApplicationInfo appInfo) {
        String appKey = getApplicationKey(appInfo.name, appInfo.basePath);
        
        // Create configuration for this application
        ApplicationOpenApiResource.ApplicationOpenApiConfig resourceConfig = 
            new ApplicationOpenApiResource.ApplicationOpenApiConfig(
                config.apiTitle() + " - " + appInfo.name,
                config.apiVersion(),
                "OpenAPI documentation for " + appInfo.name + " application"
            );
        
        // Create the resource with a spec supplier
        ApplicationOpenApiResource openApiResource = new ApplicationOpenApiResource(
            appInfo.name,
            appInfo.basePath,
            () -> appInfo.currentSpec,
            resourceConfig
        );
        
        // Register as OSGi service with JAX-RS properties
        Hashtable<String, Object> serviceProps = new Hashtable<>();
        serviceProps.put("osgi.jakartars.resource", Boolean.TRUE);
        serviceProps.put("osgi.jakartars.name", "openapi-" + appInfo.name);
        serviceProps.put("osgi.jakartars.application.base", "/openapi/" + appInfo.name);
        
        ServiceRegistration<Object> registration = bundleContext.registerService(Object.class, openApiResource, serviceProps);
        
        resourceRegistrations.put(appKey, registration);
        
        logger.info("Created OpenAPI resource for application '" + appInfo.name + "' at /openapi/" + appInfo.name + "/spec");
    }
    
    /**
     * Unregisters the OpenAPI resource for an application
     */
    private void unregisterOpenApiResourceForApplication(String appKey) {
        ServiceRegistration<Object> registration = resourceRegistrations.remove(appKey);
        if (registration != null) {
            try {
                registration.unregister();
            } catch (IllegalStateException e) {
                // Already unregistered, ignore
            }
        }
    }
    
    /**
     * Generates OpenAPI specification for a specific application
     */
    private void generateOpenApiSpecForApplication(ApplicationInfo appInfo) {
        logger.fine("Generating OpenAPI specification for application '" + appInfo.name + "' with " + appInfo.resources.size() + " resources");
        
        if (appInfo.resources.isEmpty()) {
            appInfo.currentSpec = "";
            return;
        }
        
        StringBuilder specBuilder = new StringBuilder();
        specBuilder.append("openapi: 3.0.0\n");
        specBuilder.append("info:\n");
        specBuilder.append("  title: ").append(config.apiTitle()).append(" - ").append(appInfo.name).append("\n");
        specBuilder.append("  version: ").append(config.apiVersion()).append("\n");
        specBuilder.append("  description: Generated OpenAPI documentation for ").append(appInfo.name).append(" application\n");
        specBuilder.append("servers:\n");
        
        // Use the actual server URL from the runtime
        String serverUrl = currentServerUrl;
        // Note: Don't append application base path since runtime endpoint already includes the full path
        // The application base path is already included in the runtime endpoint URL
        
        specBuilder.append("  - url: ").append(serverUrl).append("\n");
        specBuilder.append("    description: ").append(appInfo.name).append(" application server\n");
        specBuilder.append("paths:\n");
        
        for (OpenApiResource resource : appInfo.resources.values()) {
            specBuilder.append("  ").append(resource.resourcePath).append(":\n");
            
            for (Map.Entry<String, OperationInfo> operation : resource.operations.entrySet()) {
                String httpMethod = operation.getKey().toLowerCase();
                OperationInfo operationInfo = operation.getValue();
                
                specBuilder.append("    ").append(httpMethod).append(":\n");
                specBuilder.append("      summary: ").append(operationInfo.method.getName()).append("\n");
                specBuilder.append("      operationId: ").append(resource.resourceClass.getSimpleName())
                           .append("_").append(operationInfo.method.getName()).append("\n");
                specBuilder.append("      responses:\n");
                specBuilder.append("        '200':\n");
                specBuilder.append("          description: Successful response\n");
            }
        }
        
        String specString = specBuilder.toString();
        appInfo.currentSpec = specString;
        
        // Write to file if configured
        if (config.enableFileOutput()) {
            writeSpecToFile(specString, appInfo.name);
        }
        
        logger.fine("Generated OpenAPI spec for application '" + appInfo.name + "': " + specString.length() + " characters");
    }
    
    /**
     * Extracts HTTP operations from a resource class
     */
    private Map<String, OperationInfo> extractOperations(Class<?> resourceClass) {
        Map<String, OperationInfo> operations = new HashMap<>();
        
        for (Method method : resourceClass.getDeclaredMethods()) {
            OperationInfo operation = null;
            
            if (method.isAnnotationPresent(GET.class)) {
                operation = new OperationInfo("GET", method);
            } else if (method.isAnnotationPresent(POST.class)) {
                operation = new OperationInfo("POST", method);
            } else if (method.isAnnotationPresent(PUT.class)) {
                operation = new OperationInfo("PUT", method);
            } else if (method.isAnnotationPresent(DELETE.class)) {
                operation = new OperationInfo("DELETE", method);
            }
            
            if (operation != null) {
                jakarta.ws.rs.Path methodPath = method.getAnnotation(jakarta.ws.rs.Path.class);
                operation.path = methodPath != null ? methodPath.value() : "";
                
                String operationKey = operation.httpMethod + ":" + operation.path;
                operations.put(operationKey, operation);
            }
        }
        
        return operations;
    }
    
    
    /**
     * Writes the OpenAPI specification to a file
     */
    private void writeSpecToFile(String spec, String applicationName) {
        try {
            // Create application-specific filename
            String basePath = config.outputFilePath();
            String fileName = basePath.contains(".") 
                ? basePath.replaceFirst("\\.", "-" + applicationName + ".")
                : basePath + "-" + applicationName + ".yaml";
            
            java.nio.file.Path outputPath = Paths.get(fileName);
            
            // Create parent directories if they don't exist
            java.nio.file.Path parentDir = outputPath.getParent();
            if (parentDir != null && !Files.exists(parentDir)) {
                Files.createDirectories(parentDir);
            }
            
            Files.write(outputPath, spec.getBytes());
            logger.info("OpenAPI specification for '" + applicationName + "' written to: " + outputPath.toAbsolutePath());
            
        } catch (IOException e) {
            logger.severe("Failed to write OpenAPI specification for '" + applicationName + "' to file: " + e.getMessage());
        }
    }
    
    /**
     * Internal class to hold OpenAPI resource information
     */
    private static class OpenApiResource {
        long serviceId;
        Class<?> resourceClass;
        String basePath;
        String resourcePath;
        Map<String, OperationInfo> operations;
    }
    
    /**
     * Internal class to hold operation information
     */
    private static class OperationInfo {
        String httpMethod;
        Method method;
        String path = "";
        
        OperationInfo(String httpMethod, Method method) {
            this.httpMethod = httpMethod;
            this.method = method;
        }
    }
}