/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.openapi.impl;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.eclipse.fennec.jakarta.runtime.whiteboard.api.JakartaRuntimeChangeListener;
import org.eclipse.fennec.jakarta.runtime.whiteboard.api.util.ServiceLookupHelper;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent;
import org.eclipse.fennec.jakarta.runtime.whiteboard.dto.RuntimeChangeEvent.ResourceInfo;
import org.eclipse.fennec.openapi.config.OpenApiConfig;
import org.eclipse.fennec.openapi.resource.ApplicationOpenApiResource;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.metatype.annotations.Designate;
import java.util.logging.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.swagger.v3.core.util.Json;
import io.swagger.v3.core.util.Yaml;
import io.swagger.v3.jaxrs2.Reader;
import io.swagger.v3.oas.annotations.OpenAPIDefinition;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.servers.Server;

/**
 * Swagger-based OpenAPI integration that listens to Jakarta REST runtime changes
 * and generates professional OpenAPI documentation using Swagger Core libraries.
 * 
 * <p>This implementation leverages Swagger Core for:</p>
 * <ul>
 * <li>Automatic annotation processing (@Operation, @Parameter, @Schema, etc.)</li>
 * <li>Industry-standard OpenAPI 3.0 generation</li>
 * <li>JSON/YAML serialization with Jackson</li>
 * <li>Rich documentation features</li>
 * </ul>
 * 
 * <p>Use this instead of the basic OpenApiGenerator when you need:</p>
 * <ul>
 * <li>Full Swagger annotation support</li>
 * <li>Schema validation</li>
 * <li>Professional-grade documentation</li>
 * </ul>
 * 
 * @author Mark Hoffmann
 * @since 05.10.2025
 */
@Component(name = "SwaggerOpenApiGenerator",
    configurationPolicy = ConfigurationPolicy.REQUIRE,
    service = JakartaRuntimeChangeListener.class,
    property = {
        "generator.type=swagger",
        "service.ranking=100"  // Higher ranking than basic generator
    }
)
@Designate(ocd = OpenApiConfig.class)
public class SwaggerOpenApiGenerator implements JakartaRuntimeChangeListener {

    private static final Logger logger = Logger.getLogger(SwaggerOpenApiGenerator.class.getName());
    
    // Per-application tracking
    private final Map<String, ApplicationInfo> applications = new ConcurrentHashMap<>();
    private final Map<String, ServiceRegistration<Object>> resourceRegistrations = new ConcurrentHashMap<>();
    
    private OpenApiConfig config;
    private BundleContext bundleContext;
    private final ObjectMapper jsonMapper = Json.mapper();
    private final ObjectMapper yamlMapper = Yaml.mapper();
    private String currentServerUrl = "http://localhost:8080"; // Default fallback
    
    /**
     * Information about a tracked Jakarta REST application
     */
    private static class ApplicationInfo {
        final String name;
        final String basePath;
        final Map<Long, SwaggerOpenApiResource> resources = new ConcurrentHashMap<>();
        String currentYamlSpec = "";
        String currentJsonSpec = "";
        
        ApplicationInfo(String name, String basePath) {
            this.name = name != null ? name : "default";
            this.basePath = basePath != null ? basePath : "";
        }
    }
    
    /**
     * Internal class to hold Swagger OpenAPI resource information
     */
    private static class SwaggerOpenApiResource {
        long serviceId;
        Class<?> resourceClass;
        String basePath;
        String resourcePath;
        
        SwaggerOpenApiResource(long serviceId, Class<?> resourceClass, String basePath, String resourcePath) {
            this.serviceId = serviceId;
            this.resourceClass = resourceClass;
            this.basePath = basePath;
            this.resourcePath = resourcePath;
        }
    }
    
    @Activate
    public void activate(BundleContext bundleContext, OpenApiConfig config) {
        this.bundleContext = bundleContext;
        this.config = config;
        logger.info("Swagger OpenAPI Generator activated - monitoring Jakarta REST resources");
        logger.info(String.format("Configuration: file output=%s, path=%s, title=%s", 
                   config.enableFileOutput(), config.outputFilePath(), config.apiTitle()));
    }
    
    @Deactivate
    public void deactivate() {
        // Unregister all OpenAPI resources
        for (ServiceRegistration<Object> registration : resourceRegistrations.values()) {
            try {
                registration.unregister();
            } catch (IllegalStateException e) {
                // Already unregistered, ignore
            }
        }
        resourceRegistrations.clear();
        applications.clear();
        logger.info("Swagger OpenAPI Generator deactivated - all application resources unregistered");
    }
    
    @Override
    public void onResourcesAdded(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getAddedResources()) {
            try {
                processNewResource(resourceInfo);
            } catch (Exception e) {
                logger.severe("Failed to process new resource for Swagger OpenAPI: " + e.getMessage());
            }
        }
    }
    
    @Override
    public void onResourcesRemoved(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getRemovedResources()) {
            removeResourceFromApplication(resourceInfo);
        }
    }
    
    @Override
    public void onResourcesModified(RuntimeChangeEvent event) {
        for (ResourceInfo resourceInfo : event.getModifiedResources()) {
            try {
                // Re-process the modified resource
                processNewResource(resourceInfo);
            } catch (Exception e) {
                logger.severe("Failed to update modified resource in Swagger OpenAPI: " + e.getMessage());
            }
        }
    }
    
    @Override
    public void onRuntimeChanged(RuntimeChangeEvent event) {
        logger.info("Jakarta REST runtime changed - change count: " + event.getChangeCount());
        
        // Debug endpoint extraction
        String[] endpoints = event.getEndpoints();
        logger.info("Runtime endpoints found: " + java.util.Arrays.toString(endpoints));
        
        // Update server URL from runtime endpoint information
        String newServerUrl = event.getPrimaryServerUrl();
        logger.info("Primary server URL from runtime: " + newServerUrl);
        
        if (!newServerUrl.equals(currentServerUrl)) {
            currentServerUrl = newServerUrl;
            logger.info("Updated Swagger OpenAPI server URL from '" + currentServerUrl + "' to: " + newServerUrl);
            
            // Regenerate all application specs with new server URL
            for (ApplicationInfo appInfo : applications.values()) {
                generateSwaggerOpenApiSpecForApplication(appInfo);
            }
        } else {
            logger.info("Server URL unchanged: " + currentServerUrl);
        }
    }
    
    /**
     * Processes a new resource and extracts OpenAPI information using Swagger
     */
    private void processNewResource(ResourceInfo resourceInfo) {
        // Use ServiceLookupHelper to get fresh service instance and properties
        ServiceLookupHelper.ServiceLookupResult lookupResult = 
            ServiceLookupHelper.lookupService(bundleContext, resourceInfo.getServiceId());
        
        Object serviceInstance = lookupResult.getServiceInstance();
        if (serviceInstance == null) {
            logger.warning("No service instance available for service ID: " + resourceInfo.getServiceId());
            return;
        }
        
        // Check if OpenAPI generation is enabled for this service
        if (!lookupResult.getBooleanServiceProperty("openapi", false)) {
            logger.fine("OpenAPI generation disabled for service ID: " + resourceInfo.getServiceId());
            return;
        }
        
        Class<?> serviceClass = serviceInstance.getClass();
        
        // Check for JAX-RS or OpenAPI annotations
        if (!hasRestAnnotations(serviceClass)) {
            logger.fine("Class " + serviceClass.getName() + " has no JAX-RS annotations - skipping OpenAPI generation");
            return;
        }
        
        // Get or create application info
        String appName = resourceInfo.getApplicationName();
        String appBasePath = resourceInfo.getApplicationBasePath();
        String appKey = getApplicationKey(appName, appBasePath);
        
        ApplicationInfo appInfo = applications.computeIfAbsent(appKey, 
            k -> {
                ApplicationInfo info = new ApplicationInfo(appName, appBasePath);
                // Create and register OpenAPI resource for this application
                createOpenApiResourceForApplication(info);
                return info;
            });
        
        // Create resource entry
        jakarta.ws.rs.Path classPath = serviceClass.getAnnotation(jakarta.ws.rs.Path.class);
        String resourcePath = classPath != null ? classPath.value() : "";
        
        SwaggerOpenApiResource apiResource = new SwaggerOpenApiResource(
            resourceInfo.getServiceId(), serviceClass, appBasePath, resourcePath);
        
        // Add to application
        appInfo.resources.put(resourceInfo.getServiceId(), apiResource);
        
        // Regenerate spec for this application using Swagger
        generateSwaggerOpenApiSpecForApplication(appInfo);
        
        logger.info("Added resource to Swagger OpenAPI spec for app '" + appName + "': " + serviceClass.getSimpleName() + " (service ID: " + resourceInfo.getServiceId() + ")");
    }
    
    /**
     * Checks if a class has JAX-RS or OpenAPI annotations
     */
    private boolean hasRestAnnotations(Class<?> clazz) {
        // Check for JAX-RS annotations
        if (clazz.isAnnotationPresent(jakarta.ws.rs.Path.class)) {
            return true;
        }
        
        // Check for OpenAPI annotations
        if (clazz.isAnnotationPresent(OpenAPIDefinition.class)) {
            return true;
        }
        
        // Check methods for JAX-RS annotations
        return java.util.Arrays.stream(clazz.getDeclaredMethods())
            .anyMatch(method -> 
                method.isAnnotationPresent(jakarta.ws.rs.GET.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.POST.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.PUT.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.DELETE.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.PATCH.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.HEAD.class) ||
                method.isAnnotationPresent(jakarta.ws.rs.OPTIONS.class)
            );
    }
    
    /**
     * Removes a resource from its application and handles cleanup
     */
    private void removeResourceFromApplication(ResourceInfo resourceInfo) {
        String appName = resourceInfo.getApplicationName();
        String appBasePath = resourceInfo.getApplicationBasePath();
        String appKey = getApplicationKey(appName, appBasePath);
        
        ApplicationInfo appInfo = applications.get(appKey);
        if (appInfo != null) {
            // Check if this resource was actually contributing to OpenAPI
            boolean wasTracked = appInfo.resources.containsKey(resourceInfo.getServiceId());
            
            if (wasTracked) {
                appInfo.resources.remove(resourceInfo.getServiceId());
                logger.info("Removed resource from Swagger OpenAPI spec for app '" + appName + "': service ID " + resourceInfo.getServiceId());
                
                // If no resources left in application, unregister OpenAPI resource
                if (appInfo.resources.isEmpty()) {
                    unregisterOpenApiResourceForApplication(appKey);
                    applications.remove(appKey);
                    logger.info("Removed Swagger OpenAPI resource for empty application: " + appName);
                } else {
                    // Regenerate spec for remaining resources
                    generateSwaggerOpenApiSpecForApplication(appInfo);
                }
            }
        }
    }
    
    /**
     * Creates a unique key for an application
     */
    private String getApplicationKey(String name, String basePath) {
        return (name != null ? name : "default") + "|" + (basePath != null ? basePath : "");
    }
    
    /**
     * Creates and registers an OpenAPI resource for an application
     */
    private void createOpenApiResourceForApplication(ApplicationInfo appInfo) {
        String appKey = getApplicationKey(appInfo.name, appInfo.basePath);
        
        // Create configuration for this application
        ApplicationOpenApiResource.ApplicationOpenApiConfig resourceConfig = 
            new ApplicationOpenApiResource.ApplicationOpenApiConfig(
                config.apiTitle() + " - " + appInfo.name + " (Swagger)",
                config.apiVersion(),
                "Professional OpenAPI documentation for " + appInfo.name + " application (Generated with Swagger Core)"
            );
        
        // Create the resource with a spec supplier that returns YAML by default
        ApplicationOpenApiResource openApiResource = new ApplicationOpenApiResource(
            appInfo.name,
            appInfo.basePath,
            () -> appInfo.currentYamlSpec,
            resourceConfig
        ) {
            // Override to provide JSON format
            @Override
            public jakarta.ws.rs.core.Response getOpenApiSpecJson() {
                if (appInfo.currentJsonSpec == null || appInfo.currentJsonSpec.isEmpty()) {
                    return jakarta.ws.rs.core.Response.status(404)
                                  .entity("{\"error\": \"No OpenAPI specification available for application: " + appInfo.name + "\"}")
                                  .build();
                }
                
                return jakarta.ws.rs.core.Response.ok(appInfo.currentJsonSpec)
                                  .header("Content-Type", jakarta.ws.rs.core.MediaType.APPLICATION_JSON)
                                  .header("X-Application-Name", appInfo.name)
                                  .header("X-Application-Base-Path", appInfo.basePath)
                                  .header("X-Generator", "Swagger Core")
                                  .build();
            }
        };
        
        // Register as OSGi service with JAX-RS properties
        Hashtable<String, Object> serviceProps = new Hashtable<>();
        serviceProps.put("osgi.jakartars.resource", Boolean.TRUE);
        serviceProps.put("osgi.jakartars.name", "swagger-openapi-" + appInfo.name);
        serviceProps.put("osgi.jakartars.application.base", "/swagger-openapi/" + appInfo.name);
        
        ServiceRegistration<Object> registration = bundleContext.registerService(Object.class, openApiResource, serviceProps);
        
        resourceRegistrations.put(appKey, registration);
        
        logger.info("Created Swagger OpenAPI resource for application '" + appInfo.name + "' at /swagger-openapi/" + appInfo.name + "/spec");
    }
    
    /**
     * Unregisters the OpenAPI resource for an application
     */
    private void unregisterOpenApiResourceForApplication(String appKey) {
        ServiceRegistration<Object> registration = resourceRegistrations.remove(appKey);
        if (registration != null) {
            try {
                registration.unregister();
            } catch (IllegalStateException e) {
                // Already unregistered, ignore
            }
        }
    }
    
    /**
     * Generates OpenAPI specification for a specific application using Swagger Core
     */
    private void generateSwaggerOpenApiSpecForApplication(ApplicationInfo appInfo) {
        logger.fine("Generating Swagger OpenAPI specification for application '" + appInfo.name + "' with " + appInfo.resources.size() + " resources");
        
        if (appInfo.resources.isEmpty()) {
            appInfo.currentYamlSpec = "";
            appInfo.currentJsonSpec = "";
            return;
        }
        
        try {
            // Create OpenAPI base configuration
            OpenAPI openAPI = new OpenAPI();
            
            // Set info
            io.swagger.v3.oas.models.info.Info info = new io.swagger.v3.oas.models.info.Info()
                .title(config.apiTitle() + " - " + appInfo.name)
                .version(config.apiVersion())
                .description("Professional OpenAPI documentation for " + appInfo.name + " application (Generated with Swagger Core)");
            
            // Add contact if available
            Contact contact = new Contact()
                .name("Generated by Eclipse Fennec OpenAPI")
                .url("https://github.com/eclipse-fennec/openapi");
            info.setContact(contact);
            
            openAPI.setInfo(info);
            
            // Add server with actual runtime URL
            String serverUrl = currentServerUrl;
            logger.info("Using server URL for OpenAPI spec: " + serverUrl + " (for app: " + appInfo.name + ")");
            // Note: Don't append application base path since runtime endpoint already includes the full path
            // The application base path is already included in the runtime endpoint URL
            
            Server server = new Server()
                .url(serverUrl)
                .description(appInfo.name + " application server");
            openAPI.addServersItem(server);
            
            // Use Swagger Reader to scan the resource classes
            Reader reader = new Reader(openAPI);
            Set<Class<?>> classes = new HashSet<>();
            for (SwaggerOpenApiResource resource : appInfo.resources.values()) {
                classes.add(resource.resourceClass);
            }
            
            OpenAPI generatedOpenAPI = reader.read(classes);
            
            // Generate YAML and JSON
            String yamlSpec = yamlMapper.writeValueAsString(generatedOpenAPI);
            String jsonSpec = jsonMapper.writeValueAsString(generatedOpenAPI);
            
            appInfo.currentYamlSpec = yamlSpec;
            appInfo.currentJsonSpec = jsonSpec;
            
            // Write to file if configured
            if (config.enableFileOutput()) {
                writeSpecToFile(yamlSpec, appInfo.name);
            }
            
            logger.fine("Generated Swagger OpenAPI spec for application '" + appInfo.name + "': " + yamlSpec.length() + " characters (YAML), " + jsonSpec.length() + " characters (JSON)");
            
        } catch (Exception e) {
            logger.severe("Failed to generate Swagger OpenAPI specification for application '" + appInfo.name + "': " + e.getMessage());
            
            // Fallback to empty specs
            appInfo.currentYamlSpec = "# Error generating OpenAPI spec: " + e.getMessage();
            appInfo.currentJsonSpec = "{\"error\": \"Failed to generate OpenAPI spec: " + e.getMessage() + "\"}";
        }
    }
    
    /**
     * Writes the OpenAPI specification to a file
     */
    private void writeSpecToFile(String spec, String applicationName) {
        try {
            // Create application-specific filename with swagger prefix
            String basePath = config.outputFilePath();
            String fileName = basePath.contains(".") 
                ? basePath.replaceFirst("\\.", "-swagger-" + applicationName + ".")
                : basePath + "-swagger-" + applicationName + ".yaml";
            
            java.nio.file.Path outputPath = Paths.get(fileName);
            
            // Create parent directories if they don't exist
            java.nio.file.Path parentDir = outputPath.getParent();
            if (parentDir != null && !Files.exists(parentDir)) {
                Files.createDirectories(parentDir);
            }
            
            Files.write(outputPath, spec.getBytes());
            logger.info("Swagger OpenAPI specification for '" + applicationName + "' written to: " + outputPath.toAbsolutePath());
            
        } catch (IOException e) {
            logger.severe("Failed to write Swagger OpenAPI specification for '" + applicationName + "' to file: " + e.getMessage());
        }
    }
}