/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Mark Hoffmann - initial API and implementation
 */
package org.eclipse.fennec.openapi.resource;

import java.util.function.Supplier;

import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * JAX-RS resource for serving OpenAPI specifications for a specific Jakarta REST application.
 * 
 * <p>This resource is dynamically created for each application that has OpenAPI generation
 * enabled. Each instance serves the OpenAPI spec at application-specific endpoints.</p>
 * 
 * <p>Endpoints provided:</p>
 * <ul>
 * <li>GET /openapi/{appName}/spec - OpenAPI spec in YAML format</li>
 * <li>GET /openapi/{appName}/spec.yaml - OpenAPI spec in YAML format</li>
 * <li>GET /openapi/{appName}/spec.json - OpenAPI spec in JSON format</li>
 * </ul>
 * 
 * @author Mark Hoffmann
 * @since 05.10.2025
 */
@Path("/openapi")
public class ApplicationOpenApiResource {
    
    private final String applicationName;
    private final String applicationBasePath;
    private final Supplier<String> specSupplier;
    private final ApplicationOpenApiConfig config;
    
    /**
     * Configuration for an application's OpenAPI resource
     */
    public static class ApplicationOpenApiConfig {
        private final String title;
        private final String version;
        private final String description;
        
        public ApplicationOpenApiConfig(String title, String version, String description) {
            this.title = title != null ? title : "API Documentation";
            this.version = version != null ? version : "1.0.0";
            this.description = description != null ? description : "Generated API documentation";
        }
        
        public String getTitle() { return title; }
        public String getVersion() { return version; }
        public String getDescription() { return description; }
    }
    
    /**
     * Creates a new OpenAPI resource for a specific application
     * 
     * @param applicationName the name of the Jakarta REST application
     * @param applicationBasePath the base path of the application
     * @param specSupplier supplier that provides the current OpenAPI spec
     * @param config configuration for this application's OpenAPI
     */
    public ApplicationOpenApiResource(String applicationName, String applicationBasePath, 
                                    Supplier<String> specSupplier, ApplicationOpenApiConfig config) {
        this.applicationName = applicationName != null ? applicationName : "default";
        this.applicationBasePath = applicationBasePath != null ? applicationBasePath : "";
        this.specSupplier = specSupplier;
        this.config = config;
    }
    
    /**
     * Serves the OpenAPI specification in YAML format
     */
    @GET
    @Path("/spec")
    @Produces({"application/x-yaml", "text/yaml"})
    public Response getOpenApiSpec() {
        String spec = specSupplier.get();
        if (spec == null || spec.isEmpty()) {
            return Response.status(404)
                          .entity("No OpenAPI specification available for application: " + applicationName)
                          .build();
        }
        
        return Response.ok(spec)
                      .header("Content-Type", "application/x-yaml")
                      .header("X-Application-Name", applicationName)
                      .header("X-Application-Base-Path", applicationBasePath)
                      .build();
    }
    
    /**
     * Serves the OpenAPI specification in YAML format (explicit)
     */
    @GET
    @Path("/spec.yaml") 
    @Produces({"application/x-yaml", "text/yaml"})
    public Response getOpenApiSpecYaml() {
        return getOpenApiSpec();
    }
    
    /**
     * Serves the OpenAPI specification in JSON format
     */
    @GET
    @Path("/spec.json")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getOpenApiSpecJson() {
        String spec = specSupplier.get();
        if (spec == null || spec.isEmpty()) {
            return Response.status(404)
                          .entity("{\"error\": \"No OpenAPI specification available for application: " + applicationName + "\"}")
                          .build();
        }
        
        // In a real implementation, you'd convert YAML to JSON here
        // For simplicity, we'll return the YAML format with JSON content type
        return Response.ok(spec)
                      .header("Content-Type", MediaType.APPLICATION_JSON)
                      .header("X-Application-Name", applicationName)
                      .header("X-Application-Base-Path", applicationBasePath)
                      .build();
    }
    
    /**
     * Gets information about this OpenAPI resource
     */
    @GET
    @Path("/info")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getInfo() {
        String info = String.format(
            "{\"application\":\"%s\",\"basePath\":\"%s\",\"title\":\"%s\",\"version\":\"%s\",\"description\":\"%s\"}",
            applicationName, applicationBasePath, config.getTitle(), config.getVersion(), config.getDescription()
        );
        
        return Response.ok(info)
                      .header("Content-Type", MediaType.APPLICATION_JSON)
                      .build();
    }
    
    /**
     * @return the application name this resource serves
     */
    public String getApplicationName() {
        return applicationName;
    }
    
    /**
     * @return the application base path
     */
    public String getApplicationBasePath() {
        return applicationBasePath;
    }
}