/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.emf.sensinact.model.impl;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static java.util.Objects.requireNonNull;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.eclipse.emf.common.util.EList;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.sensinact.core.twin.SensinactDigitalTwin;
import org.eclipse.sensinact.core.twin.SensinactProvider;
import org.eclipse.sensinact.core.twin.SensinactResource;
import org.eclipse.sensinact.core.twin.SensinactService;
import org.eclipse.sensinact.gateway.geojson.Coordinates;
import org.eclipse.sensinact.gateway.geojson.GeoJsonObject;
import org.eclipse.sensinact.gateway.geojson.Point;
import org.eclipse.sensinact.mapping.AdminMapping;
import org.eclipse.sensinact.mapping.MappingProfile;
import org.eclipse.sensinact.mapping.NameMapping;
import org.eclipse.sensinact.mapping.ProviderMapping;
import org.eclipse.sensinact.mapping.ProviderStrategy;
import org.eclipse.sensinact.mapping.ResourceMapping;
import org.eclipse.sensinact.mapping.ServiceMapping;
import org.eclipse.sensinact.mapping.TimestampMapping;
import org.gecko.emf.sensinact.model.ValueMapper;
import org.gecko.emf.sensinact.model.ValueMappingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of {@link ValueMapper} that transforms EObject instances into SensiNact provider values
 * using ProviderMapping configurations.
 * 
 * @author Mark Hoffmann
 * @since 31.07.2025
 */
public class ValueMapperImpl implements ValueMapper {

	private static final Logger logger = LoggerFactory.getLogger(ValueMapperImpl.class);

	private final SensinactDigitalTwin twin;
	private final ProviderMapping mapping;
	private final String providerModel;

	/**
	 * Creates a new ValueMapper instance configured for a specific provider mapping and digital twin.
	 * 
	 * @param twin The SensiNact digital twin to update with mapped values
	 * @param mapping The ProviderMapping configuration defining the transformation rules
	 */
	public ValueMapperImpl(SensinactDigitalTwin twin, ProviderMapping mapping) {
		this.twin = requireNonNull(twin, "Digital twin cannot be null");
		this.mapping = requireNonNull(mapping, "Provider mapping cannot be null");
		this.providerModel = determineProviderModel(mapping);
	}

	@Override
	public void mapInstance(EObject sourceInstance) throws ValueMappingException {
		requireNonNull(sourceInstance, "Source instance cannot be null");

		ValidationResult validation = validateInstance(sourceInstance);
		if (!validation.isValid()) {
			throw new ValueMappingException("Instance validation failed: " + validation.getErrors());
		}

		// Determine provider-level timestamp once
		Instant providerTimestamp = determineTimestamp(sourceInstance, null);
		String providerId = determineProviderId(sourceInstance);
		try {
			SensinactProvider provider = twin.getProvider(providerModel, providerId);
			if (provider == null) {
				provider = twin.createProvider(providerModel, providerId, providerTimestamp);
			}


			for (ServiceMapping serviceMapping : mapping.getServices()) {
				// Service timestamp takes precedence, fallback to provider timestamp
				Instant serviceTimestamp = determineTimestamp(sourceInstance, serviceMapping, providerTimestamp);
				mapServiceResources(sourceInstance, serviceMapping, provider, serviceTimestamp);
			}

			// Map admin service if configured
			if (mapping.getAdmin() != null) {
				// Admin timestamp takes precedence, fallback to provider timestamp
				Instant adminTimestamp = determineTimestamp(sourceInstance, mapping.getAdmin(), providerTimestamp);
				mapAdminService(sourceInstance, mapping.getAdmin(), provider, adminTimestamp);
			}

			logger.debug("Successfully mapped instance {} to provider {}", 
					sourceInstance.eClass().getName(), providerModel);

		} catch (Exception e) {
			throw new ValueMappingException("Failed to map instance to provider " + providerModel, e);
		}
	}

	@Override
	public Map<String, Object> mapResourceValues(EObject sourceInstance) {
		requireNonNull(sourceInstance, "Source instance cannot be null");

		Map<String, Object> values = new LinkedHashMap<>();

		for (ServiceMapping serviceMapping : mapping.getServices()) {
			for (ResourceMapping resourceMapping : serviceMapping.getResources()) {
				String resourcePath = serviceMapping.getMid() + "." + resourceMapping.getMid();

				try {
					Optional<Object> value = extractValue(sourceInstance, resourceMapping);
					if (value.isPresent()) {
						values.put(resourcePath, value.get());
					}
				} catch (Exception e) {
					logger.warn("Failed to extract value for resource path {}: {}", resourcePath, e.getMessage());
				}
			}
		}

		return values;
	}

	@Override
	public ValidationResult validateInstance(EObject sourceInstance) {
		requireNonNull(sourceInstance, "Source instance cannot be null");

		ValidationResultImpl result = new ValidationResultImpl();

		// Check if instance type matches any of the expected provider classes
		EClass instanceClass = sourceInstance.eClass();
		boolean typeMatches = mapping.getProviderClasses().stream()
				.anyMatch(providerClass -> providerClass.isSuperTypeOf(instanceClass) || 
						providerClass.equals(instanceClass));

		if (!typeMatches) {
			result.addError("Instance type " + instanceClass.getName() + 
					" does not match any expected provider classes");
		}

		// Validate feature paths for all resources
		for (ServiceMapping serviceMapping : mapping.getServices()) {
			for (ResourceMapping resourceMapping : serviceMapping.getResources()) {
				validateResourceMapping(sourceInstance, resourceMapping, serviceMapping.getMid(), result);
			}
		}

		return result;
	}

	/**
	 * Extracts a value from an EObject instance using a ResourceMapping configuration.
	 */
	private Optional<Object> extractValue(EObject sourceInstance, ResourceMapping resourceMapping) {
		List<EStructuralFeature> featurePath = resourceMapping.getValueFeature();
		Optional<Object> rawValue = getRawValue(sourceInstance, featurePath);

		if (!rawValue.isPresent()) {
			return Optional.empty();
		}

		// Convert the raw value using the resource's type information
		EDataType targetType = resourceMapping.getEAttributeType();
		if (targetType != null) {
			try {
				Object convertedValue = convertValue(rawValue.get(), targetType);
				return Optional.of(convertedValue);
			} catch (Exception e) {
				logger.warn("Failed to convert value {} to type {}: {}", 
						rawValue.get(), targetType.getName(), e.getMessage());
				return Optional.empty();
			}
		}

		return rawValue;
	}

	/**
	 * Maps all resources for a specific service.
	 */
	private void mapServiceResources(EObject sourceInstance, ServiceMapping serviceMapping, 
			SensinactProvider provider, Instant timestamp) throws ValueMappingException {

		SensinactService service = provider.getServices().get(serviceMapping.getMid());
		if (service == null) {
			throw new ValueMappingException("Service not found: " + serviceMapping.getMid());
		}

		for (ResourceMapping resourceMapping : serviceMapping.getResources()) {
			mapSingleResource(sourceInstance, resourceMapping, service, timestamp);
		}
	}

	/**
	 * Maps a single resource value.
	 */
	private void mapSingleResource(EObject sourceInstance, ResourceMapping resourceMapping, 
			SensinactService service, Instant timestamp) throws ValueMappingException {

		SensinactResource resource = service.getResources().get(resourceMapping.getMid());
		if (resource == null) {
			throw new ValueMappingException("Resource not found: " + resourceMapping.getMid());
		}

		Optional<Object> value = extractValue(sourceInstance, resourceMapping);
		if (value.isPresent()) {
			try {
				resource.setValue(value.get(), timestamp);
				logger.trace("Set resource {}.{} = {}", service.getName(), resource.getName(), value.get());
			} catch (Exception e) {
				throw new ValueMappingException("Failed to set resource value for " + 
						service.getName() + "." + resource.getName(), e);
			}
		} else {
			logger.debug("No value extracted for resource {}.{}", service.getName(), resource.getName());
		}
	}

	/**
	 * Maps admin service values.
	 */
	private void mapAdminService(EObject sourceInstance, AdminMapping adminMapping, 
			SensinactProvider provider, Instant timestamp) throws ValueMappingException {

		SensinactService adminService = provider.getServices().get("admin");
		if (adminService == null) {
			throw new ValueMappingException("Admin service not found");
		}
		mapFriendlyName(sourceInstance, adminMapping, timestamp, adminService);
		mapLocation(sourceInstance, adminMapping, timestamp, adminService);
	}

	/**
	 * @param sourceInstance
	 * @param adminMapping
	 * @param timestamp
	 * @param adminService
	 * @throws ValueMappingException
	 */
	private void mapFriendlyName(EObject sourceInstance, AdminMapping adminMapping, Instant timestamp,
			SensinactService adminService) throws ValueMappingException {
		String friendlyName = adminMapping.getFriendlyName();
		SensinactResource friendlyNameResource = adminService.getResources().get("friendlyName");
		requireNonNull(friendlyNameResource, "No friendly name resource available in admin service");
		// Map friendlyName resource if configured
		if (!adminMapping.getFriendlyNameFeature().isEmpty()) {
			List<EStructuralFeature> featurePath = adminMapping.getFriendlyNameFeature();
			Optional<Object> friendlyNameValue = getRawValue(sourceInstance, featurePath);
			if (friendlyNameValue.isPresent()) {
				friendlyName = (String) friendlyNameValue.get();
			} else {
				logger.debug("No value extracted for admin.friendlyName");
			}
		}
		if (Objects.nonNull(friendlyName)) {
			try {
				friendlyNameResource.setValue(friendlyName, timestamp);
				logger.trace("Set admin.friendlyName = {}", adminMapping.getFriendlyName());
			} catch (Exception e) {
				logger.warn("Failed to set admin friendlyName: {}", e.getMessage());
			}
		}
	}

	/**
	 * @param sourceInstance
	 * @param adminMapping
	 * @param timestamp
	 * @param adminService
	 * @throws ValueMappingException
	 */
	private void mapLocation(EObject sourceInstance, AdminMapping adminMapping, Instant timestamp,
			SensinactService adminService) throws ValueMappingException {
		// Map friendlyName resource if configured
		GeoJsonObject location = lookupLocation(adminMapping);
		SensinactResource locationResource = adminService.getResources().get("location");
		requireNonNull(locationResource, "No location resource available in admin service");
		if (!adminMapping.getLatitudeRef().isEmpty() && 
				!adminMapping.getLongitudeRef().isEmpty()) {
			List<EStructuralFeature> latFeaturePath = adminMapping.getLatitudeRef();
			List<EStructuralFeature> lonFeaturePath = adminMapping.getLongitudeRef();
			Optional<Object> latValue = getRawValue(sourceInstance, latFeaturePath);
			Optional<Object> lonValue = getRawValue(sourceInstance, lonFeaturePath);

			if (latValue.isPresent() && lonValue.isPresent()) {
				try {
					Coordinates coordinates = new Coordinates(toDouble(lonValue.get()), toDouble(latValue.get()));
					final Point point = new Point(coordinates, Collections.emptyList(), Collections.emptyMap());
					location = point;
					logger.trace("Set admin.location = {}", point);
					return;
				} catch (Exception e) {
					throw new ValueMappingException("Failed to set admin location resource", e);
				}
			} else {
				logger.debug("No value extracted for admin.location");
			}
		}
		if (locationResource != null) {
			try {
				locationResource.setValue(location, timestamp);
				logger.trace("Set admin.location = {}", location);
			} catch (Exception e) {
				logger.warn("Failed to set admin location: {}", e.getMessage());
			}
		}
	}

	private String determineProviderId(EObject sourceInstance) throws ValueMappingException {
		NameMapping nameMapping = mapping.getName();
		return getNameMapping(nameMapping, sourceInstance);
	}

	/**
	 * @param nameMapping
	 * @param sourceInstance
	 * @return
	 * @throws ValueMappingException 
	 */
	private String getNameMapping(NameMapping nameMapping, EObject sourceInstance) throws ValueMappingException {
		if (nameMapping == null) {
			throw new ValueMappingException("A mapping is needed to determine a name from an instance");
		}

		// First try: Extract value using the name mapping's feature path
		List<EStructuralFeature> featurePath = nameMapping.getFeaturePath();
		if (featurePath != null && !featurePath.isEmpty()) {
			Optional<Object> rawValue = getRawValue(sourceInstance, featurePath);
			if (rawValue.isPresent() && rawValue.get() != null) {
				return rawValue.get().toString();
			}
		}

		// Second try: Use the name property as fallback
		if (nameMapping.getName() != null) {
			return nameMapping.getName().toString();
		}

		// Final fallback: use mapping's MID + instance hash
		String fallback = mapping.getMid() + "-" + Integer.toHexString(sourceInstance.hashCode());
		logger.debug("Name mapping extraction failed, using fallback provider ID: {}", fallback);
		return fallback;
	}

	/**
	 * Determines the timestamp for provider-level mapping.
	 */
	private Instant determineTimestamp(EObject sourceInstance, ServiceMapping serviceMapping) {
		return determineTimestamp(sourceInstance, serviceMapping, null);
	}

	/**
	 * Determines the timestamp with proper precedence handling.
	 * 
	 * @param sourceInstance The source EObject instance
	 * @param serviceMapping The service mapping (null for provider-level)  
	 * @param providerTimestamp The provider timestamp to use as fallback (can be null)
	 * @return The determined timestamp
	 */
	private Instant determineTimestamp(EObject sourceInstance, ServiceMapping serviceMapping, Instant providerTimestamp) {
		// 1. Service timestamp takes precedence if available
		if (serviceMapping != null && serviceMapping.getTimestamp() != null) {
			Instant serviceTimestamp = extractTimestamp(sourceInstance, serviceMapping.getTimestamp());
			if (serviceTimestamp != null) {
				return serviceTimestamp;
			}
		}

		// 2. If no service timestamp but provider timestamp exists, use provider timestamp
		if (providerTimestamp != null) {
			return providerTimestamp;
		}

		// 3. Try provider-level timestamp if not already determined
		if (mapping.getTimestamp() != null) {
			Instant extractedProviderTimestamp = extractTimestamp(sourceInstance, mapping.getTimestamp());
			if (extractedProviderTimestamp != null) {
				return extractedProviderTimestamp;
			}
		}

		// 4. Final fallback: current time
		return Instant.now();
	}

	/**
	 * Extracts timestamp from source instance using TimestampMapping configuration.
	 * Similar to getNameMapping logic - first try feature path, then static timestamp value.
	 */
	private Instant extractTimestamp(EObject sourceInstance, TimestampMapping timestampMapping) {
		if (timestampMapping == null) {
			return null;
		}

		// First try: Extract value using the timestamp mapping's feature path
		List<EStructuralFeature> featurePath = timestampMapping.getFeaturePath();
		if (featurePath != null && !featurePath.isEmpty()) {
			Optional<Object> rawValue = getRawValue(sourceInstance, featurePath);
			if (rawValue.isPresent() && rawValue.get() != null) {
				return convertToInstant(rawValue.get(), timestampMapping.getHint());
			}
		}

		// Second try: Use the static timestamp value
		if (timestampMapping.getTimestamp() != null) {
			return timestampMapping.getTimestamp();
		}

		return null;
	}

	/**
	 * Converts a raw timestamp value to Instant, using optional hint for format patterns.
	 */
	private Instant convertToInstant(Object rawValue, String hint) {
		if (rawValue instanceof Instant) {
			return (Instant) rawValue;
		}
		if (rawValue instanceof Date) {
			return ((Date) rawValue).toInstant();
		}

		if (rawValue instanceof Long) {
			// Assume milliseconds since epoch
			return Instant.ofEpochMilli((Long) rawValue);
		}

		if (rawValue instanceof String) {
			String stringValue = (String) rawValue;
			boolean useHint = nonNull(hint) && !hint.trim().isEmpty();
			try {
				// If hint is provided, use it as DateTimeFormatter pattern first
				if (useHint) {
					DateTimeFormatter formatter = DateTimeFormatter.ofPattern(hint.trim());
					LocalDateTime localDateTime = LocalDateTime.parse(stringValue, formatter);
					return localDateTime.toInstant(ZoneOffset.UTC);
				} else {
					// Try parsing as ISO instant
					return Instant.parse(stringValue);
				}
			} catch (Exception e) {
				// If hint parsing failed, try ISO parsing as fallback
				if (useHint) {
					try {
						return Instant.parse(stringValue);
					} catch (Exception isoException) {
						logger.warn("Failed to parse timestamp string '{}' using pattern '{}' and as ISO instant: {}", 
								stringValue, hint, e.getMessage());
					}
				} else {
					logger.warn("Failed to parse timestamp string '{}' as ISO instant: {}", 
							stringValue, e.getMessage());
				}
			}
		}

		logger.warn("Cannot convert timestamp value of type {} to Instant: {}", 
				rawValue.getClass().getName(), rawValue);
		return null;
	}

	/**
	 * Determines the provider ID based on the mapping strategy.
	 */
	private String determineProviderModel(ProviderMapping providerMapping) {
		MappingProfile profile = providerMapping.getProfile();
		if (profile != null && profile.getProviderStrategy() == ProviderStrategy.UNIFIED) {
			return profile.getProvider().getProviderId();
		} else {
			return providerMapping.getFqMid();
		}
	}

	/**
	 * Converts a raw value to the target EDataType.
	 */
	private Object convertValue(Object rawValue, EDataType targetType) {
		if (rawValue == null) {
			return null;
		}

		Class<?> targetClass = targetType.getInstanceClass();
		if (targetClass != null && targetClass.isAssignableFrom(rawValue.getClass())) {
			// No conversion needed
			return rawValue;
		}

		// Use EMF's built-in conversion
		String stringValue = EcoreUtil.convertToString(targetType, rawValue);
		return EcoreUtil.createFromString(targetType, stringValue);
	}

	/**
	 * Validates a resource mapping against a source instance.
	 */
	private void validateResourceMapping(EObject sourceInstance, ResourceMapping resourceMapping, 
			String serviceName, ValidationResultImpl result) {
		List<EStructuralFeature> featurePath = resourceMapping.getValueFeature();
		if (featurePath == null || featurePath.isEmpty()) {
			result.addWarning("Empty feature path for resource " + serviceName + "." + resourceMapping.getMid());
			return;
		}

		// Try to navigate the path to check if it's valid
		try {
			Optional<Object> value = getRawValue(sourceInstance, featurePath);
			if (!value.isPresent()) {
				result.addWarning("Feature path returns no value for resource " + 
						serviceName + "." + resourceMapping.getMid());
			}
		} catch (Exception e) {
			result.addError("Invalid feature path for resource " + serviceName + "." + 
					resourceMapping.getMid() + ": " + e.getMessage());
		}
	}

	/**
	 * Implementation of ValidationResult.
	 */
	private static class ValidationResultImpl implements ValidationResult {
		private final List<String> errors = new java.util.ArrayList<>();
		private final List<String> warnings = new java.util.ArrayList<>();

		void addError(String error) {
			errors.add(error);
		}

		void addWarning(String warning) {
			warnings.add(warning);
		}

		@Override
		public boolean isValid() {
			return errors.isEmpty();
		}

		@Override
		public List<String> getErrors() {
			return new java.util.ArrayList<>(errors);
		}

		@Override
		public List<String> getWarnings() {
			return new java.util.ArrayList<>(warnings);
		}
	}

	/**
	 * Traverses a path of {@link EStructuralFeature}s to retrieve a raw, untyped value.
	 * Handles multi-value features by taking the first element from collections.
	 * 
	 * @param source      The starting object.
	 * @param featurePath A list of features representing the path to the value.
	 * @return An {@link Optional} containing the raw value, or empty if not found.
	 */
	private Optional<Object> getRawValue(EObject source, List<EStructuralFeature> featurePath) {
		if (featurePath == null || featurePath.isEmpty()) {
			return Optional.empty();
		}

		Object currentValue = source;
		try {
			for (EStructuralFeature feature : featurePath) {
				if (currentValue instanceof EObject) {
					currentValue = ((EObject) currentValue).eGet(feature);

					// Handle multi-value features: take first element if it's a non-empty collection
					if (currentValue instanceof EList<?>) {
						List<?> list = (List<?>) currentValue;
						if (list.isEmpty()) {
							return Optional.empty();
						}
						currentValue = list.get(0);
					}
				} else {
					// The path is invalid if a non-EObject is encountered mid-path.
					return Optional.empty();
				}
			}
			return Optional.ofNullable(currentValue);
		} catch (Exception e) {
			// An error during path traversal (e.g., feature not found)
			logger.error("Error getting raw value for feature path: {} from source {}: {}", 
					featurePath, source.eClass().getName(), e.getMessage());
			return Optional.empty();
		}
	}

	/**
	 * Looks up for location data in the {@link AdminMapping} 
	 * @param adminMapping the {@link AdminMapping}
	 * @return the {@link GeoJsonObject} or <code>null</code>
	 */
	private GeoJsonObject lookupLocation(AdminMapping adminMapping) {
		requireNonNull(adminMapping);
		final Object latitude, longitude, altitude;
		latitude = adminMapping.getLatitude();
		if (isNull(latitude)) {
			// No latitude: no location
			return null;
		}

		longitude = adminMapping.getLongitude();
		if (isNull(longitude)) {
			// No longitude: no location
			return null;
		}

		// Altitude is optional
		altitude = adminMapping.getElevation();

		Coordinates coordinates = new Coordinates(toDouble(longitude), toDouble(latitude), toDouble(altitude));
		final Point point = new Point(coordinates, Collections.emptyList(), Collections.emptyMap());
		return point;
	}

	/**
	 * Tries to convert the given object to a double. Returns {@link Double#NaN} in
	 * case of an error.
	 *
	 * @param value Input value
	 * @return The double representation of the given object, or {@link Double#NaN}
	 *         in case of an error
	 */
	static Double toDouble(final Object value) {
		if (value == null) {
			return Double.NaN;
		}

		if (value instanceof Number) {
			return ((Number) value).doubleValue();
		}

		if (value instanceof CharSequence) {
			return Double.parseDouble(value.toString());
		}

		return Double.NaN;
	}
}