/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.audit.tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doNothing;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.mac.audit.AuditEventData;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.osgi.messaging.MessagingService;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

/**
 * Debug test to understand exactly what audit events are generated.
 * 
 * @author Mark Hoffmann
 * @since 26.10.2025
 */
@RequireEMF
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
class AuditEventDebugTest {
	
	@Mock 
	MessagingService mockedMsgService;
	
	private ServiceRegistration<MessagingService> mockServiceRegistration;
	
	@BeforeEach
	public void beforeEach(@InjectBundleContext BundleContext ctxt) throws Exception {
		doNothing().when(mockedMsgService).publish(anyString(), any());
		

		// Register mock ChatCompletionService with highest priority
		Dictionary<String, Object> chatProps = new Hashtable<>();
		chatProps.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.MAX_VALUE);
		chatProps.put("id", "biz");

		mockServiceRegistration = ctxt.registerService(
				MessagingService.class,
				mockedMsgService,
				chatProps);
	}
	
	@AfterEach
	public void afterEach() {
		if(mockServiceRegistration != null) {
			mockServiceRegistration.unregister();
		}
	}

    @Test
    void debugBasicSessionEvents(@InjectService ServiceAware<ResourceSet> awareResSet,  @InjectService ServiceAware<ProcessAuditSessionManager> awareSessionManager) throws InterruptedException {
        
    	ProcessAuditSessionManager sessionManager = awareSessionManager.waitForService(2000l);
    	
    	// Create session - should add SESSION_START
        ProcessAuditSession session = sessionManager.startSession("debug-test");
        
        System.out.println("=== After session creation ===");
        printEvents(session);
        assertEquals(1, session.getEvents().size(), "Should have 1 event after creation (SESSION_START)");
        
        // Add checkpoint - should add CHECKPOINT
        session.checkpoint("debug-checkpoint");
        
        System.out.println("=== After checkpoint ===");
        printEvents(session);
        assertEquals(2, session.getEvents().size(), "Should have 2 events after checkpoint");
        
        // Complete session - should add SESSION_COMPLETE
        session.complete("Debug test completed");
        
        System.out.println("=== After completion ===");
        printEvents(session);
        assertEquals(3, session.getEvents().size(), "Should have exactly 3 events after completion");
        
        // Verify event types
        List<AuditEventData> events = session.getEvents();
        assertEquals("SESSION_START", events.get(0).getEventType().name());
        assertEquals("CHECKPOINT", events.get(1).getEventType().name());
        assertEquals("SESSION_COMPLETE", events.get(2).getEventType().name());
    }
    
    @Test
    void debugServiceRegistrationEvents(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware) throws InterruptedException {
        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        ProcessAuditSession session = sessionManager.startSession("service-debug-test");
        session.addContext("testType", "service-registration");
        
        System.out.println("=== After session + context ===");
        printEvents(session);
        
        session.checkpoint("test-service-registered");
        
        System.out.println("=== After first checkpoint ===");
        printEvents(session);
        
        session.checkpoint("service-reference-obtained");
        
        System.out.println("=== After second checkpoint ===");
        printEvents(session);
        
        session.complete("Service registration test completed");
        
        System.out.println("=== After completion ===");
        printEvents(session);
        
        // Should be: SESSION_START + checkpoint1 + checkpoint2 + SESSION_COMPLETE = 4 events
        assertEquals(4, session.getEvents().size(), "Should have exactly 4 events");
    }
    
    @Test
    void verifyAuditObjectsAreCreated(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware) throws InterruptedException {
        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        // Create session and add various events
        ProcessAuditSession session = sessionManager.startSession("audit-object-test");
        session.checkpoint("checkpoint-1");
        session.checkpoint("checkpoint-2", "Detailed checkpoint");
        session.error("error-point", "Test error", new RuntimeException("Test exception"));
        session.addMethodCall("TestService", "testMethod", 1000L, null);
        session.complete("Test completed");

        // Verify all events have Audit objects
        List<AuditEventData> events = session.getEvents();
        for (AuditEventData event : events) {
            assertNotNull(event.getAudit(),
                "Event " + event.getEventType() + " should have an Audit object");
            assertNotNull(event.getAudit().getId(), "Audit ID should not be null");
            assertNotNull(event.getAudit().getSource(), "Audit source should not be null");
            assertNotNull(event.getAudit().getMessage(), "Audit message should not be null");
            assertNotNull(event.getAudit().getTimestamp(), "Audit timestamp should not be null");
            assertNotNull(event.getAudit().getAction(), "Audit action should not be null");
            assertNotNull(event.getAudit().getCategory(), "Audit category should not be null");

            System.out.println("Event: " + event.getEventType() +
                             " | Audit ID: " + event.getAudit().getId() +
                             " | Action: " + event.getAudit().getAction() +
                             " | Category: " + event.getAudit().getCategory());
        }

        // Verify specific action types
        assertEquals("SESSION_STARTED", events.get(0).getAudit().getAction().name());
        assertEquals("SESSION_CHECKPOINT", events.get(1).getAudit().getAction().name());
        assertEquals("FAILURE", events.get(3).getAudit().getAction().name()); // error event
        assertEquals("SESSION_COMPLETED", events.get(events.size() - 1).getAudit().getAction().name());
    }

    private void printEvents(ProcessAuditSession session) {
        List<AuditEventData> events = session.getEvents();
        System.out.println("Total events: " + events.size());
        for (int i = 0; i < events.size(); i++) {
            AuditEventData event = events.get(i);
            System.out.println("  " + (i+1) + ". " + event.getEventType() +
                             " - " + event.getCheckpoint() +
                             " - " + event.getDetails() +
                             (event.getAudit() != null ? " | Audit: " + event.getAudit().getAction() : ""));
        }
        System.out.println();
    }
}