/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.audit.tests;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doNothing;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.atomic.AtomicReference;

import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.osgi.messaging.MessagingService;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.hooks.service.EventListenerHook;
import org.osgi.framework.hooks.service.FindHook;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

/**
 * Integration tests for the complete OSGi audit system.
 * Tests FindHook registration, service interception, and transparent auditing.
 *
 * @author Mark Hoffmann
 * @since 26.10.2025
 */
@RequireEMF
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
class AuditSystemIntegrationTest {

	@InjectBundleContext
	BundleContext bundleContext;

	@Mock
	MessagingService mockedMsgService;

	private ServiceRegistration<MessagingService> mockServiceRegistration;

	public void setUp() throws Exception {
		doNothing().when(mockedMsgService).publish(anyString(), any());

		// Register mock MessagingService with highest priority
		Dictionary<String, Object> msgProps = new Hashtable<>();
		msgProps.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.MAX_VALUE);
		msgProps.put("id", "biz");

		mockServiceRegistration = bundleContext.registerService(
				MessagingService.class,
				mockedMsgService,
				msgProps);
	}

	@AfterEach
	public void afterEach() {
		if (mockServiceRegistration != null) {
			mockServiceRegistration.unregister();
		}
	}

	@Test
	void testAuditSystemActivation(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
			throws Exception {

		setUp();

		// Verify the audit session manager is available
		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Test basic session creation
		ProcessAuditSession session = sessionManager.startSession("integration-test");
		assertNotNull(session);
		assertTrue(session.isActive());
		assertEquals("integration-test", session.getProcessName());

		session.checkpoint("system-activated");
		session.complete("Integration test basic functionality verified");

		assertFalse(session.isActive());
		// Should be: SESSION_START (auto) + CHECKPOINT + SESSION_COMPLETE = 3 events
		assertEquals(3, session.getEvents().size(), "Should have exactly 3 events (SESSION_START + checkpoint + SESSION_COMPLETE)");
	}

	@Test 
	void testFindHookRegistration(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware, @InjectService(cardinality = 0) ServiceAware<FindHook> findHookAware) 
			throws Exception {

		setUp();
		// Verify the audit session manager is available
		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Verify the audit FindHook is registered
		FindHook findHook = findHookAware.waitForService(10000);
		assertNotNull(findHook, "Audit FindHook should be registered");

		//		Needed to show that mock is really used, otherwise we get an exception on the mock
		sessionManager.startSession("test");

		// The FindHook should be from our audit system
		ServiceReference<FindHook> ref = findHookAware.getServiceReference();
		Object bundleSymbolicName = ref.getBundle().getSymbolicName();
		assertTrue(bundleSymbolicName.toString().contains("audit"), 
				"FindHook should be from audit bundle, was: " + bundleSymbolicName);
	}

	@Test
	void testEventListenerHookRegistration(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware, @InjectService(cardinality = 0) ServiceAware<EventListenerHook> eventHookAware) 
			throws Exception {

		setUp();

		// Verify the audit session manager is available
		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");
		
		// Verify the audit EventListenerHook is registered
		EventListenerHook eventHook = eventHookAware.waitForService(10000);
		assertNotNull(eventHook, "Audit EventListenerHook should be registered");

		//		Needed to show that mock is really used, otherwise we get an exception on the mock
		sessionManager.startSession("test");

		// The EventListenerHook should be from our audit system
		ServiceReference<EventListenerHook> ref = eventHookAware.getServiceReference();
		Object bundleSymbolicName = ref.getBundle().getSymbolicName();
		assertTrue(bundleSymbolicName.toString().contains("audit"), 
				"EventListenerHook should be from audit bundle, was: " + bundleSymbolicName);
	}

	/**
	 * Test interface for audit testing
	 */
	public interface TestService {
		String processData(String input) throws IllegalArgumentException;
		int calculateSum(int a, int b);
		void doSomething() throws IllegalArgumentException;
	}

	/**
	 * Simple test service implementation
	 */
	public static class TestServiceImpl implements TestService {

		@Override
		public String processData(String input) {
			if (input == null) {
				throw new IllegalArgumentException("Input cannot be null");
			}
			return "processed:" + input.toUpperCase();
		}

		@Override
		public int calculateSum(int a, int b) {
			return a + b;
		}

		@Override
		public void doSomething() throws IllegalArgumentException {
			throw new IllegalArgumentException("This method always fails for testing");
		}
	}

	@Test
	void testServiceRegistrationAndAuditing(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
			throws Exception {

		setUp();

		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Register a test service WITH audit.me=true
		TestService testService = new TestServiceImpl();
		Dictionary<String, Object> auditProps = new Hashtable<>();
		auditProps.put("audit.me", "true");
		auditProps.put("service.description", "Test service for audit integration");

		bundleContext.registerService(TestService.class, testService, auditProps);

		// Give the audit system time to process the registration
		Thread.sleep(100);

		// Start an audit session
		ProcessAuditSession session = sessionManager.startSession("service-audit-test");
		session.addContext("testType", "service-registration");
		session.checkpoint("test-service-registered");

		// Now try to get the service - it should be audited if the FindHook worked
		ServiceReference<TestService> serviceRef = bundleContext.getServiceReference(TestService.class);
		assertNotNull(serviceRef, "Test service should be available");

		// Check if the service has audit.me property
		Object auditMe = serviceRef.getProperty("audit.me");
		assertEquals("true", auditMe, "Service should have audit.me=true property");

		session.checkpoint("service-reference-obtained");
		session.complete("Service registration test completed");

		// Should be: SESSION_START (auto) + checkpoint1 + checkpoint2 + SESSION_COMPLETE = 4 events
		assertEquals(4, session.getEvents().size(), "Should have exactly 4 events (SESSION_START + 2 checkpoints + SESSION_COMPLETE)");
		assertFalse(session.isActive());
	}

	@Test
	void testProxiedServiceMethodCalls(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
			throws Exception {

		setUp();

		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Register test service with auditing enabled
		TestService testService = new TestServiceImpl();
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("audit.me", "true");
		props.put("test.service", "proxied");

		bundleContext.registerService(TestService.class, testService, props);
		Thread.sleep(100); // Let audit system process registration

		// Start audit session
		ProcessAuditSession session = sessionManager.startSession("proxy-method-test");
		session.addContext("testScenario", "proxied-service-calls");
		session.checkpoint("session-started");

		// Get service (should return proxy if FindHook is working)
		ServiceReference<TestService> ref = bundleContext.getServiceReference(TestService.class);
		TestService service = bundleContext.getService(ref);
		assertNotNull(service, "Service should be available");

		session.checkpoint("service-obtained");

		// Call service methods - these should be audited if proxy is working
		String result1 = service.processData("test-input");
		assertEquals("processed:TEST-INPUT", result1);

		int result2 = service.calculateSum(5, 3);
		assertEquals(8, result2);

		session.checkpoint("service-methods-called");

		// Test exception handling
		try {
			service.doSomething();
			fail("Expected IllegalArgumentException");
		} catch (IllegalArgumentException e) {
			// Expected exception - audit should capture this
			session.checkpoint("exception-handled");
		}

		session.complete("Proxy method test completed");

		// Verify session has all expected events
		assertTrue(session.getEvents().size() >= 5); // START + checkpoints + method calls + COMPLETE
		assertFalse(session.isActive());

		// Clean up
		bundleContext.ungetService(ref);
	}

	@Test
	void testMultiThreadServiceAuditing(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
			throws Exception {

		setUp();

		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Register auditable service
		TestService testService = new TestServiceImpl();
		Dictionary<String, Object> props = new Hashtable<>();
		props.put("audit.me", "true");
		props.put("test.type", "multithread");

		bundleContext.registerService(TestService.class, testService, props);
		Thread.sleep(100);

		// Start main session
		ProcessAuditSession mainSession = sessionManager.startSession("multithread-audit-test");
		mainSession.addContext("scenario", "multithread-service-calls");
		mainSession.checkpoint("main-thread-started");

		AtomicReference<Exception> workerException = new AtomicReference<>();
		AtomicReference<String> workerResult = new AtomicReference<>();

		// Create worker thread that will use the same session
		Thread workerThread = new Thread(() -> {
			try {
				// Set the session in worker thread
				sessionManager.setCurrentSession(mainSession);

				// Get and use the service in worker thread
				ServiceReference<TestService> ref = bundleContext.getServiceReference(TestService.class);
				TestService service = bundleContext.getService(ref);

				mainSession.checkpoint("worker-thread-service-obtained");
				mainSession.addContext("workerThread", Thread.currentThread().getName());

				// Call service method in worker thread
				String result = service.processData("worker-data");
				workerResult.set(result);

				mainSession.checkpoint("worker-thread-completed");
				bundleContext.ungetService(ref);

			} catch (Exception e) {
				workerException.set(e);
				mainSession.error("worker-thread", "Worker thread failed", e);
			}
		});

		workerThread.start();
		workerThread.join(5000); // Wait up to 5 seconds

		// Back in main thread
		mainSession.checkpoint("worker-thread-joined");

		// Verify worker thread execution
		assertNull(workerException.get(), "Worker thread should not have thrown exception");
		assertEquals("processed:WORKER-DATA", workerResult.get());

		mainSession.complete("Multithread audit test completed");

		// Verify session captured events from both threads
		assertTrue(mainSession.getEvents().size() >= 6);
		assertFalse(mainSession.isActive());

		// Verify context from worker thread is present
		assertNotNull(mainSession.getContext("workerThread"));
	}

	@Test
	void testAuditSessionPersistence(@InjectService(cardinality = 0) ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
			throws Exception {

		setUp();

		ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(5000);
		assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

		// Create session with comprehensive data
		ProcessAuditSession session = sessionManager.startSession("persistence-test");
		session.addContext("user", "testuser");
		session.addContext("operation", "data-export");
		session.addContext("version", "1.0.0");

		session.checkpoint("initialization");
		session.checkpoint("data-loaded", "Loaded 100 records");
		session.checkpoint("processing", "Applied business rules");

		// Simulate some method calls
		session.addMethodCall("DataService", "loadRecords", 1500000L, null);
		session.addMethodCall("ProcessingService", "applyRules", 2300000L, null);
		session.addMethodCall("ExportService", "generateReport", null, 
				new RuntimeException("Export temporarily unavailable"));

		session.complete("Test completed with mixed results");

		// Verify EMF model is accessible and contains all data
		assertNotNull(session);
		assertEquals("persistence-test", session.getProcessName());
		assertEquals("testuser", session.getContext("user"));
		assertEquals("data-export", session.getContext("operation"));

		// Verify all events are present
		assertTrue(session.getEvents().size() >= 7); // START + 3 checkpoints + 3 method calls + COMPLETE

		// The session should be serializable via EMF (this tests EMF integration)
		assertNotNull(session.getSessionId());
		assertNotNull(session.getStartTime());
		assertFalse(session.isActive());
	}
}