/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.audit.tests;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doNothing;

import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.gecko.mac.audit.AuditEventData;
import org.gecko.mac.audit.EventType;
import org.gecko.mac.auditapi.ProcessAuditSession;
import org.gecko.mac.auditapi.ProcessAuditSessionManager;
import org.gecko.osgi.messaging.MessagingService;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

/**
 * Integration tests focused on service proxy auditing functionality.
 * Tests transparent method interception and audit event generation.
 *
 * @author Mark Hoffmann
 * @since 26.10.2025
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@ExtendWith(MockitoExtension.class)
class ServiceProxyAuditTest {

    @InjectBundleContext
    BundleContext bundleContext;

    @Mock
    MessagingService mockedMsgService;

    private ServiceRegistration<MessagingService> mockServiceRegistration;

    @BeforeEach
    public void beforeEach() throws Exception {
        doNothing().when(mockedMsgService).publish(anyString(), any());

        // Register mock MessagingService with highest priority
        Dictionary<String, Object> msgProps = new Hashtable<>();
        msgProps.put(org.osgi.framework.Constants.SERVICE_RANKING, Integer.MAX_VALUE);
        msgProps.put("id", "biz");

        mockServiceRegistration = bundleContext.registerService(
                MessagingService.class,
                mockedMsgService,
                msgProps);
    }

    @AfterEach
    public void afterEach() {
        if (mockServiceRegistration != null) {
            mockServiceRegistration.unregister();
        }
    }

    /**
     * Complex test service interface for comprehensive auditing tests
     */
    public interface BusinessService {
        // Simple operations
        String processOrder(String orderId, String customerId);
        double calculateDiscount(double amount, String customerType);
        
        // Operations with different return types
        int countItems(String[] items);
        boolean validatePayment(String paymentToken);
        void logActivity(String activity);
        
        // Operations that can throw exceptions
        String authenticateUser(String username, String password) throws SecurityException;
        void processPayment(double amount) throws IllegalArgumentException;
        
        // Complex operations for performance testing
        String generateReport(String type, int recordCount);
    }

    /**
     * Implementation of the business service for testing
     */
    public static class BusinessServiceImpl implements BusinessService {

        @Override
        public String processOrder(String orderId, String customerId) {
            if (orderId == null || customerId == null) {
                throw new IllegalArgumentException("OrderId and CustomerId cannot be null");
            }
            return "ORDER_" + orderId + "_PROCESSED_FOR_" + customerId;
        }

        @Override
        public double calculateDiscount(double amount, String customerType) {
            if (amount < 0) {
                throw new IllegalArgumentException("Amount cannot be negative");
            }
            switch (customerType.toLowerCase()) {
                case "premium": return amount * 0.15;
                case "gold": return amount * 0.10;
                case "silver": return amount * 0.05;
                default: return 0.0;
            }
        }

        @Override
        public int countItems(String[] items) {
            return items != null ? items.length : 0;
        }

        @Override
        public boolean validatePayment(String paymentToken) {
            return paymentToken != null && paymentToken.startsWith("PAY_") && paymentToken.length() > 10;
        }

        @Override
        public void logActivity(String activity) {
            // Simulate logging - no return value
            System.out.println("ACTIVITY: " + activity);
        }

        @Override
        public String authenticateUser(String username, String password) throws SecurityException {
            if ("admin".equals(username) && "secret".equals(password)) {
                return "AUTH_TOKEN_12345";
            }
            throw new SecurityException("Invalid credentials");
        }

        @Override
        public void processPayment(double amount) throws IllegalArgumentException {
            if (amount <= 0) {
                throw new IllegalArgumentException("Payment amount must be positive");
            }
            if (amount > 10000) {
                throw new IllegalArgumentException("Payment amount exceeds limit");
            }
            // Payment processed successfully
        }

        @Override
        public String generateReport(String type, int recordCount) {
            // Simulate some processing time
            try {
                Thread.sleep(10 + recordCount); // Variable delay based on record count
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            return type.toUpperCase() + "_REPORT_" + recordCount + "_RECORDS";
        }
    }

    @Test
    void testBasicServiceMethodAuditing(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
            throws InterruptedException {

        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        // Register auditable business service
        BusinessService businessService = new BusinessServiceImpl();
        Dictionary<String, Object> props = new Hashtable<>();
        props.put("audit.me", "true");
        props.put("service.name", "business-service");
        
        ServiceRegistration<BusinessService> registration = 
            bundleContext.registerService(BusinessService.class, businessService, props);
        
        try {
            Thread.sleep(100); // Allow audit system to process registration
            
            // Start audit session
            ProcessAuditSession session = sessionManager.startSession("business-service-audit");
            session.addContext("testCase", "basic-method-auditing");
            session.checkpoint("test-started");
            
            // Get the service (should be proxied)
            ServiceReference<BusinessService> ref = bundleContext.getServiceReference(BusinessService.class);
            BusinessService service = bundleContext.getService(ref);
            
            session.checkpoint("service-obtained");
            
            // Call various service methods
            String orderResult = service.processOrder("ORD-123", "CUST-456");
            assertEquals("ORDER_ORD-123_PROCESSED_FOR_CUST-456", orderResult);
            
            double discount = service.calculateDiscount(100.0, "premium");
            assertEquals(15.0, discount, 0.01);
            
            int itemCount = service.countItems(new String[]{"item1", "item2", "item3"});
            assertEquals(3, itemCount);
            
            boolean paymentValid = service.validatePayment("PAY_VALID_TOKEN_123");
            assertTrue(paymentValid);
            
            service.logActivity("test-activity-logged");
            
            session.checkpoint("all-methods-called");
            session.complete("Basic service auditing test completed");
            
            // Verify audit events were created for method calls
            List<AuditEventData> events = session.getEvents();
            assertTrue(events.size() >= 8); // START + checkpoints + method calls + COMPLETE
            
            // Count SERVICE_METHOD_CALL events
            long methodCallEvents = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .count();
            assertEquals(5, methodCallEvents, "Should have 5 audited method calls");
            
            // Verify specific method call audit events
            boolean foundProcessOrder = events.stream()
                .anyMatch(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL && 
                             e.getCheckpoint().contains("processOrder"));
            assertTrue(foundProcessOrder, "Should find processOrder method call audit");
            
            bundleContext.ungetService(ref);
            
        } finally {
            registration.unregister();
        }
    }

    @Test
    void testExceptionHandlingInAuditedMethods(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
            throws InterruptedException {

        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        BusinessService businessService = new BusinessServiceImpl();
        Dictionary<String, Object> props = new Hashtable<>();
        props.put("audit.me", "true");
        props.put("service.type", "exception-testing");
        
        ServiceRegistration<BusinessService> registration = 
            bundleContext.registerService(BusinessService.class, businessService, props);
        
        try {
            Thread.sleep(100);
            
            ProcessAuditSession session = sessionManager.startSession("exception-handling-test");
            session.addContext("testType", "exception-scenarios");
            
            ServiceReference<BusinessService> ref = bundleContext.getServiceReference(BusinessService.class);
            BusinessService service = bundleContext.getService(ref);
            
            // Test successful authentication
            String authToken = service.authenticateUser("admin", "secret");
            assertEquals("AUTH_TOKEN_12345", authToken);
            
            // Test failed authentication (should throw SecurityException)
            try {
                service.authenticateUser("user", "wrong");
                fail("Expected SecurityException");
            } catch (SecurityException e) {
                // Expected - audit should capture this exception
                session.checkpoint("authentication-failed-as-expected");
            }
            
            // Test successful payment
            service.processPayment(100.0);
            
            // Test failed payment (negative amount)
            try {
                service.processPayment(-50.0);
                fail("Expected IllegalArgumentException");
            } catch (IllegalArgumentException e) {
                session.checkpoint("negative-payment-rejected");
            }
            
            // Test failed payment (amount too high)
            try {
                service.processPayment(15000.0);
                fail("Expected IllegalArgumentException");
            } catch (IllegalArgumentException e) {
                session.checkpoint("excessive-payment-rejected");
            }
            
            session.complete("Exception handling test completed");
            
            // Verify audit captured both successful and failed method calls
            List<AuditEventData> events = session.getEvents();
            
            // Count method call events
            long methodCallEvents = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .count();
            assertEquals(5, methodCallEvents, "Should have 5 method call events");
            
            // Verify some method calls have exceptions recorded
            long failedMethodCalls = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .filter(e -> e.getException() != null)
                .count();
            assertTrue(failedMethodCalls >= 3, "Should have at least 3 failed method calls recorded");
            
            bundleContext.ungetService(ref);
            
        } finally {
            registration.unregister();
        }
    }

    @Test
    void testPerformanceAuditingWithDuration(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
            throws InterruptedException {

        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        BusinessService businessService = new BusinessServiceImpl();
        Dictionary<String, Object> props = new Hashtable<>();
        props.put("audit.me", "true");
        props.put("service.category", "performance-testing");
        
        ServiceRegistration<BusinessService> registration = 
            bundleContext.registerService(BusinessService.class, businessService, props);
        
        try {
            Thread.sleep(100);
            
            ProcessAuditSession session = sessionManager.startSession("performance-audit-test");
            session.addContext("testFocus", "method-duration-tracking");
            
            ServiceReference<BusinessService> ref = bundleContext.getServiceReference(BusinessService.class);
            BusinessService service = bundleContext.getService(ref);
            
            // Call methods with different expected durations
            service.generateReport("SALES", 1);     // Short duration
            service.generateReport("INVENTORY", 10); // Medium duration
            service.generateReport("ANALYTICS", 50); // Longer duration
            
            session.complete("Performance auditing test completed");
            
            // Verify duration tracking
            List<AuditEventData> events = session.getEvents();
            List<AuditEventData> methodCalls = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .filter(e -> e.getCheckpoint().contains("generateReport"))
                .toList();
            
            assertEquals(3, methodCalls.size(), "Should have 3 generateReport calls");
            
            // Verify all method calls have duration recorded
            for (AuditEventData event : methodCalls) {
                assertNotNull(event.getDurationNs(), "Duration should be recorded for: " + event.getCheckpoint());
                assertTrue(event.getDurationNs() > 0, "Duration should be positive for: " + event.getCheckpoint());
                
                // Verify duration is reasonable (should be at least 1ms for our test delays)
                assertTrue(event.getDurationNs() >= 1_000_000, // 1ms in nanoseconds
                    "Duration should be at least 1ms for: " + event.getCheckpoint() + 
                    " but was: " + event.getDurationNs() + " ns");
            }
            
            bundleContext.ungetService(ref);
            
        } finally {
            registration.unregister();
        }
    }

    @Test
    void testConcurrentServiceAuditing(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
            throws Exception {

        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        BusinessService businessService = new BusinessServiceImpl();
        Dictionary<String, Object> props = new Hashtable<>();
        props.put("audit.me", "true");
        props.put("service.usage", "concurrent");
        
        ServiceRegistration<BusinessService> registration = 
            bundleContext.registerService(BusinessService.class, businessService, props);
        
        try {
            Thread.sleep(100);
            
            ProcessAuditSession session = sessionManager.startSession("concurrent-audit-test");
            session.addContext("concurrencyTest", "multiple-threads");
            
            ServiceReference<BusinessService> ref = bundleContext.getServiceReference(BusinessService.class);
            BusinessService service = bundleContext.getService(ref);
            
            // Create multiple concurrent tasks
            CompletableFuture<String> task1 = CompletableFuture.supplyAsync(() -> {
                sessionManager.setCurrentSession(session);
                return service.processOrder("CONC-001", "CUST-001");
            });
            
            CompletableFuture<Double> task2 = CompletableFuture.supplyAsync(() -> {
                sessionManager.setCurrentSession(session);
                return service.calculateDiscount(200.0, "gold");
            });
            
            CompletableFuture<String> task3 = CompletableFuture.supplyAsync(() -> {
                sessionManager.setCurrentSession(session);
                return service.generateReport("CONCURRENT", 5);
            });
            
            CompletableFuture<Boolean> task4 = CompletableFuture.supplyAsync(() -> {
                sessionManager.setCurrentSession(session);
                return service.validatePayment("PAY_CONCURRENT_TOKEN_456");
            });
            
            // Wait for all tasks to complete
            CompletableFuture.allOf(task1, task2, task3, task4).get(10, TimeUnit.SECONDS);
            
            // Verify results
            assertEquals("ORDER_CONC-001_PROCESSED_FOR_CUST-001", task1.get());
            assertEquals(20.0, task2.get(), 0.01);
            assertTrue(task3.get().contains("CONCURRENT_REPORT"));
            assertTrue(task4.get());
            
            session.complete("Concurrent auditing test completed");
            
            // Verify all method calls were audited
            List<AuditEventData> events = session.getEvents();
            long methodCallEvents = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .count();
            assertEquals(4, methodCallEvents, "Should have 4 concurrent method calls audited");
            
            // Verify different thread IDs were captured
            long distinctThreads = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .mapToLong(AuditEventData::getThreadId)
                .distinct()
                .count();
            assertTrue(distinctThreads >= 2, "Should have method calls from multiple threads");
            
            bundleContext.ungetService(ref);
            
        } finally {
            registration.unregister();
        }
    }

    @Test
    void testNonAuditedServiceNotProxied(@InjectService ServiceAware<ProcessAuditSessionManager> sessionManagerAware)
            throws InterruptedException {

        ProcessAuditSessionManager sessionManager = sessionManagerAware.waitForService(2000);
        assertNotNull(sessionManager, "ProcessAuditSessionManager service should be available");

        // Register service WITHOUT audit.me=true
        BusinessService regularService = new BusinessServiceImpl();
        Dictionary<String, Object> props = new Hashtable<>();
        props.put("service.type", "regular-no-audit");
        // Note: NO audit.me property
        
        ServiceRegistration<BusinessService> registration = 
            bundleContext.registerService(BusinessService.class, regularService, props);
        
        try {
            Thread.sleep(100);
            
            ProcessAuditSession session = sessionManager.startSession("non-audited-service-test");
            session.checkpoint("testing-non-audited-service");
            
            ServiceReference<BusinessService> ref = bundleContext.getServiceReference(BusinessService.class);
            BusinessService service = bundleContext.getService(ref);
            
            // Call methods on non-audited service
            String result = service.processOrder("NON-AUDIT-001", "CUST-REGULAR");
            assertEquals("ORDER_NON-AUDIT-001_PROCESSED_FOR_CUST-REGULAR", result);
            
            service.calculateDiscount(50.0, "silver");
            service.logActivity("non-audited-activity");
            
            session.checkpoint("non-audited-methods-called");
            session.complete("Non-audited service test completed");
            
            // Verify NO method call audit events were created
            List<AuditEventData> events = session.getEvents();
            long methodCallEvents = events.stream()
                .filter(e -> e.getEventType() == EventType.SERVICE_METHOD_CALL)
                .count();
            assertEquals(0, methodCallEvents, "Non-audited service should not generate method call events");
            
            // Should only have our manual checkpoints and session events
            assertTrue(events.size() <= 4); // START + 2 checkpoints + COMPLETE
            
            bundleContext.ungetService(ref);
            
        } finally {
            registration.unregister();
        }
    }
}