/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.event.handler;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.fennec.codec.options.CodecResourceOptions;
import org.gecko.mac.audit.Audit;
import org.gecko.mac.audit.AuditPackage;
import org.gecko.osgi.messaging.MessagingService;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.typedevent.TypedEventHandler;

/**
 * 
 * @author ilenia
 * @since Oct 22, 2025
 */
@Designate(ocd = AuditEventHandlerConfig.class)
@Component(name = "AuditEventHandler", service = TypedEventHandler.class, configurationPid = "AuditEventHandler", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class AuditEventHandler implements TypedEventHandler<Audit> {
	
	private static final Map<String, Object> EMF_CONFIG = Map.ofEntries(
			Map.entry(CodecResourceOptions.CODEC_ROOT_OBJECT, AuditPackage.Literals.AUDIT)
			);
	
	private static final Logger LOGGER = Logger.getLogger(AuditEventHandler.class.getName());
	
	private MessagingService messagingService;
	private AuditEventHandlerConfig config;
	private ResourceSet resourceSet;

	@Activate
	public AuditEventHandler(AuditEventHandlerConfig config, @Reference(target = "(id=biz)") MessagingService messagingService, 
			@Reference ResourceSet resourceSet) {
		this.config = config;
		this.messagingService = messagingService;
		this.resourceSet = resourceSet;		
	}

	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.typedevent.TypedEventHandler#notify(java.lang.String, java.lang.Object)
	 */
	@Override
	public void notify(String topic, Audit audit) {
		ByteArrayOutputStream bao = createMessageContent(audit);
		if(bao != null) {
			try {
				messagingService.publish(config.mqtt_forward_topic(), ByteBuffer.wrap(bao.toByteArray()));
			} catch(Exception e) {
				LOGGER.log(Level.SEVERE, String.format("Exception while sending Audit via MQTT for %s", audit.getId()), e);
			}
		}
	}
	
	private ByteArrayOutputStream createMessageContent(Audit audit) {
		Resource resource = resourceSet.createResource(URI.createURI(UUID.randomUUID().toString()+".xmi"));
		try {
			resource.getContents().add(audit);
			ByteArrayOutputStream bao = new ByteArrayOutputStream();
			resource.save(bao, EMF_CONFIG);
			return bao;
		} catch(IOException e) {
			LOGGER.log(Level.SEVERE, String.format("IOException while wrapping Audit to be sent via MQTT for %s", audit.getId()), e);
			return null;
		} finally {
			resource.getContents().clear();
			resourceSet.getResources().remove(resource);
		}
	}

}
