/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.event.handler;

import java.io.IOException;
import java.util.Dictionary;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.typedevent.TypedEventHandler;

@Designate(ocd = ConfigurationEventHandlerConfig.class)
@Component(name = "ConfigurationEventHandler", service = TypedEventHandler.class, configurationPid = "ConfigurationEventHandler", configurationPolicy = ConfigurationPolicy.REQUIRE)
public class ConfigurationEventHandler implements TypedEventHandler<String[]>{

	private static final Logger LOGGER = Logger.getLogger(ConfigurationEventHandler.class.getName());
	
	@Reference
	ConfigurationAdmin configAdmin;
	
	/* 
	 * (non-Javadoc)
	 * @see org.osgi.service.typedevent.TypedEventHandler#notify(java.lang.String, java.lang.Object)
	 */
	@Override
	public void notify(String topic, String event[]) {
		
		String[] topicSplit = topic.split("/");
		if(topicSplit.length < 2) {
			LOGGER.log(Level.SEVERE, String.format("Expecting topic to be of the form configuration/<ADD/REMOVE>/<configurationPID> but instead it was %s", topic));
			return;
		}
		
		String configurationPID = topicSplit[topicSplit.length-1];
		String configStrategy = topicSplit[topicSplit.length-2];
		if(!"ADD".equals(configStrategy) && !"REMOVE".equals(configStrategy)) {
			LOGGER.log(Level.SEVERE, String.format("Expecting configuration strategy to be either ADD or REMOVE but instead was %s", configStrategy));
			return;
		}
		if("ADD".equals(configStrategy) && event.length != 2) {
			LOGGER.log(Level.SEVERE, String.format("Exepcted event to contain 2 elements for strategy ADD. Instead it contains %s elements.", event == null ? "0" : event.length));
			return;
		}
		if("REMOVE".equals(configStrategy) && event.length != 1) {
			LOGGER.log(Level.SEVERE, String.format("Exepcted event to contain 1 element for strategy REMOVE. Instead it contains %s elements.", event == null ? "0" : event.length));
			return;
		}
		try {
			Configuration configuration = configAdmin.getConfiguration(configurationPID, "?");
			Dictionary<String, Object> properties = configuration.getProperties();
			if(configuration.getProperties() != null) {
				if("ADD".equals(configStrategy)) {					
					properties.put(event[0], event[1]);
					configuration.update(properties);
				} else {
					properties.remove(event[0]);
					configuration.update(properties);
				} 			
			} else {
				LOGGER.log(Level.WARNING, String.format("Configuration %s does not exsist yet. We do not want to create a new one. We only want to update an existing one.", configurationPID));
			}
		} catch (IOException e) {
			LOGGER.log(Level.SEVERE, String.format("Failed to access Configuration %s", configurationPID), e);
		}		
	}
}
