/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.golf.features;

import org.locationtech.jts.algorithm.Orientation;
import org.locationtech.jts.geom.Coordinate;
import org.osgi.service.component.annotations.Component;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.Hole;
import com.playertour.backend.golfcourse.model.golfcourse.Location;
import com.playertour.backend.golfcourse.model.golfcourse.Shape;
import com.playertour.backend.golfcourse.model.golfcourse.ShapeObject;

import geojson.AbstractGeometry;
import geojson.Feature;
import geojson.FeatureCollection;
import geojson.GeojsonFactory;
import geojson.LineString;
import geojson.Point;
import geojson.Polygon;

/**
 * Service for transforming golf course data to GeoJSON FeatureCollection format.
 * 
 * @author Mark Hoffmann
 * @since 14.10.2025
 */
@Component(service = GolfCourseFeatureTransformer.class)
public class GolfCourseFeatureTransformer {

    private final PromiseFactory promiseFactory = new PromiseFactory(Runnable::run);
    private final GeojsonFactory geojsonFactory = GeojsonFactory.eINSTANCE;

    /**
     * Transforms a golf course to a GeoJSON FeatureCollection.
     * 
     * @param golfCourse the golf course to transform
     * @return Promise resolving to a FeatureCollection
     */
    public Promise<FeatureCollection> transformToGeoJSON(GolfCourse golfCourse) {
        return promiseFactory.submit(() -> {
            FeatureCollection featureCollection = geojsonFactory.createFeatureCollection();
            
            // Add course-level feature
            Feature courseFeature = createCourseFeature(golfCourse);
            featureCollection.getFeatures().add(courseFeature);
            
            // Add hole features
            if (golfCourse.getCourseGPSVector() != null && 
                golfCourse.getCourseGPSVector().getHoles() != null) {
                
                for (Hole hole : golfCourse.getCourseGPSVector().getHoles()) {
                    Feature holeFeature = createHoleFeature(hole, golfCourse);
                    featureCollection.getFeatures().add(holeFeature);
                    
                    // Add hole element features (bunkers, greens, fairways, etc.)
                    addHoleElementFeatures(hole, featureCollection, golfCourse);
                }
            }
            
            // Add course-level shape features (trees, water, etc.)
            if (golfCourse.getCourseGPSVector() != null) {
                addCourseShapeFeatures(golfCourse.getCourseGPSVector(), featureCollection, golfCourse);
            }
            
            return featureCollection;
        });
    }

    /**
     * Creates a feature representing the entire golf course.
     */
    private Feature createCourseFeature(GolfCourse golfCourse) {
        Feature feature = geojsonFactory.createFeature();
        
        // Set geometry from course location
        if (golfCourse.getCourseDetails() != null && 
            golfCourse.getCourseDetails().getLocation() != null) {
            Point point = createPointGeometry(golfCourse.getCourseDetails().getLocation());
            feature.setGeometry(point);
        }
        
        // Set feature type
        feature.setType("Feature");
        
        // Set properties using EMap
        feature.getProperties().put("type", "golf_course");
        feature.getProperties().put("id", golfCourse.getId());
        feature.getProperties().put("courseId", golfCourse.getCourseId());
        
        if (golfCourse.getCourseDetails() != null) {
            feature.getProperties().put("name", golfCourse.getCourseDetails().getCourseName());
            feature.getProperties().put("holeCount", String.valueOf(golfCourse.getCourseDetails().getHoleNum()));
            feature.getProperties().put("countryId", golfCourse.getCourseDetails().getCountryId());
            feature.getProperties().put("stateId", golfCourse.getCourseDetails().getStateId());
            if (golfCourse.getCourseDetails().getType() != null) {
                feature.getProperties().put("courseType", golfCourse.getCourseDetails().getType().toString());
            }
            
            if (golfCourse.getCourseDetails().getAddress() != null) {
                feature.getProperties().put("address_route", golfCourse.getCourseDetails().getAddress().getRoute());
                feature.getProperties().put("address_zipCode", golfCourse.getCourseDetails().getAddress().getZipCode());
                feature.getProperties().put("address_city", golfCourse.getCourseDetails().getAddress().getCity());
                feature.getProperties().put("address_state", golfCourse.getCourseDetails().getAddress().getState());
                feature.getProperties().put("address_country", golfCourse.getCourseDetails().getAddress().getCountry());
            }
            
            if (golfCourse.getCourseDetails().getContact() != null) {
                feature.getProperties().put("contact_email", golfCourse.getCourseDetails().getContact().getEmail());
                feature.getProperties().put("contact_phone", golfCourse.getCourseDetails().getContact().getPhone());
                feature.getProperties().put("contact_url", golfCourse.getCourseDetails().getContact().getUrl());
            }
        }
        return feature;
    }

    /**
     * Creates a feature representing a golf hole.
     */
    private Feature createHoleFeature(Hole hole, GolfCourse golfCourse) {
        Feature feature = geojsonFactory.createFeature();
        
        // Try to get geometry from green center or tee box center
        AbstractGeometry geometry = null;
        if (hole.getGreenCenter() != null && !hole.getGreenCenter().getShapes().isEmpty()) {
            geometry = createPointFromShapeObject(hole.getGreenCenter());
        } else if (hole.getTeeboxCenter() != null && !hole.getTeeboxCenter().getShapes().isEmpty()) {
            geometry = createPointFromShapeObject(hole.getTeeboxCenter());
        }
        if (geometry != null) {
            feature.setGeometry(geometry);
        }
        
        // Set feature type
        feature.setType("Feature");
        
        // Set properties using EMap
        feature.getProperties().put("type", "golf_hole");
        feature.getProperties().put("holeNumber", String.valueOf(hole.getHoleNumber()));
        feature.getProperties().put("courseId", golfCourse.getCourseId());
        if (golfCourse.getCourseDetails() != null) {
            feature.getProperties().put("courseName", golfCourse.getCourseDetails().getCourseName());
        }
        return feature;
    }

    /**
     * Adds features for hole elements (bunkers, greens, fairways, etc.).
     */
    private void addHoleElementFeatures(Hole hole, FeatureCollection featureCollection, GolfCourse golfCourse) {
        // Add bunker features
        if (hole.getBunker() != null) {
            addShapeObjectFeatures(hole.getBunker(), "bunker", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
        
        // Add green features
        if (hole.getGreen() != null) {
            addShapeObjectFeatures(hole.getGreen(), "green", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
        
        // Add fairway features
        if (hole.getFairway() != null) {
            addShapeObjectFeatures(hole.getFairway(), "fairway", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
        
        // Add teebox features
        if (hole.getTeebox() != null) {
            addShapeObjectFeatures(hole.getTeebox(), "teebox", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
        
        // Add perimeter features
        if (hole.getPerimeter() != null) {
            addShapeObjectFeatures(hole.getPerimeter(), "perimeter", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
        
        // Add central path features
        if (hole.getCentralPath() != null) {
            addShapeObjectFeatures(hole.getCentralPath(), "central_path", hole.getHoleNumber(), 
                golfCourse, featureCollection);
        }
    }

    /**
     * Adds course-level shape features (trees, water, etc.).
     */
    private void addCourseShapeFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector gpsVector, 
                                       FeatureCollection featureCollection, GolfCourse golfCourse) {
        
        // Add tree features
        if (gpsVector.getTree() != null) {
            addShapeObjectFeatures(gpsVector.getTree(), "tree", null, golfCourse, featureCollection);
        }
        
        // Add water features
        if (gpsVector.getWater() != null) {
            addShapeObjectFeatures(gpsVector.getWater(), "water", null, golfCourse, featureCollection);
        }
        
        // Add pond features
        if (gpsVector.getPond() != null) {
            addShapeObjectFeatures(gpsVector.getPond(), "pond", null, golfCourse, featureCollection);
        }
        
        // Add lake features
        if (gpsVector.getLake() != null) {
            addShapeObjectFeatures(gpsVector.getLake(), "lake", null, golfCourse, featureCollection);
        }
        
        // Add sand features
        if (gpsVector.getSand() != null) {
            addShapeObjectFeatures(gpsVector.getSand(), "sand", null, golfCourse, featureCollection);
        }
        
        // Add path features
        if (gpsVector.getPath() != null) {
            addShapeObjectFeatures(gpsVector.getPath(), "path", null, golfCourse, featureCollection);
        }
        
        // Add clubhouse features
        if (gpsVector.getClubhouse() != null) {
            addShapeObjectFeatures(gpsVector.getClubhouse(), "clubhouse", null, golfCourse, featureCollection);
        }
        
        // Add vegetation features
        if (gpsVector.getVegetation() != null) {
            addShapeObjectFeatures(gpsVector.getVegetation(), "vegetation", null, golfCourse, featureCollection);
        }
    }

    /**
     * Adds features for a ShapeObject to the feature collection.
     */
    private void addShapeObjectFeatures(ShapeObject shapeObject, String featureType, Integer holeNumber,
                                       GolfCourse golfCourse, FeatureCollection featureCollection) {
        if (shapeObject.getShapes() == null) {
            return;
        }
        
        for (Shape shape : shapeObject.getShapes()) {
            Feature feature = createFeatureFromShape(shape, featureType, holeNumber, golfCourse);
            if (feature != null) {
                featureCollection.getFeatures().add(feature);
            }
        }
    }

    /**
     * Creates a feature from a Shape.
     */
    private Feature createFeatureFromShape(Shape shape, String featureType, Integer holeNumber, GolfCourse golfCourse) {
        if (shape.getWrapper() == null || shape.getWrapper().getPoints() == null || 
            shape.getWrapper().getPoints().isEmpty()) {
            return null;
        }
        
        Feature feature = geojsonFactory.createFeature();
        
        // Create geometry based on number of points
        AbstractGeometry geometry;
        if (shape.getWrapper().getPoints().size() == 1) {
            // Single point
            geometry = createPointGeometry(shape.getWrapper().getPoints().get(0));
        } else {
        	if(isLastLocationEqualToFirst(shape.getWrapper().getPoints())) {
        		 // Multiple points - first location equal to last one - create Polygon
                geometry = createPolygonGeometry(shape.getWrapper().getPoints());
        	} else {
//        		Multiple points - first location NOT equal to last one - create a LineString
        		geometry = createLineStringGeometry(shape.getWrapper().getPoints());
        	}
           
        }
        feature.setGeometry(geometry);
        
        // Set feature type
        feature.setType("Feature");
        
        // Set properties using EMap
        feature.getProperties().put("type", featureType);
        feature.getProperties().put("courseId", golfCourse.getCourseId());
        
        if (holeNumber != null) {
            feature.getProperties().put("holeNumber", String.valueOf(holeNumber));
        }
        
        // Add shape attributes
        if (shape.getAttributes() != null) {
            if (shape.getAttributes().getVegetation() != null) {
                feature.getProperties().put("vegetation", shape.getAttributes().getVegetation().toString());
            }
            if (shape.getAttributes().getSize() != null) {
                feature.getProperties().put("size", shape.getAttributes().getSize().toString());
            }
            if (shape.getAttributes().getDescription() != null) {
                feature.getProperties().put("description", shape.getAttributes().getDescription().toString());
            }
        }
        return feature;
    }

    /**
     * Creates a Point geometry from a Location.
     */
    private Point createPointGeometry(Location location) {
        Point point = geojsonFactory.createPoint();
        // Set coordinates as [longitude, latitude, elevation]
        point.getCoordinates().add(location.getLongitude());
        point.getCoordinates().add(location.getLatitude());
        // Add elevation if available (default 0.0)
        point.getCoordinates().add(0.0);
        return point;
    }

    /**
     * Creates a Point geometry from the first shape in a ShapeObject.
     */
    private AbstractGeometry createPointFromShapeObject(ShapeObject shapeObject) {
        if (shapeObject.getShapes() != null && !shapeObject.getShapes().isEmpty()) {
            Shape firstShape = shapeObject.getShapes().get(0);
            if (firstShape.getWrapper() != null && firstShape.getWrapper().getPoints() != null &&
                !firstShape.getWrapper().getPoints().isEmpty()) {
                return createPointGeometry(firstShape.getWrapper().getPoints().get(0));
            }
        }
        return null;
    }

    /**
     * Creates a Polygon geometry from a list of Locations.
     */
    private Polygon createPolygonGeometry(java.util.List<Location> locations) {
        Polygon polygon = geojsonFactory.createPolygon();
        
        // Create coordinate array for the exterior ring
        Double[][] ring = new Double[locations.size()][];
        for (int i = 0; i < locations.size(); i++) {
            Location location = locations.get(i);
            ring[i] = new Double[]{location.getLongitude(), location.getLatitude()};
        }
//        The exterior ring of a Polygon has to be counter-clock-wise
        if(!isCounterClockwise(ring)) reverseWinding(ring);
        
        polygon.getCoordinates().add(ring);
        
        return polygon;
    }
    
    private LineString createLineStringGeometry(java.util.List<Location> locations) {
    	LineString lineString = geojsonFactory.createLineString();
    	
    	for(Location location : locations) {
    		lineString.getCoordinates().add(new Double[] {location.getLongitude(), location.getLatitude()});
    	}
    	return lineString;
    }
    
    private boolean isLastLocationEqualToFirst(java.util.List<Location> locations) {
//      The first and last coordinates of a polygon must be the same. If it's not, we copy the first coordinate at the end
        Location first = locations.get(0);
        Location last = locations.get(locations.size()-1);
        
        if(first.getLatitude() != last.getLatitude() || first.getLongitude() != last.getLongitude()) {
        	return false;
        }
        return true;
    }
    
    
    /**
     * Reverses the winding order of a closed coordinate ring (double[][]) 
     * in-place to enforce the GeoJSON right-hand rule.
     * * Assumes the array is in the format: [[lon1, lat1], [lon2, lat2], ..., [lonN, latN]],
     * where [lon1, lat1] is identical to [lonN, latN] (a closed ring).
     * * @param coordinates The 2D array of coordinates to reverse.
     */
    private void reverseWinding(Double[][] coordinates) {
        if (coordinates == null || coordinates.length < 4) {
            // A valid linear ring must have at least 4 points (including the repeated end point)
            return;
        }

        int N = coordinates.length;
        // The first and last points MUST remain the same to keep the ring closed.
        // We only reverse the sequence from index 1 up to index N-2.

        for (int i = 1; i <= (N - 2) / 2; i++) {
            // Calculate indices for the swap pair
            int indexA = i;
            int indexB = N - 1 - i;

            // Swap the coordinate arrays (the [lon, lat] pairs)
            Double[] temp = coordinates[indexA];
            coordinates[indexA] = coordinates[indexB];
            coordinates[indexB] = temp;
        }
    }
    
    /**
     * Converts a raw double[][] array (GeoJSON format: [lon, lat]) 
     * into a JTS Coordinate array.
     * @param rawCoords The 2D double array of coordinates.
     * @return A JTS Coordinate array.
     */
    private Coordinate[] toJtsCoordinates(Double[][] rawCoords) {
        Coordinate[] jtsCoords = new Coordinate[rawCoords.length];
        for (int i = 0; i < rawCoords.length; i++) {
            // GeoJSON order is [longitude (x), latitude (y)]
            jtsCoords[i] = new Coordinate(rawCoords[i][0], rawCoords[i][1]);
        }
        return jtsCoords;
    }

    /**
     * Checks if the coordinate ring is Counter-Clockwise (CCW).
     * * @param coordinates The 2D array of coordinates.
     * @return true if the winding is CCW, false if Clockwise (CW) or degenerate.
     */
    private boolean isCounterClockwise(Double[][] coordinates) {
        if (coordinates == null || coordinates.length < 4) {
            // A degenerate ring cannot have a defined winding order
            return false;
        }

        Coordinate[] jtsCoords = toJtsCoordinates(coordinates);

        // Orientation.isCCW() calculates the signed area (or equivalent) 
        // to determine if the order is counter-clockwise.
        return Orientation.isCCW(jtsCoords);
    }
}