/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.golf.features;

import java.io.FileWriter;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;

import geojson.FeatureCollection;

/**
 * Simplified service for loading golf course XMI files and transforming them to GeoJSON.
 * This version saves the local GeoJSON model directly as JSON.
 * 
 * @author Mark Hoffmann
 * @since 14.10.2025
 */
@Component(immediate = true, service = SimpleGolfCourseTransformationService.class)
public class SimpleGolfCourseTransformationService {

    private static final Logger logger = Logger.getLogger(SimpleGolfCourseTransformationService.class.getName());
    
    private final PromiseFactory promiseFactory = new PromiseFactory(Runnable::run);
    
    @Reference
    private GolfCourseFeatureTransformer transformer;
    @Reference(target = "(emf.name=golfcourse)")
    private ResourceSet resourceSet;
    
    /**
     * Loads a golf course from an XMI file and transforms it to GeoJSON.
     * 
     * @param xmiFilePath the path to the XMI file
     * @return Promise resolving to a GeoJSON FeatureCollection
     */
    public Promise<FeatureCollection> loadAndTransform(String xmiFilePath) {
        return promiseFactory.submit(() -> {
            try {
                logger.info("Loading golf course from: " + xmiFilePath);
                
                // Load the golf course XMI file
                URI xmiUri = URI.createFileURI(xmiFilePath);
                Resource resource = resourceSet.getResource(xmiUri, true);
                
                if (resource.getContents().isEmpty()) {
                    throw new IllegalArgumentException("No content found in XMI file: " + xmiFilePath);
                }
                
                // Get the golf course from the resource
                Object rootObject = resource.getContents().get(0);
                if (!(rootObject instanceof GolfCourse)) {
                    throw new IllegalArgumentException("Root object is not a GolfCourse: " + rootObject.getClass().getName());
                }
                
                GolfCourse golfCourse = (GolfCourse) rootObject;
                logger.info("Loaded golf course: " + (golfCourse.getCourseDetails() != null ? 
                    golfCourse.getCourseDetails().getCourseName() : golfCourse.getId()));
                
                // Transform to GeoJSON
                return transformer.transformToGeoJSON(golfCourse).getValue();
                
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Failed to transform golf course from " + xmiFilePath, e);
                throw new RuntimeException("Failed to transform golf course: " + e.getMessage(), e);
            }
        });
    }
    
    /**
     * Saves feature count and basic info to a text file.
     * 
     * @param featureCollection the FeatureCollection
     * @param outputPath the output file path
     * @return Promise resolving when save is complete
     */
    public Promise<Void> saveFeatureInfo(FeatureCollection featureCollection, String outputPath) {
        return promiseFactory.submit(() -> {
            try {
                logger.info("Saving feature info to: " + outputPath);
                
                try (FileWriter writer = new FileWriter(outputPath)) {
                    writer.write("GeoJSON FeatureCollection Summary\n");
                    writer.write("=====================================\n\n");
                    writer.write("Total Features: " + featureCollection.getFeatures().size() + "\n\n");
                    
                    // Count features by type
                    java.util.Map<String, Integer> typeCounts = new java.util.HashMap<>();
                    for (geojson.Feature feature : featureCollection.getFeatures()) {
                        String type = feature.getProperties().get("type");
                        typeCounts.put(type != null ? type : "unknown", 
                            typeCounts.getOrDefault(type != null ? type : "unknown", 0) + 1);
                    }
                    
                    writer.write("Features by Type:\n");
                    for (java.util.Map.Entry<String, Integer> entry : typeCounts.entrySet()) {
                        writer.write("  " + entry.getKey() + ": " + entry.getValue() + "\n");
                    }
                    
                    writer.write("\nFirst 5 Features:\n");
                    int count = 0;
                    for (geojson.Feature feature : featureCollection.getFeatures()) {
                        if (count++ >= 5) break;
                        writer.write("  - Type: " + feature.getProperties().get("type") + 
                                   ", Properties: " + feature.getProperties().size() + 
                                   ", Geometry: " + (feature.getGeometry() != null ? 
                                       feature.getGeometry().getClass().getSimpleName() : "null") + "\n");
                    }
                }
                
                logger.info("Successfully saved feature info with " + featureCollection.getFeatures().size() + " features");
                return null;
                
            } catch (IOException e) {
                logger.log(Level.SEVERE, "Failed to save feature info to " + outputPath, e);
                throw new RuntimeException("Failed to save feature info: " + e.getMessage(), e);
            }
        });
    }
    
    /**
     * Complete workflow: load golf course XMI, transform to GeoJSON, and save info to file.
     * 
     * @param xmiFilePath the input XMI file path
     * @param outputPath the output info file path
     * @return Promise resolving when workflow is complete
     */
    public Promise<FeatureCollection> transformAndSaveInfo(String xmiFilePath, String outputPath) {
        return loadAndTransform(xmiFilePath)
            .then(featureCollection -> {
                saveFeatureInfo(featureCollection.getValue(), outputPath);
                return featureCollection;
            });
    }
}