/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package org.gecko.mac.golf.loader;

import java.io.IOException;

import java.lang.System.Logger;
import java.lang.System.Logger.Level;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.mongo.annotations.RequireMongoEMFRepository;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;

/**
 * 
 * @author grune
 * @since Oct 14, 2025
 */
@Component
@RequireMongoEMFRepository
public class GolfCourseLoader {
	private static final Logger LOGGER = System.getLogger(GolfCourseLoader.class.getName());

	private static final String[] IDS = { 
			// Jena
			"63f5a3a08a278f2a7b063c74", "64050bdfa768b2b3ec9f5ff4", 
			// Dittmannsdorf
			"64d4bc31a7f4c2517b696b13", 
			// Barcelona
			"63fe376bb3844536b105adf4", "63fe3be9b3844536b105ae6e", "63fe3c1db3844536b105ae73",
			"63fe3c59b3844536b105ae79" 
	};

	@Reference
	private ComponentServiceObjects<EMFRepository> repoServiceObjects;
	@Reference
	private ComponentServiceObjects<ResourceSet> resourceServiceObjects;

	@Reference
	private GolfCoursePackage ePackage;

	@Activate
	public void activate() {
		LOGGER.log(Level.INFO, "GolfCourseLoader activated - ready to load golf courses from MongoDB");

		List<GolfCourse> golfCourses = loadGolfCourses(IDS);
		for(GolfCourse golfCourse : golfCourses) {
			saveGolfCourse(golfCourse, System.getProperty("data")+golfCourse.getId()+".xmi");
		}
//		saveGolfCourses(golfCourses);
	}

	private List<GolfCourse> loadGolfCourses(String... ids) {
		List<GolfCourse> courses = new ArrayList<>();
		EMFRepository repo = repoServiceObjects.getService();
		try {
			EClass golfCourseEClass = GolfCoursePackage.eINSTANCE.getGolfCourse();
			for (String id : ids) {

				GolfCourse golfCourse = repo.getEObject(golfCourseEClass, id);
				if (golfCourse == null) {
					LOGGER.log(Level.ERROR, "Golf course with ID {0} not found", id);
					continue;
				}
				courses.add(EcoreUtil.copy(golfCourse));
				String courseName = golfCourse.getCourseDetails().getCourseName();
				LOGGER.log(Level.INFO, "Successfully loaded golf course: {0} (ID: {1})", courseName,
						golfCourse.getId());
			}
		} catch (Exception e) {
			LOGGER.log(Level.ERROR, "Failed to load golf courses", e);
		} finally {
			repoServiceObjects.ungetService(repo);
		}
		return courses;
	}

	private void saveGolfCourses(List<GolfCourse> golfCourses) {
		ResourceSet resourceSet = resourceServiceObjects.getService();
		try {
			Resource resource = resourceSet.createResource(URI.createFileURI("./golf.xmi"));
			EList<EObject> contents = resource.getContents();
			golfCourses.forEach(contents::add);
			resource.save(Collections.emptyMap());
		} catch (IOException e) {
			LOGGER.log(Level.ERROR, "Failed to save golf courses", e);
		} finally {
			resourceServiceObjects.ungetService(resourceSet);
		}
	}
	
	private void saveGolfCourse(GolfCourse golfCourse, String outputFile) {
		ResourceSet resourceSet = resourceServiceObjects.getService();
		try {
			Resource resource = resourceSet.createResource(URI.createFileURI(outputFile));
			EList<EObject> contents = resource.getContents();
			contents.add(golfCourse);
			resource.save(Collections.emptyMap());
		} catch (IOException e) {
			LOGGER.log(Level.ERROR, "Failed to save golf courses", e);
		} finally {
			resourceServiceObjects.ungetService(resourceSet);
		}
	}

}
