/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.governance.rest.tests;

import static java.util.Objects.nonNull;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.emf.rest.annotations.RequireEMFMessageBodyReaderWriter;
import org.gecko.mac.governance.rest.tests.WorkflowResourceTest.MockEObjectWorkflowService;
import org.gecko.mac.mgmt.governanceapi.EObjectWorkflowService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.mgmt.management.ObjectStatus;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.annotations.RequireConfigurationAdmin;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * Integration tests for EPackage REST operations through the workflow API.
 *
 * <p>Tests focus on:</p>
 * <ul>
 * <li>EPackage upload as string content (XMI format)</li>
 * <li>EPackage metadata creation and validation</li>
 * <li>XMI serialization/deserialization issues</li>
 * <li>Content-Type handling for EPackage uploads</li>
 * <li>Error scenarios and debugging</li>
 * </ul>
 *
 * <p>This test class is specifically designed to debug EPackage upload issues
 * and test various REST operation patterns.</p>
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
@RequireEMF
@RequireEMFMessageBodyReaderWriter
@RequireJakartarsWhiteboard
@RequireConfigurationAdmin
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class EPackageRestOperationsTest {

    private static final Logger logger = Logger.getLogger(EPackageRestOperationsTest.class.getName());

    private static final String BASE_URL = "http://localhost:8185/rest/api/workflow";
    private static final String TEST_UPLOAD_USER = "test.developer@company.com";
    private static final String TEST_SOURCE_CHANNEL = "REST_API_TEST";

    @InjectService(filter = "(emf.name=governance)")
    ResourceSet resourceSet;

    @InjectService
    ClientBuilder clientBuilder;

	private Client client;
    private EObjectWorkflowService<EObject> mockWorkflowService;
    @SuppressWarnings("rawtypes")
	private ServiceRegistration<EObjectWorkflowService> mockServiceRegistration;

    @BeforeEach
    void setUp(@InjectBundleContext BundleContext context) throws InterruptedException {
        client = clientBuilder.build();
        logger.info("Set up EPackage REST operations test client");
        // Create and register mock EObjectWorkflowService
        mockWorkflowService = new MockEObjectWorkflowService();
        
        Dictionary<String, Object> serviceProps = new Hashtable<>();
        serviceProps.put("service.ranking", Integer.MAX_VALUE);
        
        mockServiceRegistration = context.registerService(
                EObjectWorkflowService.class,
                mockWorkflowService,
                serviceProps);
        
        // Small delay to allow service registration to propagate
        Thread.sleep(200);
        
        // Wait for the WorkflowResource to be registered in Jakarta REST runtime
        ResourceAware resourceAware = ResourceAware.create(context, "workflow");
        boolean resourceReady = resourceAware.waitForResource(15, TimeUnit.SECONDS);
        
        Thread.sleep(2000l);
        
        assertTrue(resourceReady, 
                "WorkflowResource should be registered within 15 seconds. " +
                "Check that the resource is properly configured and the Jakarta REST runtime is working.");
    }

    @AfterEach
    void tearDown() throws InterruptedException {
    	if (nonNull(mockServiceRegistration)) {
    		mockServiceRegistration.unregister();
    		mockServiceRegistration = null;
    		
    		// Small delay to allow service unregistration to propagate
    		Thread.sleep(200);
    	}
    	
    	if (nonNull(client)) {
    		client.close();
    		client = null;
    	}
        logger.info("Cleaned up EPackage REST operations test client");
    }

    /**
     * Test reproducing the colleague's deserialization error with conference EPackage.
     * This test uses the exact XMI content that's failing in production.
     */
    @Test
    void testColleagueConferenceEPackageUpload() {
        logger.info("Testing colleague's conference EPackage upload");

        // Use the exact XMI content from the colleague
        String conferenceXmiContent = getColleagueConferenceXMI();
        logger.info("Conference XMI content length: " + conferenceXmiContent.length());

        // Test the exact endpoint the colleague is using
        testColleagueEndpoint(conferenceXmiContent);
    }

    /**
     * Test reproducing the colleague's simple test EPackage.
     * @throws InterruptedException 
     */
    @Test
    void testColleagueSimpleTestEPackageUpload() throws InterruptedException {
        logger.info("Testing colleague's simple test EPackage upload");

        // Use the exact simple XMI content from the colleague
        String simpleXmiContent = getColleagueSimpleTestXMI();
        logger.info("Simple test XMI content length: " + simpleXmiContent.length());

        // Test the exact endpoint the colleague is using
        testColleagueEndpoint(simpleXmiContent);
    }

    /**
     * Test uploading a simple EPackage as XMI string content.
     * This test helps debug EPackage upload issues.
     */
    @Test
    void testUploadEPackageAsStringContent() throws IOException {
        logger.info("Testing EPackage upload as string content");

        // Create a simple test EPackage
        EPackage testPackage = createTestEPackage();
        
        // Serialize EPackage to XMI string
        String xmiContent = serializeEPackageToXMI(testPackage);
        logger.info("Generated XMI content length: " + xmiContent.length());
        logger.info("XMI content preview: " + xmiContent.substring(0, Math.min(200, xmiContent.length())) + "...");

        // Create metadata for the upload
        ObjectMetadata metadata = createTestMetadata("test-epackage-string-upload");

        // Serialize metadata to XMI string
        String metadataXmi = serializeMetadataToXMI(metadata);
        logger.info("Generated metadata XMI length: " + metadataXmi.length());

        // Test 1: Upload with application/xmi content type
        testUploadWithContentType(xmiContent, metadataXmi, "application/xmi");

        // Test 2: Upload with text/plain content type
        testUploadWithContentType(xmiContent, metadataXmi, "text/plain");

        // Test 3: Upload with application/xml content type
        testUploadWithContentType(xmiContent, metadataXmi, "application/xml");
    }

    /**
     * Test uploading EPackage with different content types to debug issues.
     */
    private void testUploadWithContentType(String xmiContent, String metadataXmi, String contentType) {
        logger.info("Testing upload with content type: " + contentType);

        try {
            Response response = client.target(BASE_URL)
                .path("upload")
                .queryParam("content", xmiContent)
                .queryParam("metadata", metadataXmi)
                .request()
                .post(Entity.entity("", contentType));

            logger.info("Upload response status: " + response.getStatus());
            logger.info("Upload response headers: " + response.getHeaders());

            if (response.hasEntity()) {
                String responseBody = response.readEntity(String.class);
                logger.info("Upload response body: " + responseBody);
            }

            // Log success or failure details
            if (response.getStatus() >= 200 && response.getStatus() < 300) {
                logger.info("SUCCESS: EPackage upload with " + contentType + " succeeded");
            } else {
                logger.warning("FAILED: EPackage upload with " + contentType + " failed with status " + response.getStatus());
            }

            response.close();

        } catch (Exception e) {
            logger.severe("ERROR during upload with " + contentType + ": " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Test retrieving uploaded EPackage content.
     */
    @Test
    void testRetrieveEPackageContent() {
        logger.info("Testing EPackage content retrieval");

        // First upload an EPackage (simplified version)
        String testObjectId = uploadSimpleEPackage();

        if (testObjectId != null) {
            try {
                // Retrieve the content
                Response response = client.target(BASE_URL)
                    .path("objects")
                    .path(testObjectId)
                    .path("content")
                    .request("application/xmi")
                    .get();

                logger.info("Retrieve response status: " + response.getStatus());

                if (response.hasEntity()) {
                    String content = response.readEntity(String.class);
                    logger.info("Retrieved content length: " + content.length());
                    logger.info("Retrieved content preview: " + content.substring(0, Math.min(200, content.length())) + "...");

                    assertTrue(content.contains("ecore:EPackage"), "Content should contain EPackage");
                }

                response.close();

            } catch (Exception e) {
                logger.severe("ERROR during content retrieval: " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    /**
     * Test listing draft objects to verify EPackage uploads.
     */
    @Test
    void testListDraftObjects() {
        logger.info("Testing draft objects listing");

        try {
            Response response = client.target(BASE_URL)
                .path("drafts")
                .request(MediaType.APPLICATION_XML)
                .get();

            logger.info("List drafts response status: " + response.getStatus());

            if (response.hasEntity()) {
                String content = response.readEntity(String.class);
                logger.info("Drafts list content length: " + content.length());
                logger.info("Drafts list preview: " + content.substring(0, Math.min(500, content.length())) + "...");
            }

            response.close();

        } catch (Exception e) {
            logger.severe("ERROR during drafts listing: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Helper method to create a simple test EPackage.
     */
    private EPackage createTestEPackage() {
        EcoreFactory ecoreFactory = EcoreFactory.eINSTANCE;
        
        EPackage ePackage = ecoreFactory.createEPackage();
        ePackage.setName("TestPackage");
        ePackage.setNsURI("https://test.domain/testpackage/1.0");
        ePackage.setNsPrefix("test");

        // Add a simple EClass
        EClass eClass = ecoreFactory.createEClass();
        eClass.setName("TestClass");
        ePackage.getEClassifiers().add(eClass);

        logger.info("Created test EPackage: " + ePackage.getNsURI());
        return ePackage;
    }

    /**
     * Get the exact conference XMI content that the colleague is trying to upload.
     */
    private String getColleagueConferenceXMI() {
        return """
            <?xml version="1.0" encoding="UTF-8"?>
            <ecore:EPackage xmi:version="2.0" xmlns:xmi="http://www.omg.org/XMI" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                xmlns:ecore="http://www.eclipse.org/emf/2002/Ecore" name="conference" nsURI="conference" nsPrefix="conference">
              <eAnnotations source="http://www.eclipse.org/OCL/Import">
                <details key="ecore" value="http://www.eclipse.org/emf/2002/Ecore"/>
              </eAnnotations>
              <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore">
                <details key="invocationDelegates" value="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot"/>
                <details key="settingDelegates" value="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot"/>
                <details key="validationDelegates" value="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot"/>
              </eAnnotations>
              <eClassifiers xsi:type="ecore:EClass" name="Room" eSuperTypes="#//NamedElement"/>
              <eClassifiers xsi:type="ecore:EClass" name="Person">
                <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore">
                  <details key="constraints" value="noConflict"/>
                </eAnnotations>
                <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot">
                  <details key="noConflict" value="self.attends->forAll(t1:Talk | self.attends->forAll(t2:Talk| (t1.timeBegin &lt; t2.timeBegin and t1.timeEnd &lt;= t2.timeBegin) or (t2.timeBegin &lt; t1.timeBegin and t2.timeEnd &lt;= t1.timeBegin)))"/>
                </eAnnotations>
                <eOperations name="meetsPersonAt" ordered="false" upperBound="-1" eType="#//Talk">
                  <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot">
                    <details key="body" value="Talk.allInstances()->select(t:Talk | (t.speakers->includes(self) or t.attendees->includes(self)) and (t.speakers->includes(other) or t.attendees->includes(other)))"/>
                  </eAnnotations>
                  <eParameters name="other" eType="#//Person"/>
                </eOperations>
                <eStructuralFeatures xsi:type="ecore:EReference" name="worksFor" eType="#//Organization"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="firstName" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="lastName" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="gives" upperBound="-1"
                    eType="#//Talk" eOpposite="#//Talk/speakers"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="attends" upperBound="-1"
                    eType="#//Talk" eOpposite="#//Talk/attendees"/>
              </eClassifiers>
              <eClassifiers xsi:type="ecore:EClass" name="Organization" eSuperTypes="#//NamedElement"/>
              <eClassifiers xsi:type="ecore:EClass" name="Track" eSuperTypes="#//NamedElement">
                <eStructuralFeatures xsi:type="ecore:EReference" name="talks" upperBound="-1"
                    eType="#//Talk" derived="true">
                  <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot">
                    <details key="derivation" value="self.conference.talks->select(t:Talk|t.track = self)"/>
                  </eAnnotations>
                </eStructuralFeatures>
                <eStructuralFeatures xsi:type="ecore:EReference" name="conference" eType="#//Conference"
                    eOpposite="#//Conference/tracks"/>
              </eClassifiers>
              <eClassifiers xsi:type="ecore:EClass" name="Talk">
                <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore">
                  <details key="constraints" value="hasDuration beginBeforeEnd"/>
                </eAnnotations>
                <eAnnotations source="http://www.eclipse.org/emf/2002/Ecore/OCL/Pivot">
                  <details key="hasDuration" value="self.timeBegin &lt;> self.timeEnd"/>
                  <details key="beginBeforeEnd" value="self.timeBegin &lt; self.timeEnd"/>
                </eAnnotations>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="title" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="timeBegin" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EDate"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="timeEnd" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EDate"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="track" eType="#//Track"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="room" eType="#//Room"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="speakers" upperBound="-1"
                    eType="#//Person" eOpposite="#//Person/gives"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="attendees" upperBound="-1"
                    eType="#//Person" eOpposite="#//Person/attends"/>
              </eClassifiers>
              <eClassifiers xsi:type="ecore:EClass" name="Conference" eSuperTypes="#//NamedElement">
                <eStructuralFeatures xsi:type="ecore:EReference" name="venue" eType="#//Venue"
                    containment="true"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="talks" upperBound="-1"
                    eType="#//Talk" containment="true"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="attendees" upperBound="-1"
                    eType="#//Person" containment="true"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="tracks" upperBound="-1"
                    eType="#//Track" containment="true" eOpposite="#//Track/conference"/>
                <eStructuralFeatures xsi:type="ecore:EReference" name="organizations" upperBound="-1"
                    eType="#//Organization" containment="true"/>
              </eClassifiers>
              <eClassifiers xsi:type="ecore:EClass" name="Venue" eSuperTypes="#//NamedElement">
                <eStructuralFeatures xsi:type="ecore:EReference" name="rooms" upperBound="-1"
                    eType="#//Room" containment="true"/>
              </eClassifiers>
              <eClassifiers xsi:type="ecore:EClass" name="NamedElement" abstract="true" interface="true">
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="name" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
              </eClassifiers>
            </ecore:EPackage>
            """;
    }

    /**
     * Get the exact simple test XMI content that the colleague is trying to upload.
     */
    private String getColleagueSimpleTestXMI() {
        return """
            <?xml version="1.0" encoding="UTF-8"?>
            <ecore:EPackage xmi:version="2.0" xmlns:xmi="http://www.omg.org/XMI" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                xmlns:ecore="http://www.eclipse.org/emf/2002/Ecore" name="test" nsURI="http://test/1.0" nsPrefix="test">
              <eClassifiers xsi:type="ecore:EClass" name="Person">
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="firstName" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="lastName" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="age" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EInt"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="email" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EString"/>
                <eStructuralFeatures xsi:type="ecore:EAttribute" name="dateOfBirth" eType="ecore:EDataType http://www.eclipse.org/emf/2002/Ecore#//EDate"/>
              </eClassifiers>
            </ecore:EPackage>
            """;
    }

    /**
     * Test using the exact endpoint and parameters that the colleague is using.
     * This reproduces the "Error de-serializing incoming data" issue.
     * THIS TEST SHOULD FAIL until the deserialization issue is fixed.
     */
    private void testColleagueEndpoint(String xmiContent) {
        logger.info("Testing colleague's exact endpoint with XMI content of length: " + xmiContent.length());
        logger.info("XMI content preview: " + xmiContent.substring(0, Math.min(300, xmiContent.length())) + "...");

        try {
            // The colleague is POSTing XMI content as string with application/xmi content type
            // to: POST /drafts with query parameters
            Response response = client.target(BASE_URL)
                .path("drafts")
                .queryParam("uploadUser", "developer")
                .queryParam("objectType", "EPackage")
                .queryParam("sourceChannel", "PACKAGE") 
                .queryParam("contentHash", "sha256:abc123def456...")
                .queryParam("fileExtension", "ecore")
                .request()
                .post(Entity.entity(xmiContent, "application/xmi"));

            logger.info("=== COLLEAGUE ENDPOINT TEST RESULTS ===");
            logger.info("Response status: " + response.getStatus());
            logger.info("Response headers: " + response.getHeaders());

            String responseBody = null;
            if (response.hasEntity()) {
                responseBody = response.readEntity(String.class);
                logger.info("Response body: " + responseBody);
            }

            response.close();

            // THIS TEST SHOULD FAIL - we expect HTTP 201 success but currently get HTTP 500
            assertEquals(201, response.getStatus(), 
                "EXPECTED FAILURE: EPackage upload should succeed with HTTP 201, but currently fails with: " + 
                response.getStatus() + " - " + responseBody);

            // If we get here, the issue is fixed
            assertNotNull(responseBody, "Response should contain the object ID");
            assertTrue(responseBody.trim().length() > 0, "Response should not be empty");
            logger.info("SUCCESS: Colleague's endpoint now works! Response: " + responseBody);

        } catch (Exception e) {
            logger.severe("ERROR with colleague's endpoint: " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException("EPackage upload failed with exception", e);
        }
    }

    /**
     * Helper method to create test metadata.
     */
    private ObjectMetadata createTestMetadata(String objectId) {
        ManagementFactory factory = ManagementFactory.eINSTANCE;
        ObjectMetadata metadata = factory.createObjectMetadata();
        
        metadata.setObjectId(objectId);
        metadata.setObjectName("Test EPackage");
        metadata.setObjectType("EPackage");
        metadata.setVersion("1.0.0");
        metadata.setStatus(ObjectStatus.DRAFT);
        metadata.setUploadUser(TEST_UPLOAD_USER);
        metadata.setUploadTime(Instant.now());
        metadata.setSourceChannel(TEST_SOURCE_CHANNEL);
        metadata.setLastChangeUser(TEST_UPLOAD_USER);
        metadata.setLastChangeTime(Instant.now());

        logger.info("Created test metadata for object: " + objectId);
        return metadata;
    }

    /**
     * Helper method to serialize EPackage to XMI string.
     */
    private String serializeEPackageToXMI(EPackage ePackage) throws IOException {
        // Create a temporary resource
        Resource resource = resourceSet.createResource(URI.createURI("temp://test.ecore"));
        resource.getContents().add(ePackage);

        // Serialize to XMI
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        resource.save(outputStream, Collections.emptyMap());
        
        String xmiContent = outputStream.toString("UTF-8");
        outputStream.close();
        
        // Clean up
        resource.getContents().clear();
        resourceSet.getResources().remove(resource);
        
        return xmiContent;
    }

    /**
     * Helper method to serialize metadata to XMI string.
     */
    private String serializeMetadataToXMI(ObjectMetadata metadata) throws IOException {
        // Create a temporary resource for metadata
        Resource resource = resourceSet.createResource(URI.createURI("temp://metadata.xmi"));
        resource.getContents().add(metadata);

        // Serialize to XMI
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        resource.save(outputStream, Collections.emptyMap());
        
        String xmiContent = outputStream.toString("UTF-8");
        outputStream.close();
        
        // Clean up
        resource.getContents().clear();
        resourceSet.getResources().remove(resource);
        
        return xmiContent;
    }

    /**
     * Helper method to upload a simple EPackage and return the object ID.
     */
    private String uploadSimpleEPackage() {
        try {
            EPackage testPackage = createTestEPackage();
            String xmiContent = serializeEPackageToXMI(testPackage);
            ObjectMetadata metadata = createTestMetadata("simple-epackage-test");
            String metadataXmi = serializeMetadataToXMI(metadata);

            Response response = client.target(BASE_URL)
                .path("upload")
                .queryParam("content", xmiContent)
                .queryParam("metadata", metadataXmi)
                .request()
                .post(Entity.entity("", MediaType.APPLICATION_XML));

            if (response.getStatus() >= 200 && response.getStatus() < 300) {
                String objectId = response.readEntity(String.class);
                response.close();
                logger.info("Successfully uploaded simple EPackage with ID: " + objectId);
                return objectId.trim();
            } else {
                logger.warning("Failed to upload simple EPackage, status: " + response.getStatus());
                response.close();
                return null;
            }

        } catch (Exception e) {
            logger.severe("ERROR uploading simple EPackage: " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Test parsing colleague's conference XMI to identify deserialization issues.
     */
    @Test
    void testColleagueConferenceXMIParsing() {
        logger.info("Testing colleague's conference XMI parsing");

        try {
            String conferenceXmi = getColleagueConferenceXMI();
            logger.info("Conference XMI length: " + conferenceXmi.length());

            // Try to parse the colleague's XMI
            EPackage parsedPackage = parseEPackageFromXMI(conferenceXmi);
            
            // Verify parsing worked
            assertNotNull(parsedPackage, "Parsed conference EPackage should not be null");
            assertEquals("conference", parsedPackage.getName(), "Package name should be 'conference'");
            assertEquals("conference", parsedPackage.getNsURI(), "Package nsURI should be 'conference'");
            assertEquals("conference", parsedPackage.getNsPrefix(), "Package prefix should be 'conference'");

            logger.info("Conference XMI parsing test PASSED");
            logger.info("Parsed package: " + parsedPackage.getName() + " (" + parsedPackage.getNsURI() + ")");

        } catch (Exception e) {
            logger.severe("Conference XMI parsing test FAILED: " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException("Conference XMI parsing failed", e);
        }
    }

    /**
     * Test parsing colleague's simple test XMI to identify deserialization issues.
     */
    @Test
    void testColleagueSimpleTestXMIParsing() {
        logger.info("Testing colleague's simple test XMI parsing");

        try {
            String simpleXmi = getColleagueSimpleTestXMI();
            logger.info("Simple test XMI length: " + simpleXmi.length());

            // Try to parse the colleague's XMI
            EPackage parsedPackage = parseEPackageFromXMI(simpleXmi);
            
            // Verify parsing worked
            assertNotNull(parsedPackage, "Parsed simple test EPackage should not be null");
            assertEquals("test", parsedPackage.getName(), "Package name should be 'test'");
            assertEquals("http://test/1.0", parsedPackage.getNsURI(), "Package nsURI should be 'http://test/1.0'");
            assertEquals("test", parsedPackage.getNsPrefix(), "Package prefix should be 'test'");

            logger.info("Simple test XMI parsing test PASSED");
            logger.info("Parsed package: " + parsedPackage.getName() + " (" + parsedPackage.getNsURI() + ")");

        } catch (Exception e) {
            logger.severe("Simple test XMI parsing test FAILED: " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException("Simple test XMI parsing failed", e);
        }
    }

    /**
     * Test parsing XMI content back to EPackage to verify round-trip.
     */
    @Test
    void testXMIRoundTrip() {
        logger.info("Testing XMI round-trip for EPackage");

        try {
            // Create original EPackage
            EPackage originalPackage = createTestEPackage();
            
            // Serialize to XMI
            String xmiContent = serializeEPackageToXMI(originalPackage);
            logger.info("Original XMI: " + xmiContent);

            // Parse back from XMI
            EPackage parsedPackage = parseEPackageFromXMI(xmiContent);
            
            // Verify parsing worked
            assertNotNull(parsedPackage, "Parsed EPackage should not be null");
            assertEquals(originalPackage.getName(), parsedPackage.getName(), "Package names should match");
            assertEquals(originalPackage.getNsURI(), parsedPackage.getNsURI(), "Package nsURIs should match");
            assertEquals(originalPackage.getNsPrefix(), parsedPackage.getNsPrefix(), "Package prefixes should match");

            logger.info("XMI round-trip test PASSED");

        } catch (Exception e) {
            logger.severe("XMI round-trip test FAILED: " + e.getMessage());
            e.printStackTrace();
            throw new RuntimeException("XMI round-trip failed", e);
        }
    }

    /**
     * Test direct EMF loading of colleague's XMI with full diagnostic output.
     * This determines if the XMI content itself is valid or malformed.
     */
    @Test
    void testColleagueXMIDirectLoad() {
        logger.info("Testing direct EMF loading of colleague's XMI content with diagnostics");

        // Test both XMI samples with detailed diagnostics
        testXMILoadWithDiagnostics("Conference", getColleagueConferenceXMI());
        testXMILoadWithDiagnostics("Simple Test", getColleagueSimpleTestXMI());
    }

    /**
     * Load XMI content using injected ResourceSet and print detailed diagnostics on errors.
     * This test verifies that the colleague's XMI content is valid for direct EMF loading.
     */
    private void testXMILoadWithDiagnostics(String testName, String xmiContent) {
        logger.info("=== Testing EMF load of " + testName + " XMI ===");
        logger.info("XMI content length: " + xmiContent.length());

        Resource resource = null;
        EPackage ePackage = null;
        
        try {
            // Use the injected resourceSet directly
            resource = resourceSet.createResource(URI.createURI("temp://" + testName.toLowerCase() + ".ecore"));
            
            // Load from XMI string
            ByteArrayInputStream inputStream = new ByteArrayInputStream(xmiContent.getBytes("UTF-8"));
            resource.load(inputStream, Collections.emptyMap());
            inputStream.close();

            // Check for loading errors/warnings and fail test if any
            if (!resource.getErrors().isEmpty()) {
                logger.severe("=== EMF LOADING ERRORS for " + testName + " ===");
                StringBuilder errorMessage = new StringBuilder();
                for (var error : resource.getErrors()) {
                    logger.severe("ERROR: " + error.toString());
                    logger.severe("  Location: line " + error.getLine() + ", column " + error.getColumn());
                    logger.severe("  Message: " + error.getMessage());
                    errorMessage.append(error.getMessage()).append("; ");
                }
                throw new AssertionError("EMF loading failed with errors: " + errorMessage.toString());
            }

            if (!resource.getWarnings().isEmpty()) {
                logger.warning("=== EMF LOADING WARNINGS for " + testName + " ===");
                for (var warning : resource.getWarnings()) {
                    logger.warning("WARNING: " + warning.toString());
                    logger.warning("  Location: line " + warning.getLine() + ", column " + warning.getColumn());
                    logger.warning("  Message: " + warning.getMessage());
                }
            }

            // Assert content was loaded properly
            assertNotNull(resource.getContents(), "Resource contents should not be null");
            assertTrue(!resource.getContents().isEmpty(), "Resource should contain at least one object for " + testName);

            EObject rootObject = resource.getContents().get(0);
            assertNotNull(rootObject, "Root object should not be null for " + testName);
            assertTrue(rootObject instanceof EPackage, 
                "Root object should be an EPackage for " + testName + " but got: " + rootObject.getClass().getSimpleName());

            ePackage = (EPackage) rootObject;
            assertNotNull(ePackage.getName(), "EPackage name should not be null for " + testName);
            assertNotNull(ePackage.getNsURI(), "EPackage nsURI should not be null for " + testName);
            
            logger.info("SUCCESS: " + testName + " XMI loaded successfully");
            logger.info("Package name: " + ePackage.getName());
            logger.info("Package nsURI: " + ePackage.getNsURI());
            logger.info("Package nsPrefix: " + ePackage.getNsPrefix());
            logger.info("Number of classifiers: " + ePackage.getEClassifiers().size());

        } catch (Exception e) {
            logger.severe("EXCEPTION during " + testName + " XMI loading: " + e.getMessage());
            logger.severe("Exception type: " + e.getClass().getSimpleName());
            
            if (e.getCause() != null) {
                logger.severe("Root cause: " + e.getCause().getMessage());
                logger.severe("Root cause type: " + e.getCause().getClass().getSimpleName());
            }
            
            // Print detailed stack trace
            e.printStackTrace();
            
            // Also check resource diagnostics even on exception
            if (resource != null && !resource.getErrors().isEmpty()) {
                logger.severe("=== RESOURCE ERRORS (from exception) ===");
                StringBuilder errorDetails = new StringBuilder();
                for (var error : resource.getErrors()) {
                    logger.severe("ERROR: " + error.toString());
                    errorDetails.append(error.getMessage()).append("; ");
                }
            }
            
            // Re-throw to fail the test properly
            throw new RuntimeException("Failed to load " + testName + " XMI content", e);
            
        } finally {
            // Clean up resource
            if (resource != null) {
                resource.getContents().clear();
                resourceSet.getResources().remove(resource);
            }
        }
    }

    /**
     * Helper method to parse EPackage from XMI string.
     */
    private EPackage parseEPackageFromXMI(String xmiContent) throws IOException {
        // Create a temporary resource
        Resource resource = resourceSet.createResource(URI.createURI("temp://parse.ecore"));
        
        // Load from XMI string
        ByteArrayInputStream inputStream = new ByteArrayInputStream(xmiContent.getBytes("UTF-8"));
        resource.load(inputStream, Collections.emptyMap());
        inputStream.close();

        // Get the EPackage
        EPackage ePackage = (EPackage) resource.getContents().get(0);
        
        // Clean up
        resource.getContents().clear();
        resourceSet.getResources().remove(resource);
        
        return ePackage;
    }
}