/**
 * Copyright (c) 2012 - 2025 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *      Mark Hoffmann - initial API and implementation
 */
package org.gecko.mac.governance.rest.tests;

import static java.util.Objects.nonNull;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.Instant;
import java.util.Date;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.osgi.annotation.require.RequireEMF;
import org.gecko.emf.rest.annotations.RequireEMFMessageBodyReaderWriter;
import org.gecko.mac.governance.ApprovalStatus;
import org.gecko.mac.governance.ComplianceStatus;
import org.gecko.mac.governance.GovernanceDocumentation;
import org.gecko.mac.governance.GovernanceFactory;
import org.gecko.mac.mgmt.governanceapi.GovernanceDocumentationService;
import org.gecko.mac.mgmt.management.ManagementFactory;
import org.gecko.mac.mgmt.management.ObjectMetadata;
import org.gecko.mac.policy.gdpr.GDPRFactory;
import org.gecko.mac.policy.gdpr.GDPRPolicyCheck;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.annotations.RequireConfigurationAdmin;
import org.osgi.service.jakartars.whiteboard.annotations.RequireJakartarsWhiteboard;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.Promises;

import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

/**
 * Integration tests for GovernanceDocumentationResource REST endpoints.
 *
 * <p>Tests cover:</p>
 * <ul>
 * <li>Documentation storage and retrieval operations</li>
 * <li>Latest documentation access and history tracking</li>
 * <li>Documentation existence checks and statistics</li>
 * <li>Error handling and validation scenarios</li>
 * <li>XML serialization of EMF objects via REST</li>
 * </ul>
 *
 * <p>Uses OSGi Test framework with Jakarta RS client to test actual HTTP endpoints.</p>
 * 
 * @author Mark Hoffmann
 * @since 1.0.0
 */
@RequireEMF
@RequireEMFMessageBodyReaderWriter
@RequireJakartarsWhiteboard
@RequireConfigurationAdmin
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class GovernanceDocumentationResourceTest {

    private static final String BASE_URL = "http://localhost:8185/rest/api/governance/documentation";
    private static final String TEST_OBJECT_ID = "test-epackage-model-v1.0";
    private static final String TEST_REVIEW_USER = "test.steward@company.com";

    @InjectService(filter = "(emf.name=governance)")
    ResourceSet resourceSet;

    @InjectService
    ClientBuilder clientBuilder;

    private Client restClient;
    private GovernanceDocumentationService mockDocumentationService;
    private ServiceRegistration<GovernanceDocumentationService> mockServiceRegistration;

    @BeforeEach
    public void setup(@InjectBundleContext BundleContext context) throws Exception {
        // Setup REST client with EMF message body readers/writers
        restClient = clientBuilder
                .build();

        // Create and register mock GovernanceDocumentationService
        mockDocumentationService = new MockGovernanceDocumentationService();
        
        Dictionary<String, Object> serviceProps = new Hashtable<>();
        serviceProps.put("service.ranking", Integer.MAX_VALUE);
        
        mockServiceRegistration = context.registerService(
                GovernanceDocumentationService.class,
                mockDocumentationService,
                serviceProps);
        
        // Wait for the GovernanceDocumentationResource to be registered in Jakarta REST runtime
        // This ensures the resource is fully available before any test methods execute
        ResourceAware resourceAware = ResourceAware.create(context, "governance-documentation");
        boolean resourceReady = resourceAware.waitForResource(10, TimeUnit.SECONDS);
        
        assertTrue(resourceReady, 
                "GovernanceDocumentationResource should be registered within 10 seconds. " +
                "Check that the resource is properly configured and the Jakarta REST runtime is working.");
    }

    @AfterEach
    public void teardown(@InjectBundleContext BundleContext context) throws Exception {
    	if (nonNull(mockServiceRegistration)) {
    		mockServiceRegistration.unregister();
    		mockServiceRegistration = null;
    		
    		// Small delay to allow service unregistration to propagate
    		// The GovernanceDocumentationResource stays registered (it's an OSGi component)
    		// but now uses a different service implementation  
    		Thread.sleep(200);
    	}
    	
    	if (nonNull(restClient)) {
    		restClient.close();
    		restClient = null;
    	}
    }
    // ========== Documentation Storage Tests ==========

    @Test
    public void testStoreDocumentation_Success() throws Exception {
        GovernanceDocumentation documentation = createTestDocumentation();
        String xmiContent = serializeToXMI(documentation);

        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .queryParam("reviewUser", TEST_REVIEW_USER)
                .request("application/xmi")
                .post(Entity.entity(xmiContent, "application/xmi"));

        assertEquals(201, response.getStatus(), "Should return HTTP 201 Created");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String responseMessage = response.readEntity(String.class);
        assertTrue(responseMessage.contains("Documentation stored successfully"), "Response should contain success message");
        assertTrue(responseMessage.contains(TEST_OBJECT_ID), "Response should contain object ID");
        assertTrue(responseMessage.contains(TEST_REVIEW_USER), "Response should contain review user");
    }

    @Test
    public void testStoreDocumentation_WithReason() throws Exception {
        GovernanceDocumentation documentation = createTestDocumentation();
        String xmiContent = serializeToXMI(documentation);
        String testReason = "Initial compliance documentation for testing";

        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .queryParam("reviewUser", TEST_REVIEW_USER)
                .queryParam("reason", testReason)
                .request("application/xmi")
                .post(Entity.entity(xmiContent, "application/xmi"));

        assertEquals(201, response.getStatus(), "Should return HTTP 201 Created");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String responseMessage = response.readEntity(String.class);
        assertTrue(responseMessage.contains("Documentation stored successfully"), "Response should contain success message");
        assertTrue(responseMessage.contains(TEST_OBJECT_ID), "Response should contain object ID");
        assertTrue(responseMessage.contains(TEST_REVIEW_USER), "Response should contain review user");
    }

    @Test
    public void testStoreDocumentation_MissingReviewUser() throws Exception {
        GovernanceDocumentation documentation = createTestDocumentation();
        String xmiContent = serializeToXMI(documentation);

        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .request("application/xmi")
                .post(Entity.entity(xmiContent, "application/xmi"));

        assertEquals(400, response.getStatus(), "Should return HTTP 400 Bad Request");

        // Error responses are returned as plain text
        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String errorMessage = response.readEntity(String.class);
        assertTrue(errorMessage.contains("reviewUser"), "Error should mention missing reviewUser");
    }

    @Test
    public void testStoreDocumentation_MissingDocumentationBody() {
        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .queryParam("reviewUser", TEST_REVIEW_USER)
                .request("application/xmi")
                .post(Entity.entity("", "application/xmi"));

        // When sending empty XMI content, the EMF MessageBodyReader may fail during deserialization
        // This could result in either 400 (Bad Request) or 500 (Internal Server Error) depending on framework behavior
        assertTrue(response.getStatus() == 400 || response.getStatus() == 500, 
                "Should return HTTP 400 or 500 for malformed XMI content, got: " + response.getStatus());
    }

    // ========== Documentation Retrieval Tests ==========

    @Test
    public void testGetLatestDocumentation_Success() {
        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .path("latest")
                .request("application/xmi")
                .get();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        // Use manual XMI deserialization instead of direct object reading
        String xmiContent = response.readEntity(String.class);
        assertNotNull(xmiContent, "Should return XMI content");
        assertTrue(xmiContent.contains("GovernanceDocumentation"), "Should contain GovernanceDocumentation element");
        assertTrue(xmiContent.contains("status=\"APPROVED\"") || xmiContent.contains("APPROVED"), 
                "Should contain approved status");
    }

    @Test
    public void testGetLatestDocumentation_NotFound() {
        String nonExistentObjectId = "non-existent-object";

        Response response = restClient
                .target(BASE_URL)
                .path(nonExistentObjectId)
                .path("latest")
                .request("application/xmi")
                .get();

        assertEquals(404, response.getStatus(), "Should return HTTP 404 Not Found");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String errorMessage = response.readEntity(String.class);
        assertTrue(errorMessage.contains(nonExistentObjectId), 
                "Error should mention the object ID");
    }

    @Test
    public void testGetDocumentationById_Success() {
        String documentationId = "doc-test-epackage-model-v1.0-20250115103000";

        Response response = restClient
                .target(BASE_URL)
                .path("by-id")
                .path(documentationId)
                .request("application/xmi")
                .get();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        // Use manual XMI deserialization instead of direct object reading
        String xmiContent = response.readEntity(String.class);
        assertNotNull(xmiContent, "Should return XMI content");
        assertTrue(xmiContent.contains("GovernanceDocumentation"), "Should contain GovernanceDocumentation element");
        assertTrue(xmiContent.contains("Test Model"), "Should contain test model data");
    }

    @Test
    public void testGetDocumentationHistory_Success() {
        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .path("history")
                .request("application/xmi")
                .get();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        // Now we expect XMI content with the container object
        String xmiContent = response.readEntity(String.class);
        assertNotNull(xmiContent, "Should return XMI content");
        assertTrue(xmiContent.contains("ObjectMetadataContainer"), "Should contain container element");
        
        // The mock service returns one metadata entry, so the container should contain it
        assertTrue(xmiContent.contains("ObjectMetadata") || xmiContent.contains("doc-" + TEST_OBJECT_ID), 
                "Should contain metadata for the test object");
    }

    @Test
    public void testHasDocumentation_Success() {
        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .path("exists")
                .request(MediaType.TEXT_PLAIN)
                .get();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String existsMessage = response.readEntity(String.class);
        assertTrue(existsMessage.contains(TEST_OBJECT_ID), "Response should contain object ID");
        assertTrue(existsMessage.contains("Exists: true"), "Documentation should exist");
    }

    // ========== Documentation Deletion Tests ==========

    @Test
    public void testDeleteAllDocumentation_Success() {
        Response response = restClient
                .target(BASE_URL)
                .path(TEST_OBJECT_ID)
                .request(MediaType.TEXT_PLAIN)
                .delete();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String deleteMessage = response.readEntity(String.class);
        assertTrue(deleteMessage.contains(TEST_OBJECT_ID), "Response should contain object ID");
        assertTrue(deleteMessage.contains("deleted"), "Documentation should be deleted");
        assertTrue(deleteMessage.contains("audit trail"), "Response should contain warning about audit trail");
    }

    @Test
    public void testDeleteAllDocumentation_NotFound() {
        String nonExistentObjectId = "non-existent-object";

        Response response = restClient
                .target(BASE_URL)
                .path(nonExistentObjectId)
                .request(MediaType.TEXT_PLAIN)
                .delete();

        assertEquals(404, response.getStatus(), "Should return HTTP 404 Not Found");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String deleteMessage = response.readEntity(String.class);
        assertTrue(deleteMessage.contains(nonExistentObjectId), "Response should contain object ID");
        assertTrue(deleteMessage.contains("No documentation found"), "Documentation should not be deleted");
    }

    // ========== Statistics Tests ==========

    @Test
    public void testGetDocumentationStatistics_Success() {
        Response response = restClient
                .target(BASE_URL)
                .path("statistics")
                .request(MediaType.TEXT_PLAIN)
                .get();

        assertEquals(200, response.getStatus(), "Should return HTTP 200 OK");

        assertEquals(MediaType.TEXT_PLAIN, response.getMediaType().toString());
        String statsMessage = response.readEntity(String.class);
        assertTrue(statsMessage.contains("Documentation statistics retrieved successfully"), "Response should contain success message");
        assertTrue(statsMessage.contains("Statistics:"), "Response should contain statistics");
        assertTrue(statsMessage.contains("totalDocumentationCount"), "Statistics should contain total count");
    }

    // ========== Error Handling Tests ==========

    @Test
    public void testInvalidPath_MethodNotAllowed() {
        Response response = restClient
                .target(BASE_URL)
                .path("") // Empty object ID
                .path("latest")
                .request("application/xmi")
                .get();

        // The path /governance/documentation/latest doesn't match any defined routes
        // since our resource expects /governance/documentation/{objectId}/latest
        // Jakarta REST returns 405 Method Not Allowed for unmatched path patterns
        assertEquals(405, response.getStatus(), 
                "Should return HTTP 405 Method Not Allowed for path that doesn't match any resource routes");
    }

    // ========== Helper Methods ==========

    /**
     * Serialize an EMF EObject to XMI string using the injected ResourceSet.
     */
    private String serializeToXMI(GovernanceDocumentation documentation) throws IOException {
        Resource resource = resourceSet.createResource(URI.createURI("temp.xmi"));
        resource.getContents().add(documentation);
        
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        resource.save(outputStream, null);
        
        return outputStream.toString("UTF-8");
    }

    /**
     * Create a test GovernanceDocumentation object for testing.
     */
    private GovernanceDocumentation createTestDocumentation() {
        GovernanceDocumentation documentation = GovernanceFactory.eINSTANCE.createGovernanceDocumentation();
        documentation.setModelName("Test EPackage Model");
        documentation.setVersion("1.0");
        documentation.setStatus(ApprovalStatus.APPROVED);
        documentation.setGenerationTimestamp(new Date());
        documentation.setGeneratedBy("test.generator@company.com");
        documentation.setApprovedBy(TEST_REVIEW_USER);
        documentation.setDescription("Test governance documentation");

        // Add a compliance check result using the specific GDPR factory
        GDPRPolicyCheck gdprCheck = GDPRFactory.eINSTANCE.createGDPRPolicyCheck();
        gdprCheck.setStatus(ComplianceStatus.PASSED);
        gdprCheck.setCheckTimestamp(new Date());
        gdprCheck.setSummary("No personal data detected in model");
        documentation.getComplianceChecks().add(gdprCheck);

        return documentation;
    }

    // ========== Mock Service Implementation ==========

    /**
     * Mock implementation of GovernanceDocumentationService for testing.
     */
    private static class MockGovernanceDocumentationService implements GovernanceDocumentationService {

        @Override
        public Promise<String> storeDocumentation(String objectId, GovernanceDocumentation documentation, String reviewUser, String reason) {
            // Return a mock documentation ID
            String documentationId = "doc-" + objectId + "-" + System.currentTimeMillis();
            return Promises.resolved(documentationId);
        }

        @Override
        public Optional<GovernanceDocumentation> getLatestDocumentation(String objectId) {
            if ("non-existent-object".equals(objectId)) {
                return Optional.empty();
            }
            
            GovernanceDocumentation documentation = GovernanceFactory.eINSTANCE.createGovernanceDocumentation();
            documentation.setModelName("Test Model");
            documentation.setVersion("1.0");
            documentation.setStatus(ApprovalStatus.APPROVED);
            documentation.setGenerationTimestamp(new Date());
            documentation.setApprovedBy(TEST_REVIEW_USER);
            
            return Optional.of(documentation);
        }

        @Override
        public Optional<GovernanceDocumentation> getDocumentation(String documentationId) {
            if (documentationId.startsWith("doc-")) {
                GovernanceDocumentation documentation = GovernanceFactory.eINSTANCE.createGovernanceDocumentation();
                documentation.setModelName("Test Model");
                documentation.setVersion("1.0");
                documentation.setStatus(ApprovalStatus.APPROVED);
                documentation.setGenerationTimestamp(new Date());
                return Optional.of(documentation);
            }
            return Optional.empty();
        }

        @Override
        public List<ObjectMetadata> getDocumentationHistory(String objectId) {
            ObjectMetadata metadata = ManagementFactory.eINSTANCE.createObjectMetadata();
            metadata.setObjectId("doc-" + objectId + "-v1");
            metadata.setUploadUser("test.user");
            metadata.setUploadTime(Instant.now());
            
            return List.of(metadata);
        }

        @Override
        public Boolean hasDocumentation(String objectId) {
            return !objectId.equals("non-existent-object");
        }

        @Override
        public Promise<Boolean> deleteAllDocumentation(String objectId) {
            boolean deleted = !objectId.equals("non-existent-object");
            return Promises.resolved(deleted);
        }

        @Override
        public Map<String, Object> getDocumentationStatistics() {
            return Map.of(
                "totalDocumentationCount", 150,
                "activeObjectCount", 75,
                "complianceCheckCount", 200,
                "averageComplianceScore", 0.92
            );
        }
    }
}